#include "vins_request.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("vins_request");

using namespace quasar;
using namespace NAlice;
using namespace YandexIO;

VinsRequest::VinsRequest(Json::Value event, EventSource eventSource, std::optional<std::string> requestId)
    : id_(requestId.has_value() ? std::move(requestId.value()) : makeUUID())
    , event_(std::move(event))
    , eventSource_(std::move(eventSource))
{
    switch (eventSource_.type()) {
        case TSpeechKitRequestProto_TEventSource_EType_Voice:
            event_["type"] = "voice_input";
            break;
        case TSpeechKitRequestProto_TEventSource_EType_Music:
            event_["type"] = "music_input";
            break;
        default:
            break; // do nothing
    }
}

const std::string& VinsRequest::getId() const {
    return id_;
}

bool VinsRequest::isServerAction() const {
    return tryGetString(event_, "type") == "server_action";
}

bool VinsRequest::isUserInitiatedAction() const {
    switch (eventSource_.event()) {
        case TSpeechKitRequestProto_TEventSource_EEvent_Spotter:
        case TSpeechKitRequestProto_TEventSource_EEvent_Click:
        case TSpeechKitRequestProto_TEventSource_EEvent_DoubleClick:
            return true;
        default:
            return false;
    }
}

bool VinsRequest::isVoiceInput() const {
    switch (getEventSource().type()) {
        case TSpeechKitRequestProto_TEventSource_EType_Voice:
        case TSpeechKitRequestProto_TEventSource_EType_Music: {
            return true;
        }
        default: {
            return false;
        }
    }
}

bool VinsRequest::isVoiceprintMatch() const {
    return getEventSource().type() == TSpeechKitRequestProto_TEventSource_EType_VoiceprintMatch;
}

void VinsRequest::setIgnoreAnswer(bool value)
{
    ignoreAnswer_ = value;
}

bool VinsRequest::getIgnoreAnswer() const {
    return ignoreAnswer_;
}

void VinsRequest::setIsParallel(bool value) {
    isParallel_ = value;
}

bool VinsRequest::getIsParallel() const {
    return isParallel_;
}

const Json::Value& VinsRequest::getEvent() const {
    return event_;
}

const VinsRequest::EventSource& VinsRequest::getEventSource() const {
    return eventSource_;
}

std::string VinsRequest::getActivationType() const {
    switch (eventSource_.event()) {
        case TSpeechKitRequestProto_TEventSource_EEvent_Spotter:
            return "spotter";
        case TSpeechKitRequestProto_TEventSource_EEvent_Directive:
            return "directive";
        case TSpeechKitRequestProto_TEventSource_EEvent_AutoListen:
            return "auto_listening";
        case TSpeechKitRequestProto_TEventSource_EEvent_Click:
        case TSpeechKitRequestProto_TEventSource_EEvent_DoubleClick:
            return "okniks";
        case TSpeechKitRequestProto_TEventSource_EEvent_AutoDirective:
        case TSpeechKitRequestProto_TEventSource_EEvent_Scheduler:
        case TSpeechKitRequestProto_TEventSource_EEvent_Push:
            return "auto_activation";
        default:
            return "directive";
    }
}

void VinsRequest::setIsReminder(bool value) {
    isReminder_ = value;
}

bool VinsRequest::getIsReminder() const {
    return isReminder_;
}

void VinsRequest::setAsrText(const std::string& value) {
    asrText_ = value;
}

const std::string& VinsRequest::getAsrText() const {
    return asrText_;
}

Json::Value VinsRequest::buildNewDialogSessionEvent(const std::string& dialogId) {
    Json::Value payload;
    payload["request"] = "";
    payload["dialog_id"] = dialogId;

    Json::Value event;
    event["payload"] = payload;
    event["ignore_answer"] = false;
    event["type"] = "server_action";
    event["name"] = "new_dialog_session";

    return event;
}

std::shared_ptr<VinsRequest> VinsRequest::createListenRequest(
    const Directive::Data& data, const EventSource& eventSource, std::optional<std::string> requestId) {
    auto request = std::make_shared<VinsRequest>(Json::Value(), eventSource, std::move(requestId));

    request->setIsSilent(false);

    const int startingSilenceTimeoutMs = tryGetInt(data.payload, "starting_silence_timeout_ms", -1);
    if (startingSilenceTimeoutMs > 0) {
        request->setStartingSilenceTimeout(std::chrono::milliseconds(startingSilenceTimeoutMs));
    }

    return request;
}

Json::Value VinsRequest::buildReminderEvent(const std::string& reminder) {
    const Json::Value payload = parseJson(reminder);
    const std::string uri = getString(getJson(payload, "reminder"), "uri");

    auto urlParams = getUrlParams(uri);
    const Json::Value array = parseJson(urlParams["directives"]);
    if (array.empty()) {
        throw std::invalid_argument{"Received empty array of reminders"};
    }

    return array[0];
}

Json::Value VinsRequest::buildTextActionEvent(const std::string& textAction) {
    Json::Value action = parseJson(textAction);

    if (action["to_alice"].isNull() || !action["to_alice"].isArray()) {
        throw std::invalid_argument{"action[\"to_alice\"] should be an array"};
    }

    if (action["to_alice"].empty() || action["to_alice"][0].empty()) {
        throw std::invalid_argument{"Received empty array of text actions"};
    }

    if (action["to_alice"].size() > 1) {
        YIO_LOG_ERROR_EVENT("VinsRequest.IgnoreExtraTextActions", "Got more than one text action. Ignore all but first. Incoming message:" + textAction);
    }

    const auto text = action["to_alice"][0].asString();

    return buildTextInputEvent(text);
}

std::shared_ptr<VinsRequest> VinsRequest::createEventRequest(const Json::Value& event, EventSource eventSource, std::optional<std::string> requestId) {
    auto request = std::make_shared<VinsRequest>(event, eventSource, std::move(requestId));

    if (eventSource.event() == TSpeechKitRequestProto_TEventSource_EEvent_Click || eventSource.event() == TSpeechKitRequestProto_TEventSource_EEvent_Spotter) {
        request->setIsSilent(false);
    }

    return request;
}

Json::Value VinsRequest::buildTextInputEvent(const std::string& text) {
    Json::Value event;
    event["text"] = text;
    event["type"] = "text_input";

    return event;
}

std::shared_ptr<VinsRequest> VinsRequest::createServerActionRequest(const Directive::Data& data) {
    Json::Value event;
    event["name"] = data.name;
    event["payload"] = data.payload;

    if (!data.type.empty()) {
        event["type"] = data.type;
    } else {
        event["type"] = "server_action";
    }

    auto request = createEventRequest(event, createSoftwareDirectiveEventSource());

    request->setIsParallel(data.isParallel);
    request->setIgnoreAnswer(data.ignoreAnswer);
    request->setIsSilent(data.isLedSilent);

    return request;
}

VinsRequest::EventSource VinsRequest::createHardwareButtonClickEventSource(const std::string& id) {
    EventSource protobuf;
    protobuf.set_event(TSpeechKitRequestProto_TEventSource_EEvent_Click);
    protobuf.set_source(TSpeechKitRequestProto_TEventSource_ESource_Hardware);
    protobuf.set_type(TSpeechKitRequestProto_TEventSource_EType_Voice);
    protobuf.set_id(TString(id));
    return protobuf;
}

VinsRequest::EventSource VinsRequest::createSoftwareAutoDirectiveEventSource(const std::string& id) {
    EventSource protobuf;
    protobuf.set_event(TSpeechKitRequestProto_TEventSource_EEvent_AutoDirective);
    protobuf.set_source(TSpeechKitRequestProto_TEventSource_ESource_Software);
    protobuf.set_type(TSpeechKitRequestProto_TEventSource_EType_Text);
    protobuf.set_id(TString(id));
    return protobuf;
}

VinsRequest::EventSource VinsRequest::createSoftwareDirectiveEventSource(const std::string& id) {
    EventSource protobuf;
    protobuf.set_event(TSpeechKitRequestProto_TEventSource_EEvent_Directive);
    protobuf.set_source(TSpeechKitRequestProto_TEventSource_ESource_Software);
    protobuf.set_type(TSpeechKitRequestProto_TEventSource_EType_Text);
    protobuf.set_id(TString(id));
    return protobuf;
}

VinsRequest::EventSource VinsRequest::createSoftwareSpotterEventSource(const std::string& id) {
    EventSource protobuf;
    protobuf.set_event(TSpeechKitRequestProto_TEventSource_EEvent_Spotter);
    protobuf.set_source(TSpeechKitRequestProto_TEventSource_ESource_Software);
    protobuf.set_type(TSpeechKitRequestProto_TEventSource_EType_Voice);
    protobuf.set_id(TString(id));
    return protobuf;
}

VinsRequest::EventSource VinsRequest::createVoiceprintMatchEventSource(const std::string& id) {
    EventSource protobuf;
    protobuf.set_event(TSpeechKitRequestProto_TEventSource_EEvent_Directive);
    protobuf.set_source(TSpeechKitRequestProto_TEventSource_ESource_Software);
    protobuf.set_type(TSpeechKitRequestProto_TEventSource_EType_VoiceprintMatch);
    protobuf.set_id(TString(id));
    return protobuf;
}

Json::Value VinsRequest::buildNextTrackSemanticFrame(bool setPause) {
    Json::Value semanticFrame;
    semanticFrame["name"] = "@@mm_semantic_frame";
    semanticFrame["type"] = "server_action";

    auto& semanticFramePayload = semanticFrame["payload"];

    auto& nextTrack = semanticFramePayload["typed_semantic_frame"]["player_next_track_semantic_frame"];
    nextTrack["set_pause"]["bool_value"] = setPause;

    auto& analytics = semanticFramePayload["analytics"];
    analytics["origin"] = "RemoteControl";
    analytics["purpose"] = "player_next_track";

    return semanticFrame;
}

Json::Value VinsRequest::buildPrevTrackSemanticFrame(bool setPause) {
    Json::Value semanticFrame;
    semanticFrame["name"] = "@@mm_semantic_frame";
    semanticFrame["type"] = "server_action";

    auto& semanticFramePayload = semanticFrame["payload"];

    auto& prevTrack = semanticFramePayload["typed_semantic_frame"]["player_prev_track_semantic_frame"];
    prevTrack["set_pause"]["bool_value"] = setPause;

    auto& analytics = semanticFramePayload["analytics"];
    analytics["origin"] = "RemoteControl";
    analytics["purpose"] = "player_prev_track";

    return semanticFrame;
}

Json::Value VinsRequest::buildStartMusicSemanticFrame() {
    Json::Value semanticFrame;
    semanticFrame["name"] = "@@mm_semantic_frame";
    semanticFrame["type"] = "server_action";

    auto& semanticFramePayload = semanticFrame["payload"];

    auto& musicPlay = semanticFramePayload["typed_semantic_frame"]["music_play_semantic_frame"];
    musicPlay["disable_nlg"]["bool_value"] = true;

    auto& analytics = semanticFramePayload["analytics"];
    analytics["origin"] = "SmartSpeaker";
    analytics["purpose"] = "play_music";

    return semanticFrame;
}

Json::Value VinsRequest::buildPlayerContinueSemanticFrame() {
    Json::Value semanticFrame;
    semanticFrame["name"] = "@@mm_semantic_frame";
    semanticFrame["type"] = "server_action";

    auto& semanticFramePayload = semanticFrame["payload"];

    semanticFramePayload["typed_semantic_frame"]["player_continue_semantic_frame"] = Json::objectValue;
    auto& analytics = semanticFramePayload["analytics"];
    analytics["origin"] = "SmartSpeaker";
    analytics["purpose"] = "player_continue";

    return semanticFrame;
}

Json::Value VinsRequest::buildGuestEnrollmentStartSemanticFrame(const std::string& puid) {
    Json::Value semanticFrame;
    semanticFrame["name"] = "@@mm_semantic_frame";
    semanticFrame["type"] = "server_action";

    auto& semanticFramePayload = semanticFrame["payload"];

    auto& enrollmentStart = semanticFramePayload["typed_semantic_frame"]["guest_enrollment_start_semantic_frame"] = Json::objectValue;
    enrollmentStart["puid"]["string_value"] = puid;

    auto& analytics = semanticFramePayload["analytics"];
    analytics["origin"] = "SmartSpeaker";
    analytics["purpose"] = "voiceprint_enroll";

    return semanticFrame;
}

Json::Value VinsRequest::buildEnrollmentStatusSemanticFrame(const std::string& puid, bool success, const std::string& failureReason, const std::string& details) {
    Json::Value semanticFrame;
    semanticFrame["name"] = "@@mm_semantic_frame";
    semanticFrame["type"] = "server_action";

    auto& semanticFramePayload = semanticFrame["payload"];

    auto& enrollmentStatusFrame = semanticFramePayload["typed_semantic_frame"]["multiaccount_enrollment_status_semantic_frame"] = Json::objectValue;
    if (!puid.empty()) {
        enrollmentStatusFrame["puid"]["string_value"] = puid;
    }

    auto& enrollmentStatus = enrollmentStatusFrame["status"]["enrollment_status"] = Json::objectValue;
    enrollmentStatus["success"] = success;

    if (!failureReason.empty()) {
        enrollmentStatus["failure_reason"] = failureReason;
    }
    if (!details.empty()) {
        enrollmentStatus["failure_reason_details"] = details;
    }

    auto& analytics = semanticFramePayload["analytics"];
    analytics["origin"] = "SmartSpeaker";
    analytics["purpose"] = "voiceprint_enroll";

    return semanticFrame;
}

Json::Value VinsRequest::buildGuestEnrollmentFinishSemanticFrame() {
    Json::Value semanticFrame;
    semanticFrame["name"] = "@@mm_semantic_frame";
    semanticFrame["type"] = "server_action";

    auto& semanticFramePayload = semanticFrame["payload"];
    semanticFramePayload["typed_semantic_frame"]["guest_enrollment_finish_semantic_frame"] = Json::objectValue;

    auto& analytics = semanticFramePayload["analytics"];
    analytics["origin"] = "SmartSpeaker";
    analytics["purpose"] = "voiceprint_enroll";

    return semanticFrame;
}

Json::Value VinsRequest::buildEqualizerSettingsRequest() {
    Json::Value semanticFrame;
    semanticFrame["name"] = "@@mm_semantic_frame";
    semanticFrame["type"] = "server_action";

    auto& semanticFramePayload = semanticFrame["payload"];
    semanticFramePayload["typed_semantic_frame"]["get_equalizer_settings_semantic_frame"] = Json::objectValue;

    auto& analytics = semanticFramePayload["analytics"];
    analytics["origin"] = "SmartSpeaker";
    analytics["purpose"] = "get_equalizer_settings";

    return semanticFrame;
}

void VinsRequest::setIsPrefetch(bool isPrefetch) {
    isPrefetch_ = isPrefetch;
}

bool VinsRequest::isPrefetch() const {
    return isPrefetch_;
}

bool VinsRequest::getIsSilent() const {
    return isSilent_;
}

void VinsRequest::setIsSilent(bool silent) {
    isSilent_ = silent;
}

bool VinsRequest::isIgnoreCriticalUpdate() const {
    return isIgnoreCriticalUpdate_;
}

void VinsRequest::setIsIgnoreCriticalUpdate(bool value) {
    isIgnoreCriticalUpdate_ = value;
}

bool VinsRequest::getResetSession() const {
    return resetSession_;
}

void VinsRequest::setResetSession(bool value) {
    resetSession_ = value;
}

const std::string& VinsRequest::getMessageId() const {
    return messageId_;
}

void VinsRequest::setMessageId(const std::string& value) {
    messageId_ = value;
}

const std::string& VinsRequest::getParentMessageId() const {
    return parentMessageId_;
}

void VinsRequest::setParentMessageId(const std::string& value) {
    parentMessageId_ = value;
}

bool VinsRequest::getVoiceSession() const {
    return voiceSession_;
}

void VinsRequest::setVoiceSession(bool value) {
    voiceSession_ = value;
}

bool VinsRequest::isGetNext() const {
    return isGetNext_;
}

void VinsRequest::setIsGetNext(bool value) {
    isGetNext_ = value;
}

bool VinsRequest::eteShouldBindToPrevRequest() const {
    return eteShouldBindToPrevRequest_;
}

void VinsRequest::setEteShouldBindToPrevRequest(bool value) {
    eteShouldBindToPrevRequest_ = value;
}

std::optional<std::chrono::milliseconds> VinsRequest::getStartingSilenceTimeout() const {
    return startingSilenceTimeout_;
}

void VinsRequest::setStartingSilenceTimeout(std::chrono::milliseconds value) {
    startingSilenceTimeout_ = value;
}

std::string VinsRequest::toString() const {
    std::stringstream ss;
    const auto* type = TSpeechKitRequestProto_TEventSource::EType_descriptor()->FindValueByNumber(eventSource_.type());

    ss << "(" << this << ") " << type ? type->name() : "Unknown";
    if (eventSource_.type() == TSpeechKitRequestProto_TEventSource_EType_Text) {
        ss << "." << tryGetString(event_, "name");
    }

    ss << ", isPrefetch=" << isPrefetch();
    ss << ", isSilent=" << getIsSilent();
    ss << ", id=" << getId();

    return ss.str();
}

bool VinsRequest::getIsEnqueued() const {
    return isEnqueued_;
}

void VinsRequest::setEnqueued(bool value) {
    isEnqueued_ = value;
}

void VinsRequest::setGuestOptions(TGuestOptions value) {
    guestOptions_ = std::move(value);
}

const TGuestOptions& VinsRequest::getGuestOptions() const {
    return guestOptions_;
}

proto::VinsRequest VinsRequest::convertToVinsRequestProtobuf(const std::shared_ptr<VinsRequest>& request) {
    proto::VinsRequest protobuf;

    protobuf.set_id(TString(request->getId()));
    protobuf.set_event_json(jsonToString(request->getEvent()));
    protobuf.set_reset_session(request->getResetSession());
    protobuf.set_is_silent(request->getIsSilent());
    protobuf.set_is_enqueued(request->getIsEnqueued());
    protobuf.set_is_parallel(request->getIsParallel());
    protobuf.set_ignore_answer(request->getIgnoreAnswer());
    protobuf.set_is_reminder(request->getIsReminder());
    protobuf.set_is_prefetch(request->isPrefetch());
    protobuf.set_is_ignore_critical_update(request->isIgnoreCriticalUpdate());
    protobuf.set_voice_session(request->getVoiceSession());
    const auto optTimeoutMs = request->getStartingSilenceTimeout();
    if (optTimeoutMs.has_value()) {
        protobuf.set_starting_silence_timeout_ms(optTimeoutMs->count());
    }
    protobuf.mutable_guest_options()->CopyFrom(request->guestOptions_);
    protobuf.mutable_event_source()->CopyFrom(request->getEventSource());

    return protobuf;
}

std::shared_ptr<VinsRequest> VinsRequest::createVinsRequestFromProtobuf(const proto::VinsRequest& protobuf) {
    std::string requestId;
    if (!protobuf.id().empty()) {
        requestId = protobuf.id();
    } else {
        requestId = makeUUID();
    }
    Json::Value event = parseJson(protobuf.event_json());

    auto request = std::make_shared<VinsRequest>(event, protobuf.event_source(), requestId);

    request->setResetSession(protobuf.reset_session());
    request->setIsParallel(protobuf.is_parallel());
    request->setEnqueued(protobuf.is_enqueued() && !request->getIsParallel());
    request->setIsSilent(protobuf.is_silent());
    request->setIgnoreAnswer(protobuf.ignore_answer());
    request->setIsReminder(protobuf.is_reminder());
    request->setIsPrefetch(protobuf.is_prefetch());
    request->setIsIgnoreCriticalUpdate(protobuf.is_ignore_critical_update());
    request->setVoiceSession(protobuf.voice_session());
    if (protobuf.has_starting_silence_timeout_ms()) {
        request->setStartingSilenceTimeout(std::chrono::milliseconds(protobuf.starting_silence_timeout_ms()));
    }

    if (protobuf.has_guest_options()) {
        request->guestOptions_ = protobuf.guest_options();
    }

    return request;
}
