#include "converters.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/protobuf_utils/json.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/generic/function.h>

#include <google/protobuf/util/message_differencer.h>

using namespace YandexIO;
using namespace quasar;

YIO_DEFINE_LOG_MODULE("device_state_capability");

namespace {

    NAlice::TDeviceState::TAudioPlayer::TPlayerState convertAudioClientState(const proto::AudioClientState state) {
        switch (state) {
            case proto::AudioClientState::BUFFERING: {
                return NAlice::TDeviceState::TAudioPlayer::Buffering;
            }
            case proto::AudioClientState::FINISHED: {
                return NAlice::TDeviceState::TAudioPlayer::Finished;
            }
            case proto::AudioClientState::PAUSED: {
                return NAlice::TDeviceState::TAudioPlayer::Paused;
            }
            case proto::AudioClientState::PLAYING: {
                return NAlice::TDeviceState::TAudioPlayer::Playing;
            }
            case proto::AudioClientState::STOPPED: {
                return NAlice::TDeviceState::TAudioPlayer::Stopped;
            }
            case proto::AudioClientState::FAILED:
            case proto::AudioClientState::IDLE: {
                return NAlice::TDeviceState::TAudioPlayer::Idle;
            }
        }
    }

    proto::AudioClientState convertAudioClientState(const NAlice::TDeviceState::TAudioPlayer::TPlayerState state) {
        switch (state) {
            case NAlice::TDeviceState::TAudioPlayer::Buffering: {
                return proto::AudioClientState::BUFFERING;
            }
            case NAlice::TDeviceState::TAudioPlayer::Finished: {
                return proto::AudioClientState::FINISHED;
            }
            case NAlice::TDeviceState::TAudioPlayer::Paused: {
                return proto::AudioClientState::PAUSED;
            }
            case NAlice::TDeviceState::TAudioPlayer::Playing: {
                return proto::AudioClientState::PLAYING;
            }
            case NAlice::TDeviceState::TAudioPlayer::Stopped: {
                return proto::AudioClientState::STOPPED;
            }
            case NAlice::TDeviceState::TAudioPlayer::Unknown:
            case NAlice::TDeviceState::TAudioPlayer::Idle: {
                return proto::AudioClientState::IDLE;
            }
        }
    }

    NAlice::TDeviceState::TMultiroom::EMultiroomMode convertMultiroomMode(const proto::MultiroomState::Mode mode) {
        switch (mode) {
            case proto::MultiroomState::SLAVE: {
                return NAlice::TDeviceState::TMultiroom::Slave;
            }
            case proto::MultiroomState::MASTER: {
                return NAlice::TDeviceState::TMultiroom::Master;
            }
            case proto::MultiroomState::NONE:
            default: {
                return NAlice::TDeviceState::TMultiroom::Unknown;
            }
        }
    }

    std::string convertMrSessionId(const proto::MultiroomBroadcast& broadcast) {
        return broadcast.device_id() + ":" + broadcast.multiroom_token() + ":" + std::to_string(broadcast.session_timestamp_ms());
    }

    TString convertProvider(proto::Provider provider) {
        static const auto* descriptor = proto::Provider_descriptor();
        const auto* value = descriptor->FindValueByNumber(provider);
        if (!value) {
            return "unknown";
        }
        std::string name = value->name();
        std::transform(name.begin(), name.end(), name.begin(), ::tolower);
        return name;
    }

    proto::Provider convertProvider(std::string provider) {
        static const auto* descriptor = proto::Provider_descriptor();
        std::transform(provider.begin(), provider.end(), provider.begin(), ::toupper);
        const auto* value = descriptor->FindValueByName(TString(provider));
        if (!value) {
            return proto::Provider::UNKNOWN;
        }
        return proto::Provider{value->number()};
    }

    std::optional<proto::MediaItem::Type> convertMediaItemType(std::string type) {
        static const auto* descriptor = proto::MediaItem::Type_descriptor();
        std::transform(type.begin(), type.end(), type.begin(), ::toupper);
        const auto* value = descriptor->FindValueByName(TString(type));
        if (!value) {
            return std::nullopt;
        }
        return proto::MediaItem::Type{value->number()};
    }

    NAlice::TWatchedVideoItem convertVideoItem(const quasar::proto::VideoState& state, const quasar::proto::MediaItem& item) {
        NAlice::TWatchedVideoItem converted;

        const auto& progress = state.progress();
        converted.MutableProgress()->SetDuration(progress.duration());
        converted.MutableProgress()->SetPlayed(progress.played());

        converted.SetTimestamp(state.timestamp_ms() / 1000);

        if (item.has_provider()) {
            converted.SetProviderName(convertProvider(item.provider()));
        }
        if (item.has_item_id()) {
            converted.SetProviderItemId(item.item_id());
        }
        if (item.has_play_uri()) {
            converted.SetPlayUri(item.play_uri());
        }
        if (item.has_season()) {
            converted.SetSeason(item.season());
        }
        if (item.has_episode()) {
            converted.SetEpisode(item.episode());
        }

        converted.SetAudioLanguage(state.video().audio_language());
        converted.SetSubtitlesLanguage(state.video().subtitles_language());

        return converted;
    }

    quasar::proto::MediaItem convertMediaItem(const NAlice::TVideoItem& item) {
        quasar::proto::MediaItem converted;
        converted.set_provider(convertProvider(item.GetProviderName()));
        converted.set_item_id(item.GetProviderItemId());
        converted.set_play_uri(item.GetPlayUri());
        converted.set_season(item.GetSeason());
        converted.set_episode(item.GetEpisode());
        if (const auto type = convertMediaItemType(item.GetType()); type.has_value()) {
            converted.set_type(*type);
        }
        return converted;
    }

    NAlice::TWatchedTvShowItem convertTvShowItem(const quasar::proto::VideoState& state) {
        NAlice::TWatchedTvShowItem converted;

        auto convertedItem = convertVideoItem(state, state.video().item());
        converted.MutableItem()->Swap(&convertedItem);

        convertedItem = convertVideoItem(state, state.video().tv_show_item());
        converted.MutableTvShowItem()->Swap(&convertedItem);

        return converted;
    }

    void experimentsFromArray(const Json::Value& config, NAlice::TExperimentsProto& experiments) {
        auto* storage = experiments.MutableStorage();
        for (const auto& exp : config) {
            if (exp.isString()) {
                NAlice::TExperimentsProto::TValue value;
                value.SetString("1");
                storage->insert({exp.asString(), value});
            } else {
                YIO_LOG_WARN("Experiment expected to be as string");
            }
        }
    }

    void experimentsFromDict(const Json::Value& config, NAlice::TExperimentsProto& experiments) {
        auto* storage = experiments.MutableStorage();
        for (auto it = config.begin(); it != config.end(); ++it) {
            NAlice::TExperimentsProto::TValue value;
            bool success = true;
            switch (it->type()) {
                case Json::stringValue: {
                    value.SetString(it->asString());
                    break;
                }
                case Json::booleanValue: {
                    value.SetBoolean(it->asBool());
                    break;
                }
                case Json::intValue:
                case Json::uintValue: {
                    value.SetInteger(it->asInt());
                    break;
                }
                case Json::realValue: {
                    value.SetNumber(it->asDouble());
                    break;
                }
                default:
                    success = false;
                    break;
            }
            if (success) {
                storage->insert({it.name(), value});
            } else {
                YIO_LOG_WARN("Unexpected type of experiment value: " << *it);
            }
        }
    }

    std::optional<TString> getStringFromStruct(const google::protobuf::Struct& protoStruct, const TString& field) {
        if (protoStruct.fields().contains(field)) {
            const auto& value = protoStruct.fields().at(field);
            if (value.has_string_value()) {
                return value.string_value();
            }
        }
        return std::nullopt;
    }

    std::vector<google::protobuf::Struct> getStructListFromStruct(const google::protobuf::Struct& protoStruct, const TString& field) {
        std::vector<google::protobuf::Struct> result;
        if (protoStruct.fields().contains(field)) {
            const auto& value = protoStruct.fields().at(field);
            if (!value.has_list_value()) {
                return result;
            }
            const auto& listValues = value.list_value().values();
            for (const auto& val : listValues) {
                if (val.has_struct_value()) {
                    result.push_back(val.struct_value());
                }
            }
        }
        return result;
    }

    std::vector<TString> getStringListFromStruct(const google::protobuf::Struct& protoStruct, const TString& field) {
        std::vector<TString> result;
        if (protoStruct.fields().contains(field)) {
            const auto& value = protoStruct.fields().at(field);
            if (!value.has_list_value()) {
                return result;
            }
            const auto& listValues = value.list_value().values();
            for (const auto& val : listValues) {
                if (val.has_string_value()) {
                    result.push_back(val.string_value());
                }
            }
        }
        return result;
    }

    std::optional<google::protobuf::Struct> getStructFromStruct(const google::protobuf::Struct& protoStruct, const TString& field) {
        if (protoStruct.fields().contains(field)) {
            const auto& value = protoStruct.fields().at(field);
            if (value.has_struct_value()) {
                return value.struct_value();
            }
        }
        return std::nullopt;
    }

    std::optional<bool> getBoolFromStruct(const google::protobuf::Struct& protoStruct, const TString& field) {
        if (protoStruct.fields().contains(field)) {
            const auto& value = protoStruct.fields().at(field);
            if (value.has_bool_value()) {
                return value.bool_value();
            }
        }
        return std::nullopt;
    }

    std::optional<double> getNumberFromStruct(const google::protobuf::Struct& protoStruct, const TString& field) {
        if (protoStruct.fields().contains(field)) {
            const auto& value = protoStruct.fields().at(field);
            if (value.has_number_value()) {
                return value.number_value();
            }
        }
        return std::nullopt;
    }

    TString convertArtists(const std::vector<google::protobuf::Struct>& artists) {
        std::stringstream converted;
        bool first = true;
        for (const auto& artist : artists) {
            const auto name = getStringFromStruct(artist, "name");
            if (!name) {
                continue;
            }
            if (!std::exchange(first, false)) {
                converted << ", ";
            }
            converted << *name;
            {
                const auto decomposed = getStructListFromStruct(artist, "decomposed");
                for (const auto& decomposedArtist : decomposed) {
                    const auto decomposedArtistName = getStringFromStruct(decomposedArtist, "name");
                    if (!decomposedArtistName) {
                        continue;
                    }
                    converted << *decomposedArtistName;
                }
            }
            {
                const auto decomposed = getStringListFromStruct(artist, "decomposed");
                for (const auto& decomposedArtist : decomposed) {
                    converted << decomposedArtist;
                }
            }
        }
        return converted.str();
    }

    bool getRadioPause(const google::protobuf::Struct& radio) {
        if (radio.fields().contains("player")) {
            const auto& playerState = radio.fields().at("player");
            if (playerState.has_struct_value() && playerState.struct_value().fields().contains("pause")) {
                const auto& pause = playerState.struct_value().fields().at("pause");
                return pause.has_bool_value() && pause.bool_value();
            }
        }

        return true;
    }

    std::string getRadioId(const google::protobuf::Struct& radio) {
        if (radio.fields().contains("currently_playing")) {
            const auto& currentyPlaying = radio.fields().at("currently_playing");
            if (currentyPlaying.has_struct_value() && currentyPlaying.struct_value().fields().contains("radioId")) {
                const auto& radioId = currentyPlaying.struct_value().fields().at("radioId");
                if (radioId.has_string_value()) {
                    return radioId.string_value();
                }
            }
        }

        return "";
    }

    std::string getRadioTitle(const google::protobuf::Struct& radio) {
        if (radio.fields().contains("currently_playing")) {
            const auto& currentyPlaying = radio.fields().at("currently_playing");
            if (currentyPlaying.has_struct_value() && currentyPlaying.struct_value().fields().contains("radioTitle")) {
                const auto& radioTitle = currentyPlaying.struct_value().fields().at("radioTitle");
                if (radioTitle.has_string_value()) {
                    return radioTitle.string_value();
                }
            }
        }

        return "";
    }

    template <class S>
    void readValue(const std::string& sval, S setter) noexcept {
        using R = TFunctionArg<S, 0>;
        try {
            if constexpr (std::is_unsigned_v<R> && std::is_integral_v<R>) {
                R rval = static_cast<R>(std::stoull(sval));
                setter(rval);
            } else if constexpr (std::is_floating_point_v<R>) {
                R rval = static_cast<R>(std::stold(sval));
                setter(rval);
            } else {
                static_assert(TDependentFalse<R>);
            }
        } catch (...) {
        }
    }

} // namespace

NAlice::TDeviceState::TRcuState YandexIO::convertRcuState(const proto::RcuState& state) {
    NAlice::TDeviceState::TRcuState converted;
    if (state.has_is_rcu_connected()) {
        converted.SetIsRcuConnected(state.is_rcu_connected());
    }
    if (state.has_total_codesets()) {
        converted.SetTotalCodesets(state.total_codesets());
    }
    if (state.has_rcu_capabilities() && state.rcu_capabilities().has_can_make_sounds()) {
        converted.MutableRcuCapabilities()->SetCanMakeSounds(state.rcu_capabilities().can_make_sounds());
    }
    switch (state.setup_state()) {
        case proto::RcuState::LINK:
            converted.SetSetupState(NAlice::TDeviceState::TRcuState::Link);
            break;
        case proto::RcuState::AUTO:
            converted.SetSetupState(NAlice::TDeviceState::TRcuState::Auto);
            break;
        case proto::RcuState::CHECK:
            converted.SetSetupState(NAlice::TDeviceState::TRcuState::Check);
            break;
        case proto::RcuState::ADVANCED:
            converted.SetSetupState(NAlice::TDeviceState::TRcuState::Advanced);
            break;
        case proto::RcuState::MANUAL:
            converted.SetSetupState(NAlice::TDeviceState::TRcuState::Manual);
            break;
        case proto::RcuState::NONE:
        default:
            break;
    }
    return converted;
}

NAlice::TDeviceState::TScreen YandexIO::convertScreenState(const proto::AppState::MediaInfo& state) {
    NAlice::TDeviceState::TScreen converted;
    for (const auto& item : state.supported_screen_resolutions()) {
        NAlice::TDeviceState::TScreen::EScreenResolution resolution;
        if (NAlice::TDeviceState::TScreen::EScreenResolution_Parse(item, &resolution)) {
            converted.MutableSupportedScreenResolutions()->Add(resolution);
        }
    }
    for (const auto& item : state.dynamic_ranges()) {
        NAlice::TDeviceState::TScreen::EDynamicRange resolution;
        if (NAlice::TDeviceState::TScreen::EDynamicRange_Parse(item, &resolution)) {
            converted.MutableDynamicRanges()->Add(resolution);
        }
    }
    if (state.has_hdcp_level()) {
        NAlice::TDeviceState::TScreen::EHdcpLevel hdcpLevel;
        if (NAlice::TDeviceState::TScreen::EHdcpLevel_Parse(state.hdcp_level(), &hdcpLevel)) {
            converted.SetHdcpLevel(hdcpLevel);
        }
    }
    return converted;
}

NAlice::TDeviceState::TPackagesState YandexIO::convertPackagesState(const proto::PackagesState& state) {
    NAlice::TDeviceState::TPackagesState converted;
    for (const auto& package : state.installed()) {
        auto* convertedPackage = converted.MutableInstalled()->Add();
        convertedPackage->SetMainActivity(package.main_activity());
        if (package.has_package_info()) {
            convertedPackage->MutablePackageInfo()->SetName(package.package_info().name());
            convertedPackage->MutablePackageInfo()->SetHumanReadableName(package.package_info().human_readable_name());
        }
    }
    return converted;
}

NAlice::TDeviceState::TMusic YandexIO::convertMusicState(const quasar::proto::AppState::MusicState& state, int64_t lastPlayTs) {
    NAlice::TDeviceState::TMusic converted;
    converted.MutablePlayer()->SetPause(state.is_paused());
    if (state.has_timestamp_ms()) {
        converted.MutablePlayer()->SetTimestamp(state.timestamp_ms() / 1000);
    }
    converted.SetLastPlayTimestamp(lastPlayTs);
    converted.MutableCurrentlyPlaying()->SetLastPlayTimestamp(lastPlayTs);
    if (state.has_json_track_info()) {
        auto track = quasar::convertJsonToProtobuf<google::protobuf::Struct>(state.json_track_info());
        if (!track.has_value()) {
            converted.MutableCurrentlyPlaying()->ClearRawTrackInfo();
            YIO_LOG_WARN("Failed to parse music track info: " << state.json_track_info());
        } else {
            converted.MutableCurrentlyPlaying()->MutableRawTrackInfo()->Swap(&(*track));
        }
    }
    if (state.has_current_track_id()) {
        converted.MutableCurrentlyPlaying()->SetTrackId(state.current_track_id());
    }
    if (state.has_uid()) {
        converted.SetPlaylistOwner(state.uid());
    }
    if (state.has_session_id()) {
        converted.SetSessionId(state.session_id());
    }
    return converted;
}

google::protobuf::Struct YandexIO::convertRadioState(const quasar::proto::AppState::RadioState& state, int64_t lastPlayTs) {
    Json::Value jsonState;

    Json::Value currentlyPlaying = Json::objectValue;
    currentlyPlaying["radioId"] = state.radio_id();
    currentlyPlaying["radioTitle"] = state.radio_title();
    currentlyPlaying["last_play_timestamp"] = lastPlayTs;

    const bool isPaused = state.is_paused();
    Json::Value player = Json::objectValue;
    player["pause"] = isPaused;
    if (isPaused && state.has_timestamp_ms()) {
        player["timestamp"] = state.timestamp_ms() / 1000;
    }

    jsonState["currently_playing"] = std::move(currentlyPlaying);
    jsonState["player"] = std::move(player);
    jsonState["last_play_timestamp"] = lastPlayTs;
    jsonState["color"] = state.color();
    jsonState["image_url"] = state.cover_uri();
    if (state.has_uid()) {
        jsonState["playlist_owner"] = state.uid();
    }

    const auto stateStr = quasar::jsonToString(jsonState);
    if (const auto converted = quasar::convertJsonToProtobuf<google::protobuf::Struct>(stateStr)) {
        return *converted;
    } else {
        YIO_LOG_ERROR_EVENT("DeviceStateConverter.FailedConvertRadioState", "Failed to convert radio state from json: " << stateStr);
        return google::protobuf::Struct{};
    }
}

NAlice::TDeviceState::TAudioPlayer YandexIO::convertAudioPlayerState(const quasar::proto::AudioClientEvent& event) {
    NAlice::TDeviceState::TAudioPlayer converted;

    converted.SetPlayerState(convertAudioClientState(event.state()));
    converted.SetLastPlayTimestamp(event.last_play_timestamp());
    converted.SetLastStopTimestamp(event.last_stop_timestamp());
    converted.SetOffsetMs(event.audio().position_sec() * 1000);
    converted.SetPlayedMs(event.audio().played_sec() * 1000);
    converted.SetDurationMs(event.audio().duration_sec() * 1000);
    auto* currentlyPlaying = converted.MutableCurrentlyPlaying();
    currentlyPlaying->SetLastPlayTimestamp(event.last_play_timestamp());
    currentlyPlaying->SetStreamId(event.audio().id());
    currentlyPlaying->SetStreamType(event.audio().type());
    currentlyPlaying->SetTitle(event.audio().metadata().title());
    currentlyPlaying->SetSubTitle(event.audio().metadata().subtitle());

    const auto context = quasar::tryParseJson(event.audio().context());
    if (!context.has_value()) {
        YIO_LOG_WARN("Failed to prase audio context. Context: " << event.audio().context());
        return converted;
    }

    auto fillFromArray = [](auto* meta, const Json::Value& array) {
        for (const auto& item : array) {
            auto key = tryGetString(item, "key");
            auto val = tryGetString(item, "value");
            meta->insert({std::move(key), std::move(val)});
        }
    };

    auto fillFromDict = [](auto* meta, const Json::Value& dict) {
        for (auto item = dict.begin(); item != dict.end(); item++) {
            if (!item->isString()) {
                YIO_LOG_WARN("Found not string value in scenario meta: " << *item);
                continue;
            }
            meta->insert({item.name(), item->asString()});
        }
    };

    converted.ClearScenarioMeta();
    auto* meta = converted.MutableScenarioMeta();
    const auto& scenarionMeta = (*context)["scenario_meta"];
    if (scenarionMeta.isArray()) {
        fillFromArray(meta, scenarionMeta);
    } else if (scenarionMeta.isObject()) {
        fillFromDict(meta, scenarionMeta);
    }
    auto artImageUrl = event.audio().metadata().art_image_url();
    meta->insert({"__art_image_url", artImageUrl});
    meta->insert({"__vins_request_id", event.audio().analytics_context().vins_request_id()});
    meta->insert({"__provider_name", event.audio().provider_name()});
    meta->insert({"__hide_progress_bar", (event.audio().metadata().hide_progress_bar() ? "1" : "0")});

    // Multiroom  needs
    if (event.audio().basetime_ns() > 0) {
        meta->insert({"__is_multiroom_slave", (event.player_descriptor().is_multiroom_slave() ? "1" : "0")});
        meta->insert({"__play_pause_id", event.audio().play_pause_id()});
        meta->insert({"__multiroom_token", event.audio().multiroom_token()});
        meta->insert({"__basetime_ns", std::to_string(event.audio().basetime_ns())});
        meta->insert({"__position_ns", std::to_string(event.audio().position_ns())});
        meta->insert({"__url", event.audio().url()});
        meta->insert({"__clock_id", event.audio().clock_id()});
        meta->insert({"__normalization_tp", std::to_string(event.audio().normalization().true_peak())});
        meta->insert({"__normalization_il", std::to_string(event.audio().normalization().integrated_loudness())});
        meta->insert({"__normalization_tl", std::to_string(event.audio().normalization().target_lufs())});
    }

    const auto& musicMeta = event.audio().metadata().music_metadata();
    meta->insert({"__musicmeta_id", musicMeta.id()});
    meta->insert({"__musicmeta_type", musicMeta.type()});
    meta->insert({"__musicmeta_description", musicMeta.description()});
    if (musicMeta.has_shuffled()) {
        meta->insert({"__musicmeta_shuffled", (musicMeta.shuffled() ? "1" : "0")});
    }
    if (musicMeta.has_repeat_mode()) {
        meta->insert({"__musicmeta_repeat_mode", musicMeta.repeat_mode()});
    }
    meta->insert({"__musicmeta_prev_id", musicMeta.prev().id()});
    meta->insert({"__musicmeta_prev_type", musicMeta.prev().type()});
    meta->insert({"__musicmeta_next_id", musicMeta.next().id()});
    meta->insert({"__musicmeta_next_type", musicMeta.next().type()});

    return converted;
}

NAlice::TDeviceState::TBluetooth YandexIO::convertBluetoothState(const quasar::proto::BluetoothPlayerState& state) {
    NAlice::TDeviceState::TBluetooth converted;
    converted.MutablePlayer()->SetPause(state.is_paused());
    if (state.has_timestamp_ms()) {
        converted.MutablePlayer()->SetTimestamp(state.timestamp_ms());
    }
    if (state.has_last_play_timestamp_ms()) {
        converted.SetLastPlayTimestamp(state.last_play_timestamp_ms());
        converted.MutableCurrentlyPlaying()->SetLastPlayTimestamp(state.last_play_timestamp_ms());
    }
    if (state.has_track_meta_info()) {
        Json::Value trackMetaInfoJson = Json::objectValue;
        const auto& meta = state.track_meta_info();
        if (meta.has_title()) {
            trackMetaInfoJson["title"] = meta.title();
        }
        if (meta.has_artist()) {
            Json::Value name;
            name["name"] = meta.artist();
            trackMetaInfoJson["artists"] = Json::arrayValue;
            trackMetaInfoJson["artists"].append(name);
        }
        if (meta.has_album()) {
            Json::Value title;
            title["title"] = meta.album();
            trackMetaInfoJson["albums"] = Json::arrayValue;
            trackMetaInfoJson["albums"].append(title);
        }
        trackMetaInfoJson["id"] = "stub";
        const auto metaInfoStr = quasar::jsonToString(trackMetaInfoJson);
        auto track = quasar::convertJsonToProtobuf<google::protobuf::Struct>(metaInfoStr);
        if (!track.has_value()) {
            converted.MutableCurrentlyPlaying()->ClearRawTrackInfo();
            YIO_LOG_WARN("Failed to parse bluetooth track info: " << metaInfoStr);
        } else {
            converted.MutableCurrentlyPlaying()->MutableRawTrackInfo()->Swap(&(*track));
        }
    }
    return converted;
}

NAlice::TDeviceState::TVideo YandexIO::convertVideoState(const yandex_io::proto::TVideo& state) {
    const auto videoJson = convertMessageToJsonString(state);
    const auto converted = convertJsonToProtobuf<NAlice::TDeviceState::TVideo>(videoJson);
    if (!converted.has_value()) {
        YIO_LOG_ERROR_EVENT("DeviceStateConverter.FailedConvertVideoStateToServerProto", "Failed to convert video state: " << videoJson);
        return NAlice::TDeviceState::TVideo{};
    }
    return *converted;
}

NAlice::TDeviceState::TVideo YandexIO::convertVideoState(const quasar::proto::VideoState& videoState, const quasar::proto::AppState::ScreenState& screenState, int64_t lastPlayTs, bool isScreenActive) {
    NAlice::TDeviceState::TVideo state;
    if (screenState.has_search_result() && screenState.search_result().has_raw_result()) {
        if (auto parsedResult = convertJsonToProtobuf<NAlice::TDeviceState::TVideo::TScreenState>(screenState.search_result().raw_result())) {
            if (screenState.screen_type() == proto::AppState::ScreenType::AppState_ScreenType_GALLERY ||
                screenState.screen_type() == proto::AppState::ScreenType::AppState_ScreenType_SEASON_GALLERY ||
                screenState.screen_type() == proto::AppState::ScreenType::AppState_ScreenType_TV_GALLERY) {
                for (const auto& val : screenState.visible_items()) {
                    parsedResult->MutableVisibleItems()->Add(val);
                }
            }
            state.MutableScreenState()->Swap(&*parsedResult);
        } else {
            YIO_LOG_ERROR_EVENT("DeviceStateConverter.FailedConvertScreenState", "Failed to convert search result to NAlice::TDeviceState::TVideo::TScreenState proto: " << screenState.search_result().raw_result());
        }
    }
    if (screenState.has_description_item() && screenState.description_item().has_raw()) {
        if (auto parsedResult = convertJsonToProtobuf<NAlice::TDeviceState::TVideo::TScreenState>(screenState.description_item().raw())) {
            state.MutableScreenState()->Swap(&*parsedResult);
        } else {
            YIO_LOG_ERROR_EVENT("DeviceStateConverter.FailedConvertScreenState", "Failed to convert description item to NAlice::TDeviceState::TVideo::TScreenState proto: " << screenState.description_item().raw());
        }
    }
    if (screenState.has_payment_item() && screenState.payment_item().has_raw()) {
        if (auto parsedResult = convertJsonToProtobuf<NAlice::TDeviceState::TVideo::TScreenState>(screenState.payment_item().raw())) {
            state.MutableScreenState()->Swap(&*parsedResult);
        } else {
            YIO_LOG_ERROR_EVENT("DeviceStateConverter.FailedConvertScreenState", "Failed to convert payment item to NAlice::TDeviceState::TVideo::TScreenState proto: " << screenState.payment_item().raw());
        }
    }
    if (screenState.has_web_view_state()) {
        const auto& protoState = screenState.web_view_state();
        NAlice::TDeviceState::TVideo::TScreenState converted;
        std::ostringstream scenario;
        if (protoState.has_backend_scenario_name()) {
            converted.SetScenarioName(protoState.backend_scenario_name());
            scenario << protoState.backend_scenario_name();
        }
        if (protoState.has_scenario()) {
            const auto& viewKey = protoState.scenario();
            converted.SetViewKey(viewKey);
            if (scenario.tellp() > 0) {
                scenario << ":";
            }
            scenario << viewKey;
        }
        if (scenario.tellp() > 0) {
            converted.SetScenario(scenario.str());
        }
        state.MutableScreenState()->Swap(&converted);
    }
    if (screenState.has_web_page_state() && (isScreenActive || screenState.is_hdmi_connected())) {
        if (auto parsedResult = convertJsonToProtobuf<google::protobuf::Struct>(screenState.web_page_state())) {
            state.MutableViewState()->Swap(&*parsedResult);
        } else {
            YIO_LOG_ERROR_EVENT("DeviceStateConverter.FailedConvertViewState", "Failed to convert web page state to google::protobuf::Struct: " << screenState.web_page_state());
        }
    }
    if (screenState.has_screen_type()) {
        static const auto* screenTypeDescriptor = proto::AppState::ScreenType_descriptor();
        std::string type = screenTypeDescriptor->FindValueByNumber(screenState.screen_type())->name();
        std::transform(type.begin(), type.end(), type.begin(), ::tolower);
        state.SetCurrentScreen(TString(type));
    }
    if (videoState.has_video() && videoState.video().has_raw()) {
        if (videoState.has_is_paused()) {
            state.MutablePlayer()->SetPause(videoState.is_paused());
        }

        if (auto currenlyPlayingParsed = convertJsonToProtobuf<NAlice::TDeviceState::TVideo::TCurrentlyPlaying>(videoState.video().raw(), true)) {
            currenlyPlayingParsed->SetLastPlayTimestamp(lastPlayTs);

            if (videoState.has_progress()) {
                currenlyPlayingParsed->MutableProgress()->SetDuration(videoState.progress().duration());
                currenlyPlayingParsed->MutableProgress()->SetPlayed(videoState.progress().played());
            }
            if (videoState.video().has_audio_language()) {
                currenlyPlayingParsed->SetAudioLanguage(videoState.video().audio_language());
            }
            if (videoState.video().has_subtitles_language()) {
                currenlyPlayingParsed->SetSubtitlesLanguage(videoState.video().subtitles_language());
            }

            state.MutableCurrentlyPlaying()->Swap(&*currenlyPlayingParsed);
        } else {
            YIO_LOG_ERROR_EVENT("DeviceStateConverter.FailedConvertVideoCurrentlyPlaying", "Cant convert video state to NAlice::TDeviceState::TVideo::TCurrentlyPlaying: " << videoState.video().raw());
        }
    }
    state.SetLastPlayTimestamp(lastPlayTs);
    return state;
}

NAlice::TDeviceState::TMultiroom YandexIO::convertMultiroomState(const proto::MultiroomState& state) {
    NAlice::TDeviceState::TMultiroom converted;
    converted.SetMode(convertMultiroomMode(state.mode()));
    for (const auto& peer : state.peers()) {
        converted.AddVisiblePeerDeviceIds(peer.device_id());
    }

    if (state.has_multiroom_broadcast()) {
        converted.SetMultiroomSessionId(convertMrSessionId(state.multiroom_broadcast()));
        converted.SetMasterDeviceId(state.multiroom_broadcast().device_id());
        converted.SetMultiroomToken(state.multiroom_broadcast().multiroom_token());

        for (const auto& roomDeviceId : state.multiroom_broadcast().room_device_ids()) {
            converted.AddRoomDeviceIds(roomDeviceId);
        }

        if (state.multiroom_broadcast().has_multiroom_params() && state.multiroom_broadcast().multiroom_params().has_music_params()) {
            auto* convertedMusic = converted.MutableMusic();
            const auto& music = state.multiroom_broadcast().multiroom_params().music_params();
            convertedMusic->MutablePlayer()->SetPause(music.is_paused());
            if (music.has_timestamp_ms()) {
                convertedMusic->MutablePlayer()->SetTimestamp(music.timestamp_ms() / 1000);
            }
            convertedMusic->SetSessionId(music.session_id());
            convertedMusic->SetLastPlayTimestamp(state.multiroom_broadcast().session_timestamp_ms());
            convertedMusic->SetPlaylistOwner(music.uid());
            convertedMusic->MutableCurrentlyPlaying()->SetTrackId(music.current_track_id());
            convertedMusic->MutableCurrentlyPlaying()->SetLastPlayTimestamp(state.multiroom_broadcast().session_timestamp_ms());
            if (auto trackInfo = convertJsonToProtobuf<google::protobuf::Struct>(music.json_track_info())) {
                convertedMusic->MutableCurrentlyPlaying()->MutableRawTrackInfo()->Swap(&*trackInfo);
            }
        }
    }

    return converted;
}

NAlice::TDeviceState::TLastWatched YandexIO::convertLastWatched(const quasar::proto::WatchedVideoState& state) {
    NAlice::TDeviceState::TLastWatched lastWatched;
    for (const auto& state : state.video()) {
        auto convertedState = convertVideoItem(state, state.video().item());
        lastWatched.MutableRawVideos()->Add()->Swap(&convertedState);
    }
    for (const auto& state : state.movies()) {
        auto convertedState = convertVideoItem(state, state.video().item());
        lastWatched.MutableRawMovies()->Add()->Swap(&convertedState);
    }
    for (const auto& state : state.episodes()) {
        auto convertedState = convertTvShowItem(state);
        lastWatched.MutableRawTvShows()->Add()->Swap(&convertedState);
    }
    return lastWatched;
}

NAlice::TDeviceState::TTimers YandexIO::convertTimers(const quasar::proto::TimersState& state, const std::optional<quasar::proto::Alarm>& currentlyPlaying) {
    NAlice::TDeviceState::TTimers converted;

    for (const auto& timer : state.timers()) {
        converted.MutableActiveTimers()->Add(convertTimer(timer, currentlyPlaying.has_value() && currentlyPlaying->id() == timer.id()));
    }
    if (currentlyPlaying.has_value() && currentlyPlaying->alarm_type() == proto::Alarm::TIMER) {
        converted.MutableActiveTimers()->Add(convertTimer(*currentlyPlaying, true));
    }

    return converted;
}

NAlice::TDeviceState::TTimers::TTimer YandexIO::convertTimer(const quasar::proto::Alarm& state, bool isPlaying) {
    NAlice::TDeviceState::TTimers::TTimer converted;

    converted.SetTimerId(state.id());
    converted.SetStartTimestamp(state.start_timestamp_ms() / 1000);
    converted.SetDuration(state.duration_seconds());
    converted.SetCurrentlyPlaying(isPlaying);
    converted.SetPaused(state.has_pause_timestamp_sec());

    const auto lastTimerCountdownTS = converted.GetPaused() ? state.pause_timestamp_sec() : getNowTimestampMs() / 1000;
    const auto pastSeconds = lastTimerCountdownTS - state.start_timestamp_ms() / 1000 - state.paused_seconds();
    converted.SetRemaining(pastSeconds < state.duration_seconds() ? state.duration_seconds() - pastSeconds : 0);

    if (state.alarm_type() == proto::Alarm::COMMAND_TIMER) {
        for (const auto& command : state.command_list()) {
            NAlice::TUntypedDirective directive;
            directive.SetName(command.name());
            auto payload = quasar::convertJsonToProtobuf<google::protobuf::Struct>(command.payload());
            if (!payload) {
                YIO_LOG_ERROR_EVENT("DeviceStateConverter.FailedToParseDirectivePayload", "Payload: " << command.payload());
                continue;
            }
            directive.MutablePayload()->Swap(&*payload);
            converted.MutableDirectives()->Add(std::move(directive));
        }
    }
    return converted;
}

std::unordered_set<TString> YandexIO::convertFeatures(const Json::Value& features) {
    std::unordered_set<TString> converted;
    if (features.isArray()) {
        for (const auto& feature : features) {
            if (feature.isString()) {
                converted.insert(feature.asString());
            } else {
                YIO_LOG_WARN("Feature expected to be as string");
            }
        }
    }
    return converted;
}

NAlice::TExperimentsProto YandexIO::convertExperiments(const Json::Value& arrayExps, const Json::Value& dictExps) {
    NAlice::TExperimentsProto converted;
    if (arrayExps.isArray()) {
        experimentsFromArray(arrayExps, converted);
    }
    if (dictExps.isObject()) {
        experimentsFromDict(dictExps, converted);
    }
    return converted;
}

NAlice::TDeviceState YandexIO::convertAppState(const quasar::proto::AppState& appState) {
    NAlice::TDeviceState state;
    if (appState.has_audio_player_state()) {
        auto converted = convertAudioPlayerState(appState.audio_player_state());
        state.MutableAudioPlayer()->Swap(&converted);
    }
    if (appState.has_bluetooth_player_state()) {
        auto converted = convertBluetoothState(appState.bluetooth_player_state());
        state.MutableBluetooth()->Swap(&converted);
    }
    if (appState.has_music_state()) {
        auto converted = convertMusicState(appState.music_state());
        state.MutableMusic()->Swap(&converted);
    }
    if (appState.has_radio_state()) {
        auto converted = convertRadioState(appState.radio_state());
        state.MutableRadio()->Swap(&converted);
    }
    if (appState.has_rcu_state()) {
        auto converted = convertRcuState(appState.rcu_state());
        state.MutableRcuState()->Swap(&converted);
    }
    if (appState.has_packages_state()) {
        auto converted = convertPackagesState(appState.packages_state());
        state.MutablePackagesState()->Swap(&converted);
    }
    if (appState.has_media_info()) {
        auto converted = convertScreenState(appState.media_info());
        state.MutableScreen()->Swap(&converted);
    }
    if (appState.has_video()) {
        auto converted = convertVideoState(appState.video());
        state.MutableVideo()->Swap(&converted);
    } else if (appState.has_video_state() || appState.has_screen_state()) {
        auto converted = convertVideoState(appState.video_state(), appState.screen_state());
    }
    return state;
}

void YandexIO::fillFromAppState(const std::shared_ptr<YandexIO::IDeviceStateCapability>& capability, const quasar::proto::AppState& appState) {
    const auto state = convertAppState(appState);
    const auto prevState = capability->getState().GetDeviceStateCapability().GetState().GetDeviceState();
    using namespace google::protobuf::util;
    if (state.HasAudioPlayer() && !MessageDifferencer::Equals(prevState.GetAudioPlayer(), state.GetAudioPlayer())) {
        capability->setAudioPlayerState(state.GetAudioPlayer());
    }
    if (state.HasRadio() && !MessageDifferencer::Equals(prevState.GetRadio(), state.GetRadio())) {
        capability->setRadioState(state.GetRadio());
    }
    if (state.HasMusic() && !MessageDifferencer::Equals(prevState.GetMusic(), state.GetMusic())) {
        capability->setMusicState(state.GetMusic());
    }
    if (state.HasVideo() && !MessageDifferencer::Equals(prevState.GetVideo(), state.GetVideo())) {
        capability->setVideoState(state.GetVideo());
    }
    if (state.HasRcuState() && !MessageDifferencer::Equals(prevState.GetRcuState(), state.GetRcuState())) {
        capability->setRcuState(state.GetRcuState());
    }
    if (state.HasScreen() && !MessageDifferencer::Equals(prevState.GetScreen(), state.GetScreen())) {
        capability->setScreenState(state.GetScreen());
    }
    if (state.HasPackagesState() && !MessageDifferencer::Equals(prevState.GetPackagesState(), state.GetPackagesState())) {
        capability->setPackagesState(state.GetPackagesState());
    }
}

quasar::proto::BluetoothPlayerState YandexIO::convertBluetoothState(const NAlice::TDeviceState::TBluetooth& state) {
    quasar::proto::BluetoothPlayerState converted;
    converted.set_is_paused(state.GetPlayer().GetPause());
    if (state.GetPlayer().HasTimestamp()) {
        converted.set_timestamp_ms(state.GetPlayer().GetTimestamp());
    }
    converted.set_last_play_timestamp_ms(state.GetLastPlayTimestamp());
    if (state.GetCurrentlyPlaying().HasRawTrackInfo()) {
        const auto& rawTrackInfo = state.GetCurrentlyPlaying().GetRawTrackInfo();
        auto* btMetaInfo = converted.mutable_track_meta_info();
        if (const auto title = getStringFromStruct(rawTrackInfo, "title")) {
            btMetaInfo->set_title(*title);
        }
        if (const auto artists = getStructListFromStruct(rawTrackInfo, "artists"); !artists.empty()) {
            const auto& artist = artists[0];
            if (artist.fields().contains("name")) {
                const auto& name = artist.fields().at("name");
                if (name.has_string_value()) {
                    btMetaInfo->set_artist(name.string_value());
                }
            }
        }
        if (const auto albums = getStructListFromStruct(rawTrackInfo, "albums"); !albums.empty()) {
            const auto& album = albums[0];
            if (album.fields().contains("title")) {
                const auto& title = album.fields().at("title");
                if (title.has_string_value()) {
                    btMetaInfo->set_album(title.string_value());
                }
            }
        }
    }
    return converted;
}

quasar::proto::RcuState YandexIO::convertRcuState(const NAlice::TDeviceState::TRcuState& state) {
    quasar::proto::RcuState converted;
    if (state.HasIsRcuConnected()) {
        converted.set_is_rcu_connected(state.GetIsRcuConnected());
    }
    if (state.HasTotalCodesets()) {
        converted.set_total_codesets(state.GetTotalCodesets());
    }
    if (state.HasRcuCapabilities() && state.GetRcuCapabilities().HasCanMakeSounds()) {
        converted.mutable_rcu_capabilities()->set_can_make_sounds(state.GetRcuCapabilities().GetCanMakeSounds());
    }
    switch (state.GetSetupState()) {
        case NAlice::TDeviceState::TRcuState::Link:
            converted.set_setup_state(quasar::proto::RcuState::LINK);
            break;
        case NAlice::TDeviceState::TRcuState::Auto:
            converted.set_setup_state(quasar::proto::RcuState::AUTO);
            break;
        case NAlice::TDeviceState::TRcuState::Check:
            converted.set_setup_state(quasar::proto::RcuState::CHECK);
            break;
        case NAlice::TDeviceState::TRcuState::Advanced:
            converted.set_setup_state(quasar::proto::RcuState::ADVANCED);
            break;
        case NAlice::TDeviceState::TRcuState::Manual:
            converted.set_setup_state(quasar::proto::RcuState::MANUAL);
            break;
        case NAlice::TDeviceState::TRcuState::None:
        default:
            converted.set_setup_state(quasar::proto::RcuState::NONE);
            break;
    }
    return converted;
}

quasar::proto::AppState::MediaInfo YandexIO::convertScreenState(const NAlice::TDeviceState::TScreen& state) {
    quasar::proto::AppState::MediaInfo converted;
    for (const auto& item : state.GetSupportedScreenResolutions()) {
        static const auto* desc = NAlice::TDeviceState::TScreen::EScreenResolution_descriptor();
        if (const auto* val = desc->FindValueByNumber(item)) {
            auto name = val->name();
            converted.mutable_supported_screen_resolutions()->Add(std::move(name));
        }
    }
    for (const auto& item : state.GetDynamicRanges()) {
        static const auto* desc = NAlice::TDeviceState::TScreen::EDynamicRange_descriptor();
        if (const auto* val = desc->FindValueByNumber(item)) {
            auto name = val->name();
            converted.mutable_dynamic_ranges()->Add(std::move(name));
        }
    }
    if (state.HasHdcpLevel()) {
        static const auto* desc = NAlice::TDeviceState::TScreen::EHdcpLevel_descriptor();
        if (const auto* val = desc->FindValueByNumber(state.GetHdcpLevel())) {
            converted.set_hdcp_level(val->name());
        }
    }
    return converted;
}

quasar::proto::PackagesState YandexIO::convertPackagesState(const NAlice::TDeviceState::TPackagesState& state) {
    quasar::proto::PackagesState converted;
    for (const auto& package : state.GetInstalled()) {
        auto* convertedPackage = converted.mutable_installed()->Add();
        convertedPackage->set_main_activity(package.GetMainActivity());
        if (package.HasPackageInfo()) {
            convertedPackage->mutable_package_info()->set_name(package.GetPackageInfo().GetName());
            convertedPackage->mutable_package_info()->set_human_readable_name(package.GetPackageInfo().GetHumanReadableName());
        }
    }
    return converted;
}

yandex_io::proto::TVideo YandexIO::convertVideoState(const NAlice::TDeviceState::TVideo& state) {
    const auto videoJson = convertMessageToJsonString(state);
    const auto converted = convertJsonToProtobuf<yandex_io::proto::TVideo>(videoJson, true);
    if (!converted.has_value()) {
        YIO_LOG_ERROR_EVENT("DeviceStateConverter.FailedConvertVideoStateToSdkProto", "Failed to convert video state: " << videoJson);
        return yandex_io::proto::TVideo{};
    }
    return *converted;
}

quasar::proto::AppState::RadioState YandexIO::convertRadioState(const google::protobuf::Struct& state) {
    quasar::proto::AppState::RadioState converted;
    if (const auto currentlyPlaying = getStructFromStruct(state, "currently_playing")) {
        if (const auto radioId = getStringFromStruct(*currentlyPlaying, "radioId")) {
            converted.set_radio_id(*radioId);
        }
        if (const auto radioTitle = getStringFromStruct(*currentlyPlaying, "radioTitle")) {
            converted.set_radio_title(*radioTitle);
        }
    }
    if (const auto player = getStructFromStruct(state, "player")) {
        converted.set_is_paused(getBoolFromStruct(*player, "pause").value_or(true));
        if (const auto timestamp = getNumberFromStruct(*player, "timestamp")) {
            converted.set_timestamp_ms(*timestamp * 1000);
        }
    }
    if (const auto uid = getStringFromStruct(state, "playlist_owner")) {
        converted.set_uid(*uid);
    }
    if (const auto color = getStringFromStruct(state, "color")) {
        converted.set_color(*color);
    }
    if (const auto imageUrl = getStringFromStruct(state, "image_url")) {
        converted.set_cover_uri(*imageUrl);
    }
    return converted;
}

quasar::proto::AudioClientEvent YandexIO::convertAudioPlayerState(const NAlice::TDeviceState::TAudioPlayer& state) {
    quasar::proto::AudioClientEvent converted;
    converted.set_state(convertAudioClientState(state.GetPlayerState()));
    converted.set_event(proto::AudioClientEvent::STATE_CHANGED);
    converted.set_last_play_timestamp(state.GetLastPlayTimestamp());
    converted.set_last_stop_timestamp(state.GetLastStopTimestamp());
    auto* audio = converted.mutable_audio();
    audio->set_position_sec(state.GetOffsetMs() / 1000);
    audio->set_duration_sec(state.GetDurationMs() / 1000);
    audio->set_played_sec(state.GetPlayedMs() / 1000);
    audio->set_id(state.GetCurrentlyPlaying().GetStreamId());
    audio->set_type(state.GetCurrentlyPlaying().GetStreamType());
    audio->mutable_metadata()->set_title(state.GetCurrentlyPlaying().GetTitle());
    audio->mutable_metadata()->set_subtitle(state.GetCurrentlyPlaying().GetSubTitle());

    Json::Value context = Json::objectValue;
    auto& scenarioMeta = context["scenario_meta"];
    scenarioMeta = Json::objectValue;
    for (const auto& [key, val] : state.GetScenarioMeta()) {
        if (key == "__vins_request_id") {
            audio->mutable_analytics_context()->set_vins_request_id(val);
        } else if (key == "__art_image_url") {
            audio->mutable_metadata()->set_art_image_url(val);
        } else if (key == "__is_multiroom_slave") {
            converted.mutable_player_descriptor()->set_is_multiroom_slave(val == "1");
        } else if (key == "__multiroom_token") {
            audio->set_multiroom_token(val);
        } else if (key == "__play_pause_id") {
            audio->set_play_pause_id(val);
        } else if (key == "__basetime_ns") {
            readValue(val, [&](uint64_t v) { audio->set_basetime_ns(v); });
        } else if (key == "__position_ns") {
            readValue(val, [&](uint64_t v) { audio->set_position_ns(v); });
        } else if (key == "__url") {
            audio->set_url(val);
        } else if (key == "__clock_id") {
            audio->set_clock_id(val);
        } else if (key == "__normalization_tp") {
            readValue(val, [&](double v) { audio->mutable_normalization()->set_true_peak(v); });
        } else if (key == "__normalization_il") {
            readValue(val, [&](double v) { audio->mutable_normalization()->set_integrated_loudness(v); });
        } else if (key == "__normalization_tl") {
            readValue(val, [&](double v) { audio->mutable_normalization()->set_target_lufs(v); });
        } else {
            context[key] = val;
        }
    }
    audio->set_context(jsonToString(context));

    // DSC doesnt know more descriptor info, so components must listen audio player directily to get it.
    converted.mutable_player_descriptor()->set_type(quasar::proto::AudioPlayerDescriptor::AUDIO);
    return converted;
}

quasar::proto::AppState::ScreenState YandexIO::convertLegacyScreenState(const NAlice::TDeviceState::TVideo& state) {
    proto::AppState::ScreenState converted;
    if (state.HasCurrentScreen()) {
        static const auto* screenTypeDescriptor = proto::AppState::ScreenType_descriptor();
        std::string screenName = state.GetCurrentScreen();
        std::transform(screenName.begin(), screenName.end(), screenName.begin(), ::toupper);
        const auto* val = screenTypeDescriptor->FindValueByName(TString(screenName));
        if (val) {
            converted.set_screen_type(quasar::proto::AppState::ScreenType{val->number()});
        } else {
            YIO_LOG_ERROR_EVENT("DeviceStateConverter.FailedConvertLegacyScreenState", "Failed screen type: " << state.GetCurrentScreen());
        }
    }

    if (state.HasScreenState()) {
        const TString rawResult = convertMessageToJsonString(state.GetScreenState());
        converted.mutable_search_result()->set_raw_result(rawResult);
        converted.mutable_description_item()->set_raw(rawResult);
        auto* webViewState = converted.mutable_web_view_state();
        if (state.GetScreenState().HasScenarioName()) {
            webViewState->set_backend_scenario_name(state.GetScreenState().GetScenarioName());
        }
        if (state.GetScreenState().HasScenario()) {
            webViewState->set_scenario(state.GetScreenState().GetScenario());
        }
    }

    if (state.HasViewState()) {
        auto rawResult = convertMessageToJsonString(state.GetViewState());
        converted.set_web_page_state(std::move(rawResult));
    }

    if (converted.screen_type() == proto::AppState::ScreenType::AppState_ScreenType_GALLERY ||
        converted.screen_type() == proto::AppState::ScreenType::AppState_ScreenType_SEASON_GALLERY ||
        converted.screen_type() == proto::AppState::ScreenType::AppState_ScreenType_TV_GALLERY) {
        for (const auto& val : state.GetScreenState().GetVisibleItems()) {
            converted.mutable_visible_items()->Add(val);
        }
    }

    return converted;
}

quasar::proto::VideoState YandexIO::convertLegacyVideoState(const NAlice::TDeviceState::TVideo& state) {
    quasar::proto::VideoState converted;

    if (state.GetPlayer().has_pause()) {
        converted.set_is_paused(state.GetPlayer().GetPause());
    }
    if (!state.HasCurrentlyPlaying()) {
        return converted;
    }
    if (state.GetCurrentlyPlaying().HasProgress()) {
        converted.mutable_progress()->set_duration(state.GetCurrentlyPlaying().GetProgress().GetDuration());
        converted.mutable_progress()->set_played(state.GetCurrentlyPlaying().GetProgress().GetPlayed());
    }
    if (state.GetCurrentlyPlaying().HasAudioLanguage()) {
        converted.mutable_video()->set_audio_language(state.GetCurrentlyPlaying().GetAudioLanguage());
    }
    if (state.GetCurrentlyPlaying().HasSubtitlesLanguage()) {
        converted.mutable_video()->set_subtitles_language(state.GetCurrentlyPlaying().GetSubtitlesLanguage());
    }
    converted.mutable_video()->set_raw(convertMessageToJsonString(state.GetCurrentlyPlaying()));
    if (state.GetCurrentlyPlaying().HasRawItem()) {
        *converted.mutable_video()->mutable_item() = convertMediaItem(state.GetCurrentlyPlaying().GetRawItem());
    }
    if (state.GetCurrentlyPlaying().HasRawTvShowItem()) {
        *converted.mutable_video()->mutable_tv_show_item() = convertMediaItem(state.GetCurrentlyPlaying().GetRawTvShowItem());
    }
    converted.set_timestamp_ms(state.GetCurrentlyPlaying().GetLastPlayTimestamp());
    return converted;
}

quasar::proto::AppState::MusicState YandexIO::convertMusicState(const NAlice::TDeviceState::TMusic& state) {
    proto::AppState::MusicState converted;

    if (state.GetPlayer().HasPause()) {
        converted.set_is_paused(state.GetPlayer().GetPause());
    }
    if (state.GetPlayer().HasTimestamp()) {
        converted.set_timestamp_ms(state.GetPlayer().GetTimestamp() * 1000);
    }
    if (state.GetCurrentlyPlaying().HasTrackId()) {
        converted.set_current_track_id(state.GetCurrentlyPlaying().GetTrackId());
    }
    if (state.HasPlaylistOwner()) {
        converted.set_uid(state.GetPlaylistOwner());
    }
    if (state.HasSessionId()) {
        converted.set_session_id(state.GetSessionId());
    }
    if (state.GetCurrentlyPlaying().HasRawTrackInfo()) {
        converted.set_json_track_info(convertMessageToJsonString(state.GetCurrentlyPlaying().GetRawTrackInfo()));
    }
    if (auto title = getStringFromStruct(state.GetCurrentlyPlaying().GetRawTrackInfo(), "title")) {
        converted.set_title(std::move(*title));
    }
    if (auto coverUri = getStringFromStruct(state.GetCurrentlyPlaying().GetRawTrackInfo(), "coverUri")) {
        converted.set_cover_uri(std::move(*coverUri));
    }
    if (auto url = getStringFromStruct(state.GetCurrentlyPlaying().GetRawTrackInfo(), "url")) {
        converted.set_url(std::move(*url));
    }
    if (auto value = getStringFromStruct(state.GetCurrentlyPlaying().GetRawTrackInfo(), "vins_request_id")) {
        converted.set_vins_request_id(std::move(*value));
    }
    const auto artists = getStructListFromStruct(state.GetCurrentlyPlaying().GetRawTrackInfo(), "artists");
    TString convertedArtists = convertArtists(artists);
    if (!convertedArtists.Empty()) {
        converted.set_artists(std::move(convertedArtists));
    }
    if (const auto durationMs = getNumberFromStruct(state.GetCurrentlyPlaying().GetRawTrackInfo(), "durationMs")) {
        converted.set_duration_ms(*durationMs);
    }
    if (const auto progress = getNumberFromStruct(state.GetCurrentlyPlaying().GetRawTrackInfo(), "progress_sec")) {
        converted.set_progress(*progress);
    }
    if (const auto value = getNumberFromStruct(state.GetCurrentlyPlaying().GetRawTrackInfo(), "basetime_ns")) {
        converted.set_basetime_ns(*value);
    }
    if (const auto value = getNumberFromStruct(state.GetCurrentlyPlaying().GetRawTrackInfo(), "position_ns")) {
        converted.set_position_ns(*value);
    }

    return converted;
}

quasar::proto::AppState YandexIO::convertAppState(const NAlice::TDeviceState& state) {
    quasar::proto::AppState converted;
    if (state.HasBluetooth()) {
        *converted.mutable_bluetooth_player_state() = convertBluetoothState(state.GetBluetooth());
    }
    if (state.HasRadio()) {
        *converted.mutable_radio_state() = convertRadioState(state.GetRadio());
    }
    if (state.HasVideo()) {
        *converted.mutable_video_state() = convertLegacyVideoState(state.GetVideo());
        *converted.mutable_video() = convertVideoState(state.GetVideo());
        *converted.mutable_screen_state() = convertLegacyScreenState(state.GetVideo());
    }
    if (state.HasMusic()) {
        *converted.mutable_music_state() = convertMusicState(state.GetMusic());
    }
    if (state.HasAudioPlayer()) {
        *converted.mutable_audio_player_state() = convertAudioPlayerState(state.GetAudioPlayer());
    }
    if (state.HasRcuState()) {
        *converted.mutable_rcu_state() = convertRcuState(state.GetRcuState());
    }
    if (state.HasScreen()) {
        *converted.mutable_media_info() = convertScreenState(state.GetScreen());
    }
    if (state.HasPackagesState()) {
        *converted.mutable_packages_state() = convertPackagesState(state.GetPackagesState());
    }
    return converted;
}

quasar::proto::SimplePlayerState YandexIO::convertMusicToSimplePlayerState(
    const NAlice::TDeviceState::TMusic& musicState, bool showSimplePlayer) {
    quasar::proto::SimplePlayerState simplePlayerState;

    simplePlayerState.set_has_like(true);
    simplePlayerState.set_has_dislike(true);
    simplePlayerState.set_has_next(true);
    simplePlayerState.set_has_prev(true);
    simplePlayerState.set_has_progress_bar(true);
    simplePlayerState.set_live_stream_text("");
    simplePlayerState.set_player_type("music_thick");
    simplePlayerState.set_show_player(showSimplePlayer);

    simplePlayerState.set_has_pause(!musicState.GetPlayer().GetPause());
    simplePlayerState.set_id(musicState.GetCurrentlyPlaying().GetTrackId());

    const auto& rawTrackInfo = musicState.GetCurrentlyPlaying().GetRawTrackInfo();
    if (auto value = getStringFromStruct(rawTrackInfo, "type")) {
        simplePlayerState.set_type(*value);
    }
    if (auto value = getStringFromStruct(rawTrackInfo, "title")) {
        simplePlayerState.set_title(*value);
    }
    if (auto value = getStringFromStruct(rawTrackInfo, "artists")) {
        simplePlayerState.set_subtitle(*value);
    }
    if (const auto value = getNumberFromStruct(rawTrackInfo, "durationMs")) {
        simplePlayerState.set_duration(*value / 1000);
    }
    if (const auto value = getNumberFromStruct(rawTrackInfo, "progress_sec")) {
        simplePlayerState.set_position(*value);
    }

    auto extra = simplePlayerState.mutable_extra();
    extra->insert({"stateType", "music"});
    if (auto value = getStringFromStruct(rawTrackInfo, "coverUri")) {
        extra->insert({"coverURI", TString(*value)});
    }
    if (auto value = getStringFromStruct(rawTrackInfo, "vins_request_id")) {
        extra->insert({"requestID", TString(*value)});
    }

    auto entity = simplePlayerState.mutable_entity_info();
    if (auto value = getStringFromStruct(rawTrackInfo, "playlist_id")) {
        entity->set_id(*value);
    }
    if (auto value = getStringFromStruct(rawTrackInfo, "playlist_type")) {
        entity->set_type(*value);
    }
    if (auto value = getStringFromStruct(rawTrackInfo, "playlist_description")) {
        entity->set_description(*value);
    }
    if (auto value = getBoolFromStruct(rawTrackInfo, "playlist_shuffled")) {
        entity->set_shuffled(*value);
    }
    if (auto value = getStringFromStruct(rawTrackInfo, "playlist_repeat_mode")) {
        entity->set_repeat_mode(*value);
    }

    return simplePlayerState;
}

quasar::proto::SimplePlayerState YandexIO::convertRadioToSimplePlayerState(
    const google::protobuf::Struct& radio, bool showSimplePlayer) {
    quasar::proto::SimplePlayerState simplePlayerState;

    simplePlayerState.set_duration(0);
    simplePlayerState.set_position(0);
    simplePlayerState.set_type("stream");
    simplePlayerState.set_player_type("radio");
    simplePlayerState.set_live_stream_text("Прямой эфир");
    simplePlayerState.set_has_like(false);
    simplePlayerState.set_has_dislike(false);
    simplePlayerState.set_has_next(false);
    simplePlayerState.set_has_prev(false);
    simplePlayerState.set_has_progress_bar(false);

    simplePlayerState.set_id(getRadioId(radio));
    simplePlayerState.set_title(getRadioTitle(radio));
    simplePlayerState.set_show_player(showSimplePlayer);
    simplePlayerState.set_has_pause(!getRadioPause(radio));

    auto extra = simplePlayerState.mutable_extra();
    extra->insert({"stateType", "radio"});
    if (auto value = getStringFromStruct(radio, "image_url")) {
        extra->insert({"coverURI", TString(*value)});
    }
    if (auto value = getStringFromStruct(radio, "vins_request_id")) {
        extra->insert({"requestID", TString(*value)});
    }

    auto entity = simplePlayerState.mutable_entity_info();
    entity->set_id(simplePlayerState.id());
    entity->set_type("FM_RADIO");

    return simplePlayerState;
}

quasar::proto::SimplePlayerState YandexIO::convertAudioPlayerToSimplePlayerState(
    const NAlice::TDeviceState::TAudioPlayer& audioPlayer, bool showSimplePlayer) {
    quasar::proto::SimplePlayerState simplePlayerState;

    simplePlayerState.set_has_like(true);
    simplePlayerState.set_has_dislike(true);
    simplePlayerState.set_has_next(true);
    simplePlayerState.set_has_prev(true);
    simplePlayerState.set_has_progress_bar(true);
    simplePlayerState.set_live_stream_text("");
    simplePlayerState.set_player_type("music_thin");

    simplePlayerState.set_id(audioPlayer.GetCurrentlyPlaying().GetStreamId());
    simplePlayerState.set_type(audioPlayer.GetCurrentlyPlaying().GetStreamType());
    simplePlayerState.set_title(audioPlayer.GetCurrentlyPlaying().GetTitle());
    simplePlayerState.set_subtitle(audioPlayer.GetCurrentlyPlaying().GetSubTitle());
    simplePlayerState.set_position(audioPlayer.GetOffsetMs() / 1000);
    simplePlayerState.set_duration(audioPlayer.GetDurationMs() / 1000);
    simplePlayerState.set_show_player(showSimplePlayer);
    simplePlayerState.set_has_pause(audioPlayer.GetPlayerState() == NAlice::TDeviceState::TAudioPlayer::Playing);

    auto extra = simplePlayerState.mutable_extra();
    auto entity = simplePlayerState.mutable_entity_info();
    extra->insert({"stateType", "music"});

    for (const auto& [key, val] : audioPlayer.GetScenarioMeta()) {
        if (key == "__art_image_url") {
            extra->insert({"coverURI", TString(val)});
        } else if (key == "__vins_request_id") {
            extra->insert({"requestID", TString(val)});
        } else if (key == "__musicmeta_id") {
            entity->set_id(TString(val));
        } else if (key == "__musicmeta_type") {
            entity->set_type(TString(val));
        } else if (key == "__musicmeta_description") {
            entity->set_description(TString(val));
        } else if (key == "__musicmeta_shuffled") {
            entity->set_shuffled(val == "1");
        } else if (key == "__musicmeta_repeat_mode") {
            entity->set_repeat_mode(TString(val));
        } else if (key == "__musicmeta_prev_id") {
            entity->mutable_prev()->set_id(TString(val));
        } else if (key == "__musicmeta_prev_type") {
            entity->mutable_prev()->set_type(TString(val));
        } else if (key == "__musicmeta_next_id") {
            entity->mutable_next()->set_id(TString(val));
        } else if (key == "__musicmeta_next_type") {
            entity->mutable_next()->set_type(TString(val));
        } else if (key == "__provider_name") {
            simplePlayerState.set_provider_name(TString(val));
        } else if (key == "__hide_progress_bar") {
            simplePlayerState.set_hide_progress_bar(val == "1");
        }
    }

    return simplePlayerState;
}
