#include "device_state_capability.h"

#include "converters/converters.h"
#include <yandex_io/libs/audio_player/base/audio_player.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/protobuf_utils/json.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/protos/capabilities/device_state_capability.pb.h>

#include <future>

using namespace quasar;
using namespace YandexIO;

YIO_DEFINE_LOG_MODULE("device_state_capability");

DeviceStateCapability::DeviceStateCapability(std::shared_ptr<quasar::ICallbackQueue> worker)
    : worker_(std::move(worker))
{
    state_.MutableDeviceStateCapability();
}

quasar::proto::AppState DeviceStateCapability::getAppState()
{
    Y_ENSURE_THREAD(worker_);
    return YandexIO::convertAppState(state_.MutableDeviceStateCapability()->MutableState()->GetDeviceState());
}

void DeviceStateCapability::setAppStateChanged(AppStateChanged appStateChanged)
{
    Y_ENSURE_THREAD(worker_);
    appStateChanged_ = std::move(appStateChanged);
}

void DeviceStateCapability::onDeviceStatePart(const yandex_io::proto::TDeviceStatePart& statePart)
{
    Y_ENSURE_THREAD(worker_);

    if (statePart.has_video()) {
        setVideoState(YandexIO::convertVideoState(statePart.video()));
    }

    if (statePart.has_rcu()) {
        setRcuState(statePart.rcu());
    }

    if (statePart.has_packages()) {
        setPackagesState(statePart.packages());
    }
}

NAlice::TCapabilityHolder DeviceStateCapability::getState() const {
    if (worker_->isWorkingThread()) {
        return state_;
    }
    std::promise<NAlice::TCapabilityHolder> p;
    worker_->add([this, &p]() {
        p.set_value(state_);
    });
    return p.get_future().get();
}

std::shared_ptr<IDirectiveHandler> DeviceStateCapability::getDirectiveHandler() {
    return nullptr;
}

void DeviceStateCapability::addListener(std::weak_ptr<IListener> wlistener) {
    executeInWorker([this, wlistener = std::move(wlistener)]() mutable {
        if (const auto listener = wlistener.lock()) {
            listener->onCapabilityStateChanged(shared_from_this(), getState());
        }
        listeners_.push_back(std::move(wlistener));
    });
}

void DeviceStateCapability::removeListener(std::weak_ptr<IListener> wlistener) {
    executeInWorker([this, wlistener]() {
        listeners_.remove_if([wlistener](const auto& wp) {
            return !(wp.owner_before(wlistener) || wlistener.owner_before(wp));
        });
    });
}

void DeviceStateCapability::notifyStateChanged() {
    Y_ENSURE_THREAD(worker_);
    for (const auto& wlistener : listeners_) {
        if (const auto listener = wlistener.lock()) {
            listener->onCapabilityStateChanged(shared_from_this(), getState());
        }
    }
}

void DeviceStateCapability::onAppStateChanged()
{
    Y_ENSURE_THREAD(worker_);

    if (appStateChanged_) {
        appStateChanged_(getAppState());
    }
}

void DeviceStateCapability::setBluetoothState(const NAlice::TDeviceState::TBluetooth& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableBluetooth()->Swap(&state);
        notifyStateChanged();
        onAppStateChanged();
    });
}

void DeviceStateCapability::setAudioPlayerState(const NAlice::TDeviceState::TAudioPlayer& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableAudioPlayer()->Swap(&state);
        notifyStateChanged();
        onAppStateChanged();
    });
}

void DeviceStateCapability::setMusicState(const NAlice::TDeviceState::TMusic& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableMusic()->Swap(&state);
        notifyStateChanged();
        onAppStateChanged();
    });
}

void DeviceStateCapability::setRadioState(const google::protobuf::Struct& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableRadio()->Swap(&state);
        notifyStateChanged();
        onAppStateChanged();
    });
}

void DeviceStateCapability::setLastWatched(const NAlice::TDeviceState::TLastWatched& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableLastWatched()->Swap(&state);
        notifyStateChanged();
    });
}

void DeviceStateCapability::addSupportedFeature(const TString& feature) {
    executeInWorker([this, feature]() {
        addedSupportedFeatures_.insert(feature);
        state_.MutableDeviceStateCapability()->MutableState()->AddSupportedFeatures(feature);
        notifyStateChanged();
    });
}

void DeviceStateCapability::setSupportedFeatures(const std::unordered_set<TString>& features) {
    executeInWorker([this, features = features]() {
        state_.MutableDeviceStateCapability()->MutableState()->MutableSupportedFeatures()->Assign(features.begin(), features.end());
        for (const auto& feature : addedSupportedFeatures_) {
            state_.MutableDeviceStateCapability()->MutableState()->AddSupportedFeatures(feature);
        }

        notifyStateChanged();
    });
}

void DeviceStateCapability::setUnsupportedFeatures(const std::unordered_set<TString>& features) {
    executeInWorker([this, features = features]() {
        state_.MutableDeviceStateCapability()->MutableState()->MutableUnsupportedFeatures()->Assign(features.begin(), features.end());
        notifyStateChanged();
    });
}

void DeviceStateCapability::setExperiments(const NAlice::TExperimentsProto& experiments) {
    executeInWorker([this, experiments = experiments]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableExperiments()->Swap(&experiments);
        notifyStateChanged();
    });
}

void DeviceStateCapability::setEnrollmentHeaders(const NAlice::TEnrollmentHeaders& enrollmentHeaders) {
    executeInWorker([this, enrollmentHeaders = enrollmentHeaders]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableEnrollmentHeaders()->Swap(&enrollmentHeaders);
        notifyStateChanged();
    });
}

void DeviceStateCapability::setVideoState(const NAlice::TDeviceState::TVideo& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableVideo()->Swap(&state);
        notifyStateChanged();
        onAppStateChanged();
    });
}

void DeviceStateCapability::setRcuState(const NAlice::TDeviceState::TRcuState& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableRcuState()->Swap(&state);
        notifyStateChanged();
        onAppStateChanged();
    });
}

void DeviceStateCapability::setScreenState(const NAlice::TDeviceState::TScreen& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableScreen()->Swap(&state);
        notifyStateChanged();
        onAppStateChanged();
    });
}

void DeviceStateCapability::setMultiroomState(const NAlice::TDeviceState::TMultiroom& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableMultiroom()->Swap(&state);
        notifyStateChanged();
    });
}

void DeviceStateCapability::setPackagesState(const NAlice::TDeviceState::TPackagesState& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutablePackagesState()->Swap(&state);
        notifyStateChanged();
        onAppStateChanged();
    });
}

void DeviceStateCapability::setSupportedFeatures(const google::protobuf::RepeatedPtrField<TString>& features) {
    Y_ENSURE_THREAD(worker_);
    state_.MutableDeviceStateCapability()->MutableState()->MutableSupportedFeatures()->CopyFrom(features);
    for (const auto& feature : addedSupportedFeatures_) {
        state_.MutableDeviceStateCapability()->MutableState()->AddSupportedFeatures(feature);
    }
    notifyStateChanged();
}

void DeviceStateCapability::setUnsupportedFeatures(const google::protobuf::RepeatedPtrField<TString>& features) {
    Y_ENSURE_THREAD(worker_);
    state_.MutableDeviceStateCapability()->MutableState()->MutableUnsupportedFeatures()->CopyFrom(features);
    notifyStateChanged();
}

void DeviceStateCapability::setEnvironmentDeviceInfo(const NAlice::TEnvironmentDeviceInfo& deviceInfo) {
    executeInWorker([this, deviceInfo = deviceInfo]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableEnvironmentDeviceInfo()->Swap(&deviceInfo);
        notifyStateChanged();
    });
}

void DeviceStateCapability::setIsTvPluggedIn(bool isTvPluggedIn) {
    executeInWorker([this, isTvPluggedIn]() {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->SetIsTvPluggedIn(isTvPluggedIn);
        notifyStateChanged();
    });
}

void DeviceStateCapability::setICalendar(const TString& ical) {
    executeInWorker([this, ical = ical]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->SetAlarmsState(std::move(ical));
        notifyStateChanged();
    });
}

void DeviceStateCapability::setAlarmState(const NAlice::TDeviceState::TAlarmState& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableAlarmState()->Swap(&state);
        notifyStateChanged();
    });
}

void DeviceStateCapability::setTimersState(const NAlice::TDeviceState::TTimers& state) {
    executeInWorker([this, state = state]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableTimers()->Swap(&state);
        notifyStateChanged();
    });
}

void DeviceStateCapability::setMicsMuted(bool muted) {
    executeInWorker([this, muted = muted]() {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->SetMicsMuted(muted);
        notifyStateChanged();
    });
}

void DeviceStateCapability::setBattery(const NAlice::TDeviceState::TBattery& battery) {
    executeInWorker([this, battery = battery]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableBattery()->Swap(&battery);
        notifyStateChanged();
    });
}

void DeviceStateCapability::setNetworkState(const NAlice::TDeviceStateCapability::TState::TNetworkState& state) {
    executeInWorker([this, state = state]() mutable {
        auto* internetConnection = state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->MutableInternetConnection();
        if (state.HasCurrentWifi()) {
            const auto& current = state.GetCurrentWifi();
            const auto& freq = current.GetFrequency();
            if (freq > 2400 && freq < 2500) {
                return internetConnection->SetType(NAlice::TDeviceState::TInternetConnection::Wifi_2_4GHz);
            } else if (freq > 5000 && freq < 6000) {
                return internetConnection->SetType(NAlice::TDeviceState::TInternetConnection::Wifi_5GHz);
            } else {
                YIO_LOG_ERROR_EVENT("DeviceStateCapabilty.DeviceStateUnknownWifiFrequency", "Unknown wifi frequency " << freq);
                ;
            }
            internetConnection->MutableCurrent()->SetSsid(current.GetSsid());
            internetConnection->MutableCurrent()->SetBssid(current.GetBssid());
            internetConnection->MutableCurrent()->SetChannel(current.GetChannel());
        } else {
            internetConnection->SetType(NAlice::TDeviceState::TInternetConnection::Ethernet);
        }
        for (const auto& wifi : state.GetWifiList()) {
            NAlice::TDeviceState::TInternetConnection::TWifiNetwork wifiInfo;
            wifiInfo.SetSsid(wifi.GetSsid());
            wifiInfo.SetBssid(wifi.GetBssid());
            wifiInfo.SetChannel(wifi.GetChannel());
            internetConnection->MutableNeighbours()->Add(std::move(wifiInfo));
        }
        state_.MutableDeviceStateCapability()->MutableState()->MutableNetworkState()->Swap(&state);
    });
}

void DeviceStateCapability::setVolumeState(bool soundMuted, int soundLevel, std::optional<int> soundMaxLevel) {
    executeInWorker([this, soundMuted, soundLevel, soundMaxLevel]() mutable {
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->SetSoundMuted(soundMuted);
        state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->SetSoundLevel(soundLevel);
        if (soundMaxLevel.has_value()) {
            state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->SetSoundMaxLevel(*soundMaxLevel);
        } else {
            state_.MutableDeviceStateCapability()->MutableState()->MutableDeviceState()->ClearSoundMaxLevel();
        }
        notifyStateChanged();
    });
}

void DeviceStateCapability::executeInWorker(const std::function<void()>& func) {
    if (worker_->isWorkingThread()) {
        func();
    } else {
        worker_->add(func);
    }
}
