#include "device_state_capability_host.h"

#include <yandex_io/capabilities/device_state/util.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("device_state_capability");

using namespace YandexIO;

DeviceStateCapabilityHost::DeviceStateCapabilityHost(std::shared_ptr<quasar::ICallbackQueue> worker, std::weak_ptr<IRemotingRegistry> remotingRegistry)
    : DeviceStateCapability(std::move(worker))
    , IRemoteObject(std::move(remotingRegistry))
{
}

void DeviceStateCapabilityHost::init() {
    if (const auto registry = getRemotingRegistry().lock()) {
        registry->addRemoteObject(DeviceStateCapabilityUtil::REMOTE_NAME, std::static_pointer_cast<DeviceStateCapabilityHost>(shared_from_this()));
    }
}

void DeviceStateCapabilityHost::handleRemotingMessage(const quasar::proto::Remoting& message, std::shared_ptr<IRemotingConnection> /*connection*/) {
    if (!message.has_device_state_capability_method() || !message.device_state_capability_method().has_method()) {
        YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Failed to find DeviceStateCapability method in remoting message");
        return;
    }
    executeInWorker([this, method = message.device_state_capability_method()]() {
        switch (method.method()) {
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_BLUETOOTH_STATE: {
                if (!method.has_bluetooth_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing bluetooth state arg in SET_BLUETOOTH_STATE method");
                    break;
                }
                setBluetoothState(method.bluetooth_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_AUDIO_PLAYER_STATE: {
                if (!method.has_audio_player_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing audio player state arg in SET_AUDIO_PLAYER_STATE method");
                    break;
                }
                setAudioPlayerState(method.audio_player_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_MUSIC_STATE: {
                if (!method.has_music_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing music state arg in SET_MUSIC_STATE method");
                    break;
                }
                setMusicState(method.music_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_RADIO_STATE: {
                if (!method.has_radio_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing radio state arg in SET_RADIO_STATE method");
                    break;
                }
                setRadioState(method.radio_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_LAST_WATCHED: {
                if (!method.has_last_watched()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing last watched arg in SET_LAST_WATCHED method");
                    break;
                }
                setLastWatched(method.last_watched());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::ADD_SUPPORTED_FEATURE: {
                for (const auto& feature : method.supported_features()) {
                    addSupportedFeature(feature);
                    break;
                }
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_SUPPORTED_FEATURES: {
                setSupportedFeatures(method.supported_features());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_UNSUPPORTED_FEATURES: {
                setUnsupportedFeatures(method.unsupported_features());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_EXPERIMENTS: {
                if (!method.has_experiments()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing experiments arg in SET_EXPERIMENTS method");
                    break;
                }
                setExperiments(method.experiments());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_ENROLLMENT_HEADERS: {
                if (!method.has_enrollment_headers()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing enrollment headers arg in SET_ENROLLMENT_HEADERS method");
                    break;
                }
                setEnrollmentHeaders(method.enrollment_headers());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_VIDEO_STATE: {
                if (!method.has_video_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing video state arg in SET_VIDEO_STATE method");
                    break;
                }
                setVideoState(method.video_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_RCU_STATE: {
                if (!method.has_rcu_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing rcu state arg in SET_RCU_STATE method");
                    break;
                }
                setRcuState(method.rcu_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_SCREEN_STATE: {
                if (!method.has_screen_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing screen state arg in SET_SCREEN_STATE method");
                    break;
                }
                setScreenState(method.screen_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_MULTIROOM_STATE: {
                if (!method.has_multiroom_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing multiroom state arg in SET_MULTIROOM_STATE method");
                    break;
                }
                setMultiroomState(method.multiroom_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_PACKAGES_STATE: {
                if (!method.has_packages_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing packages state arg in SET_PACKAGES_STATE method");
                    break;
                }
                setPackagesState(method.packages_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_ENVIRONMENT_DEVICE_INFO: {
                if (!method.has_environment_device_info()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing environment_device_info arg in SET_ENVIRONMENT_DEVICE_INFO method");
                    break;
                }
                setEnvironmentDeviceInfo(method.environment_device_info());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_IS_TV_PLUGGED_IN: {
                if (!method.has_is_tv_plugged_in()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing is_tv_plugged_in arg in SET_IS_TV_PLUGGED_IN method");
                    break;
                }
                setIsTvPluggedIn(method.is_tv_plugged_in());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_ALARMS_STATE: {
                if (!method.has_alarm_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing alarm_state arg in SET_ALARMS_STATE method");
                    break;
                }
                setAlarmState(method.alarm_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_ICALENDAR: {
                if (!method.has_ical()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing ical arg in SET_ICALENDAR method");
                    break;
                }
                setICalendar(method.ical());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_TIMERS_STATE: {
                if (!method.has_timers_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing timers_state arg in SET_TIMERS_STATE method");
                    break;
                }
                setTimersState(method.timers_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_NETWORK_STATE: {
                if (!method.has_network_state()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing network_state arg in SET_NETWORK_STATE method");
                    break;
                }
                setNetworkState(method.network_state());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_MICS_MUTED: {
                if (!method.has_mics_muted()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing mics_muted arg in SET_MICS_MUTED method");
                    break;
                }
                setMicsMuted(method.mics_muted());
                break;
            }
            case quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_VOLUME_STATE: {
                if (!method.has_sound_muted() || !method.has_sound_level()) {
                    YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Missing network_state arg in SET_VOLUME_STATE method");
                    break;
                }

                std::optional<int> soundMaxLevel;
                if (method.has_sound_max_level()) {
                    soundMaxLevel = method.sound_max_level();
                }
                setVolumeState(method.sound_muted(), method.sound_level(), soundMaxLevel);
                break;
            }
            default: {
                static const auto* descriptor = quasar::proto::Remoting::DeviceStateCapabilityMethod::Method_descriptor();
                YIO_LOG_ERROR_EVENT("DeviceStateCapability.Remoting", "Unknown method: " << descriptor->FindValueByNumber(method.method())->name());
            }
        }
    });
}
