#include "device_state_capability_proxy.h"

#include <yandex_io/capabilities/device_state/util.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/libs/logging/logging.h>

using namespace YandexIO;

DeviceStateCapabilityProxy::DeviceStateCapabilityProxy(std::weak_ptr<IRemotingRegistry> remotingRegistry,
                                                       std::shared_ptr<quasar::ICallbackQueue> worker)
    : IRemoteObject(std::move(remotingRegistry))
    , callbackQueue_(std::move(worker))
    , remoteObjectId_(DeviceStateCapabilityUtil::REMOTE_NAME)
{
}

DeviceStateCapabilityProxy::~DeviceStateCapabilityProxy()
{
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->removeRemoteObject(remoteObjectId_);
    }
}

void DeviceStateCapabilityProxy::init()
{
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->addRemoteObject(remoteObjectId_, weak_from_this());
    }
}

void DeviceStateCapabilityProxy::handleRemotingConnect(std::shared_ptr<IRemotingConnection> connection) {
    callbackQueue_->add([this, connection]() {
        YIO_LOG_INFO("Connected to " << remoteObjectId_);
        connection_ = connection;
        isConnected_ = true;
        sendState();
    });
}

void DeviceStateCapabilityProxy::handleRemotingMessage(const quasar::proto::Remoting& /*message*/,
                                                       std::shared_ptr<IRemotingConnection> /*connection*/) {
    // ¯\_(ツ)_/¯
}

void DeviceStateCapabilityProxy::sendState() {
    if (supportedFeaturesToSet_.has_value()) {
        setSupportedFeatures(*supportedFeaturesToSet_);
    }
    for (const auto& feature : supportedFeaturesToAdd_) {
        addSupportedFeature(feature);
    }
    if (unsupportedFeatures_.has_value()) {
        setUnsupportedFeatures(*unsupportedFeatures_);
    }

    if (deviceState_.HasExperiments()) {
        setExperiments(deviceState_.GetExperiments());
    }
    if (deviceState_.HasEnrollmentHeaders()) {
        setEnrollmentHeaders(deviceState_.GetEnrollmentHeaders());
    }
    if (deviceState_.HasEnvironmentDeviceInfo()) {
        setEnvironmentDeviceInfo(deviceState_.GetEnvironmentDeviceInfo());
    }

    auto& deviceState = deviceState_.GetDeviceState();
    if (deviceState.HasBluetooth()) {
        setBluetoothState(deviceState.GetBluetooth());
    }
    if (deviceState.HasAudioPlayer()) {
        setAudioPlayerState(deviceState.GetAudioPlayer());
    }
    if (deviceState.HasMusic()) {
        setMusicState(deviceState.GetMusic());
    }
    if (deviceState.HasRadio()) {
        setRadioState(deviceState.GetRadio());
    }
    if (deviceState.HasLastWatched()) {
        setLastWatched(deviceState.GetLastWatched());
    }
    if (deviceState.HasVideo()) {
        setVideoState(deviceState.GetVideo());
    }
    if (deviceState.HasRcuState()) {
        setRcuState(deviceState.GetRcuState());
    }
    if (deviceState.HasScreen()) {
        setScreenState(deviceState.GetScreen());
    }
    if (deviceState.HasMultiroom()) {
        setMultiroomState(deviceState.GetMultiroom());
    }
    if (deviceState.HasPackagesState()) {
        setPackagesState(deviceState.GetPackagesState());
    }
    if (deviceState.HasIsTvPluggedIn()) {
        setIsTvPluggedIn(deviceState.GetIsTvPluggedIn());
    }
    if (deviceState.HasAlarmsState()) {
        setICalendar(deviceState.GetAlarmsState());
    }
    if (deviceState.HasAlarmState()) {
        setAlarmState(deviceState.GetAlarmState());
    }
    if (deviceState.HasTimers()) {
        setTimersState(deviceState.GetTimers());
    }
    if (deviceState.HasSoundLevel()) {
        setVolumeState(deviceState.GetSoundMuted(), deviceState.GetSoundLevel(), deviceState.GetSoundMaxLevel());
    }
    if (deviceState_.GetDeviceState().HasMicsMuted()) {
        setMicsMuted(deviceState_.GetDeviceState().GetMicsMuted());
    }
    if (deviceState_.GetDeviceState().HasBattery()) {
        setBattery(deviceState_.GetDeviceState().GetBattery());
    }
}

NAlice::TCapabilityHolder DeviceStateCapabilityProxy::getState() const {
    Y_VERIFY(false && "Not implemented");
    return NAlice::TCapabilityHolder();
}

std::shared_ptr<IDirectiveHandler> DeviceStateCapabilityProxy::getDirectiveHandler()
{
    return nullptr;
}

void DeviceStateCapabilityProxy::addListener(std::weak_ptr<IListener> wlistener)
{
    Y_UNUSED(wlistener);
}

void DeviceStateCapabilityProxy::removeListener(std::weak_ptr<IListener> wlistener)
{
    Y_UNUSED(wlistener);
}

void DeviceStateCapabilityProxy::setBluetoothState(const NAlice::TDeviceState::TBluetooth& state) {
    callbackQueue_->add([this, state]() {
        deviceState_.MutableDeviceState()->MutableBluetooth()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_BLUETOOTH_STATE);
        method->mutable_bluetooth_state()->CopyFrom(state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setAudioPlayerState(const NAlice::TDeviceState::TAudioPlayer& state) {
    callbackQueue_->add([this, state]() {
        deviceState_.MutableDeviceState()->MutableAudioPlayer()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_AUDIO_PLAYER_STATE);
        method->mutable_audio_player_state()->CopyFrom(state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setMusicState(const NAlice::TDeviceState::TMusic& state) {
    callbackQueue_->add([this, state]() {
        deviceState_.MutableDeviceState()->MutableMusic()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_MUSIC_STATE);
        method->mutable_music_state()->CopyFrom(state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setRadioState(const google::protobuf::Struct& state) {
    callbackQueue_->add([this, state]() {
        deviceState_.MutableDeviceState()->MutableRadio()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_RADIO_STATE);
        method->mutable_radio_state()->CopyFrom(state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setLastWatched(const NAlice::TDeviceState::TLastWatched& state) {
    callbackQueue_->add([this, state]() {
        deviceState_.MutableDeviceState()->MutableLastWatched()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_LAST_WATCHED);
        method->mutable_last_watched()->CopyFrom(state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::addSupportedFeature(const TString& feature) {
    callbackQueue_->add([this, feature]() {
        supportedFeaturesToAdd_.insert(feature);
        if (!isConnected_) {
            return;
        }

        YIO_LOG_INFO("addSupportedFeature " << feature);

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::ADD_SUPPORTED_FEATURE);
        method->add_supported_features(feature);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setSupportedFeatures(const std::unordered_set<TString>& features) {
    callbackQueue_->add([this, features]() {
        supportedFeaturesToSet_ = features;
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_SUPPORTED_FEATURES);
        method->mutable_supported_features()->Assign(features.begin(), features.end());

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setUnsupportedFeatures(const std::unordered_set<TString>& features) {
    callbackQueue_->add([this, features]() {
        unsupportedFeatures_ = features;
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_UNSUPPORTED_FEATURES);
        method->mutable_unsupported_features()->Assign(features.begin(), features.end());

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setExperiments(const NAlice::TExperimentsProto& experiments) {
    callbackQueue_->add([this, experiments]() {
        *deviceState_.MutableExperiments() = experiments;
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_EXPERIMENTS);
        method->mutable_experiments()->CopyFrom(experiments);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setEnrollmentHeaders(const NAlice::TEnrollmentHeaders& enrollmentHeaders) {
    callbackQueue_->add([this, enrollmentHeaders]() {
        *deviceState_.MutableEnrollmentHeaders() = enrollmentHeaders;
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_ENROLLMENT_HEADERS);
        method->mutable_enrollment_headers()->CopyFrom(enrollmentHeaders);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setVideoState(const NAlice::TDeviceState::TVideo& state) {
    callbackQueue_->add([this, state]() {
        deviceState_.MutableDeviceState()->MutableVideo()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_VIDEO_STATE);
        method->mutable_video_state()->CopyFrom(state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setRcuState(const NAlice::TDeviceState::TRcuState& state) {
    callbackQueue_->add([this, state]() {
        deviceState_.MutableDeviceState()->MutableRcuState()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_RCU_STATE);
        method->mutable_rcu_state()->CopyFrom(state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setScreenState(const NAlice::TDeviceState::TScreen& state) {
    callbackQueue_->add([this, state]() {
        deviceState_.MutableDeviceState()->MutableScreen()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_SCREEN_STATE);
        method->mutable_screen_state()->CopyFrom(state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setMultiroomState(const NAlice::TDeviceState::TMultiroom& state) {
    callbackQueue_->add([this, state]() {
        deviceState_.MutableDeviceState()->MutableMultiroom()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_MULTIROOM_STATE);
        method->mutable_multiroom_state()->CopyFrom(state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setPackagesState(const NAlice::TDeviceState::TPackagesState& state) {
    callbackQueue_->add([this, state]() {
        deviceState_.MutableDeviceState()->MutablePackagesState()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_PACKAGES_STATE);
        method->mutable_packages_state()->CopyFrom(state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setEnvironmentDeviceInfo(const NAlice::TEnvironmentDeviceInfo& deviceInfo) {
    callbackQueue_->add([this, deviceInfo = deviceInfo]() mutable {
        *deviceState_.MutableEnvironmentDeviceInfo() = deviceInfo;
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_ENVIRONMENT_DEVICE_INFO);
        method->mutable_environment_device_info()->Swap(&deviceInfo);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setIsTvPluggedIn(bool isTvPluggedIn) {
    callbackQueue_->add([this, isTvPluggedIn]() {
        deviceState_.MutableDeviceState()->SetIsTvPluggedIn(isTvPluggedIn);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_IS_TV_PLUGGED_IN);
        method->set_is_tv_plugged_in(isTvPluggedIn);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setICalendar(const TString& ical) {
    callbackQueue_->add([this, ical = ical]() mutable {
        deviceState_.MutableDeviceState()->SetAlarmsState(ical);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_ICALENDAR);
        method->set_ical(std::move(ical));

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setAlarmState(const NAlice::TDeviceState::TAlarmState& state) {
    callbackQueue_->add([this, state = state]() mutable {
        deviceState_.MutableDeviceState()->MutableAlarmState()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_ALARMS_STATE);
        method->mutable_alarm_state()->Swap(&state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setTimersState(const NAlice::TDeviceState::TTimers& state) {
    callbackQueue_->add([this, state = state]() mutable {
        deviceState_.MutableDeviceState()->MutableTimers()->CopyFrom(state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_TIMERS_STATE);
        method->mutable_timers_state()->Swap(&state);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setNetworkState(const NAlice::TDeviceStateCapability::TState::TNetworkState& state) {
    callbackQueue_->add([this, state = state]() mutable {
        deviceState_.MutableNetworkState()->Swap(&state);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_NETWORK_STATE);
        method->mutable_network_state()->CopyFrom(deviceState_.GetNetworkState());

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setMicsMuted(bool muted) {
    callbackQueue_->add([this, muted = muted]() mutable {
        deviceState_.MutableDeviceState()->SetMicsMuted(muted);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_MICS_MUTED);
        method->set_mics_muted(muted);

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setVolumeState(bool soundMuted, int soundLevel, std::optional<int> soundMaxLevel) {
    callbackQueue_->add([this, soundMuted, soundLevel, soundMaxLevel]() mutable {
        deviceState_.MutableDeviceState()->SetSoundMuted(soundMuted);
        deviceState_.MutableDeviceState()->SetSoundLevel(soundLevel);
        if (soundMaxLevel.has_value()) {
            deviceState_.MutableDeviceState()->SetSoundMaxLevel(*soundMaxLevel);
        } else {
            deviceState_.MutableDeviceState()->ClearSoundMaxLevel();
        }
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_VOLUME_STATE);

        const auto& deviceState = deviceState_.GetDeviceState();
        method->set_sound_muted(deviceState.GetSoundMuted());
        method->set_sound_level(deviceState.GetSoundLevel());
        if (deviceState.HasSoundMaxLevel()) {
            method->set_sound_max_level(deviceState.GetSoundMaxLevel());
        }

        connection_->sendMessage(remoting);
    });
}

void DeviceStateCapabilityProxy::setBattery(const NAlice::TDeviceState::TBattery& battery) {
    callbackQueue_->add([this, battery = battery]() mutable {
        deviceState_.MutableDeviceState()->MutableBattery()->Swap(&battery);
        if (!isConnected_) {
            return;
        }

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_device_state_capability_method();
        method->set_method(quasar::proto::Remoting::DeviceStateCapabilityMethod::SET_BATTERY);
        method->mutable_battery()->CopyFrom(deviceState_.GetDeviceState().GetBattery());
    });
}
