#include "file_player_capability_proxy.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace YandexIO;

namespace {
    const size_t MAX_QUEUE_SIZE = 20;
} // namespace

FilePlayerCapabilityProxy::FilePlayerCapabilityProxy(std::weak_ptr<IRemotingRegistry> remotingRegistry,
                                                     std::shared_ptr<quasar::ICallbackQueue> worker)
    : IRemoteObject(std::move(remotingRegistry))
    , callbackQueue_(std::move(worker))
    , remoteObjectId_("FilePlayerCapability")
{
}

void FilePlayerCapabilityProxy::init() {
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->addRemoteObject(remoteObjectId_, weak_from_this());
    }
}

FilePlayerCapabilityProxy::~FilePlayerCapabilityProxy() {
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->removeRemoteObject(remoteObjectId_);
    }
}

void FilePlayerCapabilityProxy::playSoundFile(const std::string& fileName,
                                              std::optional<quasar::proto::AudioChannel> channel,
                                              std::optional<PlayParams> params,
                                              std::shared_ptr<IPlaySoundFileListener> listener) {
    callbackQueue_->add([this, fileName, channel, params, listener]() {
        const std::string requestId = (params.has_value() && !params->requestId.empty())
                                          ? params->requestId
                                          : quasar::makeUUID();

        bool sendEvents = false;
        if (listener) {
            sendEvents = true;
            listenerByRequestId_[requestId] = listener;
        }

        ProtoFPCMethod method;
        method.set_method(ProtoFPCMethod::PLAY_SOUND_FILE);
        method.set_file_name(TString(fileName));
        if (channel.has_value()) {
            method.set_channel(channel.value());
        }
        method.set_send_events(sendEvents);

        method.mutable_play_params()->set_request_id(TString(requestId));
        if (params.has_value()) {
            method.mutable_play_params()->set_parent_request_id(TString(params->parentRequestId));
            method.mutable_play_params()->set_play_looped(params->playLooped);
            method.mutable_play_params()->set_play_times(params->playTimes);
        }

        sendOrEnqueueMethod(std::move(method));
    });
}

void FilePlayerCapabilityProxy::stopSoundFile(const std::string& fileName)
{
    callbackQueue_->add([this, fileName]() {
        ProtoFPCMethod method;
        method.set_method(ProtoFPCMethod::STOP_SOUND_FILE);
        method.set_file_name(TString(fileName));

        sendOrEnqueueMethod(std::move(method));
    });
}

void FilePlayerCapabilityProxy::handleRemotingConnect(std::shared_ptr<IRemotingConnection> connection) {
    callbackQueue_->add([this, connection]() {
        YIO_LOG_INFO("Request handshake to " << remoteObjectId_);
        connection_ = connection;

        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));
        remoting.mutable_file_player_capability_method()->set_method(ProtoFPCMethod::HANDSHAKE_REQUEST);
        connection_->sendMessage(remoting);
    });
}

void FilePlayerCapabilityProxy::handleRemotingMessage(const quasar::proto::Remoting& message,
                                                      std::shared_ptr<IRemotingConnection> /*connection*/) {
    callbackQueue_->add([this, message]() {
        handleRemotingMessageImpl(message);
    });
}

void FilePlayerCapabilityProxy::handleRemotingMessageImpl(const quasar::proto::Remoting& message) {
    if (message.has_file_player_capability_events_method()) {
        const auto& method = message.file_player_capability_events_method();

        if (method.has_handshake_response()) {
            YIO_LOG_INFO("Connected to " << remoteObjectId_);
            isConnected_ = true;
            sendQueuedMethods();
        } else if (method.has_directive_event()) {
            const auto& directiveEvent = method.directive_event();
            if (directiveEvent.has_directive()) {
                const auto& directive = directiveEvent.directive();

                auto iter = listenerByRequestId_.find(directive.request_id());
                if (iter == listenerByRequestId_.end()) {
                    return;
                }

                const auto& listener = iter->second;
                if (directiveEvent.has_started_event()) {
                    listener->onStarted();
                } else if (directiveEvent.has_completed_event()) {
                    listener->onCompleted();
                    listenerByRequestId_.erase(iter);
                }
            }
        }
    }
}

void FilePlayerCapabilityProxy::sendOrEnqueueMethod(ProtoFPCMethod method) {
    std::stringstream log;
    log << "method " << ProtoFPCMethod::Method_Name(method.method());
    log << ", file_name " << method.file_name();
    if (method.has_channel()) {
        log << ", channel " << quasar::proto::AudioChannel_Name(method.channel());
    }

    if (isConnected_) {
        YIO_LOG_INFO("Send: " << log.str());
        sendMethod(std::move(method));
    } else {
        YIO_LOG_INFO("Enqueue: " << log.str());
        const auto& fileName = method.file_name();
        const auto channel = method.has_channel() ? std::optional<quasar::proto::AudioChannel>(method.channel())
                                                  : std::nullopt;

        if (method.method() == ProtoFPCMethod::STOP_SOUND_FILE) {
            /// cancel sound by not sending any messages to capability
            queuedMethods_.remove_if([&fileName](ProtoFPCMethod protoMethod) {
                return protoMethod.file_name() == fileName;
            });
            return;
        }

        queuedMethods_.remove_if([&fileName, &channel](ProtoFPCMethod protoMethod) {
            /// cancel earlier sound, if the new one is to be played in the same channel
            if (protoMethod.has_channel() && channel.has_value() && protoMethod.channel() == channel.value()) {
                return true;
            }
            /// cancel earlier play request if we want to play the same sound again
            return protoMethod.file_name() == fileName;
        });
        queuedMethods_.emplace_back(std::move(method));

        while (queuedMethods_.size() > MAX_QUEUE_SIZE) {
            YIO_LOG_WARN("Queue overflow. Cancel play request for sound " << queuedMethods_.front().file_name());
            queuedMethods_.pop_front();
        }
    }
}

void FilePlayerCapabilityProxy::sendMethod(ProtoFPCMethod method) {
    quasar::proto::Remoting remoting;
    remoting.set_remote_object_id(TString(remoteObjectId_));
    remoting.mutable_file_player_capability_method()->CopyFrom(method);
    connection_->sendMessage(remoting);
}

void FilePlayerCapabilityProxy::sendQueuedMethods() {
    while (!queuedMethods_.empty()) {
        auto method = queuedMethods_.front();
        sendMethod(std::move(method));
        queuedMethods_.pop_front();
    }
}
