#include "playback_control_capability_proxy.h"

#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/libs/logging/logging.h>

using namespace YandexIO;

namespace {
    constexpr const auto REMOTE_NAME = "PlaybackControlCapability";
} // namespace

PlaybackControlCapabilityProxy::PlaybackControlCapabilityProxy(std::weak_ptr<IRemotingRegistry> remotingRegistry,
                                                               std::shared_ptr<quasar::ICallbackQueue> worker)
    : IRemoteObject(std::move(remotingRegistry))
    , callbackQueue_(std::move(worker))
    , remoteObjectId_(REMOTE_NAME)
{
}

PlaybackControlCapabilityProxy::~PlaybackControlCapabilityProxy()
{
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->removeRemoteObject(remoteObjectId_);
    }
}

void PlaybackControlCapabilityProxy::init()
{
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->addRemoteObject(remoteObjectId_, weak_from_this());
    }
}

void PlaybackControlCapabilityProxy::handleRemotingMessage(const quasar::proto::Remoting& /*message*/,
                                                           std::shared_ptr<IRemotingConnection> /*connection*/) {
    // ¯\_(ツ)_/¯
}

void PlaybackControlCapabilityProxy::handleRemotingConnect(std::shared_ptr<IRemotingConnection> connection) {
    callbackQueue_->add([this, connection]() {
        YIO_LOG_INFO("Connected to " << remoteObjectId_);
        connection_ = connection;
    });
}

void PlaybackControlCapabilityProxy::play() {
    callbackQueue_->add([this]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_playback_control_capability_method();
        method->set_method(quasar::proto::Remoting::PlaybackControlCapabilityMethod::PLAY);

        trySendMessage(remoting);
    });
}

void PlaybackControlCapabilityProxy::pause() {
    callbackQueue_->add([this]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_playback_control_capability_method();
        method->set_method(quasar::proto::Remoting::PlaybackControlCapabilityMethod::PAUSE);

        trySendMessage(remoting);
    });
}

void PlaybackControlCapabilityProxy::togglePlayPause(bool canRequestMusic) {
    callbackQueue_->add([this, canRequestMusic]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_playback_control_capability_method();
        method->set_method(quasar::proto::Remoting::PlaybackControlCapabilityMethod::TOGGLE_PLAY_PAUSE);
        method->set_can_request_music(canRequestMusic);

        trySendMessage(remoting);
    });
}

void PlaybackControlCapabilityProxy::rewind(std::uint32_t time) {
    callbackQueue_->add([this, time]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_playback_control_capability_method();
        method->set_method(quasar::proto::Remoting::PlaybackControlCapabilityMethod::REWIND);
        method->set_rewind_amount(time);
        trySendMessage(remoting);
    });
}

void PlaybackControlCapabilityProxy::like() {
    callbackQueue_->add([this]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_playback_control_capability_method();
        method->set_method(quasar::proto::Remoting::PlaybackControlCapabilityMethod::LIKE);

        trySendMessage(remoting);
    });
}

void PlaybackControlCapabilityProxy::dislike() {
    callbackQueue_->add([this]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_playback_control_capability_method();
        method->set_method(quasar::proto::Remoting::PlaybackControlCapabilityMethod::DISLIKE);

        trySendMessage(remoting);
    });
}

void PlaybackControlCapabilityProxy::next() {
    callbackQueue_->add([this]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_playback_control_capability_method();
        method->set_method(quasar::proto::Remoting::PlaybackControlCapabilityMethod::NEXT);

        trySendMessage(remoting);
    });
}

void PlaybackControlCapabilityProxy::prev() {
    callbackQueue_->add([this]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_playback_control_capability_method();
        method->set_method(quasar::proto::Remoting::PlaybackControlCapabilityMethod::PREV);

        trySendMessage(remoting);
    });
}

void PlaybackControlCapabilityProxy::trySendMessage(const quasar::proto::Remoting& message) {
    if (!connection_) {
        YIO_LOG_WARN("Not connected to " << remoteObjectId_);
        return;
    }
    connection_->sendMessage(message);
}
