#include "spotter_capability_proxy.h"

#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/libs/logging/logging.h>

using namespace YandexIO;

namespace {
    constexpr const auto ACTIVATION_SPOTTER_REMOTE_OBJECT_ID = "ActivationSpotterCapability";
    constexpr const auto COMMAND_SPOTTER_REMOTE_OBJECT_ID = "CommandSpotterCapability";
    constexpr const auto NAVI_OLD_SPOTTER_REMOTE_OBJECT_ID = "NaviOldSpotterCapability";
} // namespace

std::shared_ptr<SpotterCapabilityProxy> SpotterCapabilityProxy::createActivation(std::weak_ptr<IRemotingRegistry> remotingRegistry,
                                                                                 std::shared_ptr<quasar::ICallbackQueue> worker) {
    return std::shared_ptr<SpotterCapabilityProxy>(new SpotterCapabilityProxy(std::move(remotingRegistry),
                                                                              std::move(worker),
                                                                              ACTIVATION_SPOTTER_REMOTE_OBJECT_ID));
}

std::shared_ptr<SpotterCapabilityProxy> SpotterCapabilityProxy::createCommand(std::weak_ptr<IRemotingRegistry> remotingRegistry,
                                                                              std::shared_ptr<quasar::ICallbackQueue> worker) {
    return std::shared_ptr<SpotterCapabilityProxy>(new SpotterCapabilityProxy(std::move(remotingRegistry),
                                                                              std::move(worker),
                                                                              COMMAND_SPOTTER_REMOTE_OBJECT_ID));
}

std::shared_ptr<SpotterCapabilityProxy> SpotterCapabilityProxy::createNaviOld(std::weak_ptr<IRemotingRegistry> remotingRegistry,
                                                                              std::shared_ptr<quasar::ICallbackQueue> worker) {
    return std::shared_ptr<SpotterCapabilityProxy>(new SpotterCapabilityProxy(std::move(remotingRegistry),
                                                                              std::move(worker),
                                                                              NAVI_OLD_SPOTTER_REMOTE_OBJECT_ID));
}

SpotterCapabilityProxy::SpotterCapabilityProxy(std::weak_ptr<IRemotingRegistry> remotingRegistry,
                                               std::shared_ptr<quasar::ICallbackQueue> worker,
                                               const std::string& remoteObjectId)
    : IRemoteObject(std::move(remotingRegistry))
    , worker_(std::move(worker))
    , remoteObjectId_(remoteObjectId)
{
}

SpotterCapabilityProxy::~SpotterCapabilityProxy() {
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->removeRemoteObject(remoteObjectId_);
    }
}

void SpotterCapabilityProxy::init()
{
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->addRemoteObject(remoteObjectId_, weak_from_this());
    }
}

void SpotterCapabilityProxy::handleRemotingConnect(std::shared_ptr<IRemotingConnection> connection) {
    worker_->add([this, connection] {
        YIO_LOG_INFO("Connected to " << remoteObjectId_);
        connection_ = connection;
        setModelPaths();
        setSpotterWord();
    });
}

void SpotterCapabilityProxy::handleRemotingMessage(const quasar::proto::Remoting& message,
                                                   std::shared_ptr<IRemotingConnection> /*connection*/) {
    if (!message.has_spotter_capability_listener_method()) {
        return;
    }

    worker_->add([this, message] {
        handleListenerMethod(message.spotter_capability_listener_method());
    });
}

void SpotterCapabilityProxy::setModelPaths(const std::map<std::string, std::string>& spotterTypeToModelPath) {
    worker_->add([this, spotterTypeToModelPath] {
        quasar::proto::Remoting::SpotterCapabilityMethod method;
        method.set_method(quasar::proto::Remoting::SpotterCapabilityMethod::SET_MODEL_PATHS);
        for (const auto& [type, path] : spotterTypeToModelPath) {
            auto modelPathProto = method.add_model_path();
            modelPathProto->set_type(TString(type));
            modelPathProto->set_path(TString(path));
        }

        setModelPaths_ = std::move(method);

        setModelPaths();
    });
}

void SpotterCapabilityProxy::setSpotterWord(const std::string& spotterWord) {
    worker_->add([this, spotterWord] {
        quasar::proto::Remoting::SpotterCapabilityMethod method;
        method.set_method(quasar::proto::Remoting::SpotterCapabilityMethod::SET_SPOTTER_WORD);
        method.set_spotter_word(TString(spotterWord));

        setSpotterWord_ = std::move(method);

        setSpotterWord();
    });
}

void SpotterCapabilityProxy::addListener(std::weak_ptr<IListener> listener) {
    worker_->add([this, listener{std::move(listener)}]() mutable {
        listeners_.insert(std::move(listener));
    });
}

void SpotterCapabilityProxy::removeListener(std::weak_ptr<IListener> listener) {
    worker_->add([this, listener{std::move(listener)}] {
        listeners_.erase(listener);
    });
}

void SpotterCapabilityProxy::handleListenerMethod(const quasar::proto::Remoting::SpotterCapabilityListenerMethod& method) {
    std::set<std::string> spotterTypes{method.spotter_type().cbegin(), method.spotter_type().cend()};
    switch (method.method()) {
        case quasar::proto::Remoting::SpotterCapabilityListenerMethod::ON_MODEL_ERROR: {
            notifyModelError(spotterTypes);
            break;
        }
        case quasar::proto::Remoting::SpotterCapabilityListenerMethod::ON_MODEL_SET: {
            notifyModelSet(spotterTypes);
            break;
        }
    }
}

void SpotterCapabilityProxy::setModelPaths() {
    callMethod(setModelPaths_);
}

void SpotterCapabilityProxy::setSpotterWord() {
    callMethod(setSpotterWord_);
}

void SpotterCapabilityProxy::callMethod(std::optional<quasar::proto::Remoting::SpotterCapabilityMethod>& method) {
    if (method.has_value() && connection_) {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));
        remoting.mutable_spotter_capability_method()->CopyFrom(*method);

        connection_->sendMessage(remoting);
    }
}

void SpotterCapabilityProxy::notifyModelError(const std::set<std::string>& spotterTypes) {
    for (const auto& wlistener : listeners_) {
        if (auto listener = wlistener.lock()) {
            listener->onModelError(shared_from_this(), spotterTypes);
        }
    }
}

void SpotterCapabilityProxy::notifyModelSet(const std::set<std::string>& spotterTypes) {
    for (const auto& wlistener : listeners_) {
        if (auto listener = wlistener.lock()) {
            listener->onModelSet(shared_from_this(), spotterTypes);
        }
    }
}
