#include "volume_capability.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/json_utils/json_utils.h>

#include <alice/protos/endpoint/capabilities/volume/capability.pb.h>

using namespace quasar;
using namespace YandexIO;

VolumeCapability::VolumeCapability(std::shared_ptr<AliceVolumeSetter> aliceVolumeSetter)
    : aliceVolumeSetter_(std::move(aliceVolumeSetter))
{
}

NAlice::TCapabilityHolder VolumeCapability::getState() const {
    NAlice::TCapabilityHolder capabilityHolder;
    auto volumeCapability = capabilityHolder.mutable_volumecapability();

    volumeCapability->mutable_meta();
    volumeCapability->mutable_parameters();
    volumeCapability->mutable_state();

    return capabilityHolder;
}

IDirectiveHandlerPtr VolumeCapability::getDirectiveHandler()
{
    return shared_from_this();
}

void VolumeCapability::addListener(std::weak_ptr<IListener> wlistener)
{
    Y_UNUSED(wlistener);
}

void VolumeCapability::removeListener(std::weak_ptr<IListener> wlistener)
{
    Y_UNUSED(wlistener);
}

const std::string& VolumeCapability::getHandlerName() const {
    static const std::string CAPABILITY_NAME{"VolumeCapability"};
    return CAPABILITY_NAME;
}

const std::set<std::string>& VolumeCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> names = {
        Directives::SOUND_LOUDER,
        Directives::SOUND_QUITER,
        Directives::SOUND_MUTE,
        Directives::SOUND_UNMUTE,
        Directives::SOUND_SET_LEVEL,
    };

    return names;
}

void VolumeCapability::handleDirective(const std::shared_ptr<Directive>& directive)
{
    const auto& data = directive->getData();
    const std::string& name = data.name;
    const Json::Value& payload = data.payload;

    if (name == Directives::SOUND_LOUDER) {
        handleSoundLouder();
    } else if (name == Directives::SOUND_QUITER) {
        handleSoundQuieter();
    } else if (name == Directives::SOUND_MUTE) {
        handleSoundMute();
    } else if (name == Directives::SOUND_UNMUTE) {
        handleSoundUnmute();
    } else if (name == Directives::SOUND_SET_LEVEL && !payload.isNull() && !payload["new_level"].isNull()) {
        handleSoundSetLevel(payload);
    }
}

void VolumeCapability::handleSoundLouder()
{
    aliceVolumeSetter_->volumeUp();
}

void VolumeCapability::handleSoundQuieter()
{
    aliceVolumeSetter_->volumeDown();
}

void VolumeCapability::handleSoundMute()
{
    aliceVolumeSetter_->mute();
}

void VolumeCapability::handleSoundUnmute()
{
    aliceVolumeSetter_->unmute();
}

void VolumeCapability::handleSoundSetLevel(const Json::Value& payload)
{
    aliceVolumeSetter_->setVolume(getInt(payload, "new_level"));
}
