#pragma once

/* http_header.h                                                 -*- C++ -*-
   Jeremy Barnes, 18 February 2011
   Copyright (c) 2011 Datacratic.  All rights reserved.

   http header parsing class.
*/

#pragma once

#include <string>
#include <map>
#include <unordered_map>
#include <iostream>
#include <vector>
#include <yandex_io/external_libs/datacratic/jml/arch/exception.h>

namespace Datacratic {
    enum class HttpVerb {
        GET,
        HEAD,
        POST,
        PUT,
        DELETE,
        TRACE,
        OPTIONS,
        PATCH,
        UNKNOWN
    };

} // namespace Datacratic

namespace std {
    template <>
    struct hash<::Datacratic::HttpVerb> {
        using argument_type = ::Datacratic::HttpVerb;
        using underlying_type = std::underlying_type<argument_type>::type;
        using result_type = std::hash<underlying_type>::result_type;
        result_type operator()(const argument_type& arg) const {
            std::hash<underlying_type> hasher;
            return hasher(static_cast<underlying_type>(arg));
        }
    };
} // namespace std

namespace Datacratic {
    static const std::unordered_map<HttpVerb, std::string> VerbToString = {
        {HttpVerb::GET, "GET"}, {HttpVerb::HEAD, "HEAD"}, {HttpVerb::POST, "POST"}, {HttpVerb::PUT, "PUT"}, {HttpVerb::DELETE, "DELETE"}, {HttpVerb::TRACE, "TRACE"}, {HttpVerb::OPTIONS, "OPTIONS"}, {HttpVerb::PATCH, "PATCH"}};

    static const std::unordered_map<std::string, HttpVerb> StringToVerb = {
        {
            "GET",
            HttpVerb::GET,
        },
        {"HEAD", HttpVerb::HEAD},
        {"POST", HttpVerb::POST},
        {"PUT", HttpVerb::PUT},
        {"DELETE", HttpVerb::DELETE},
        {"TRACE", HttpVerb::TRACE},
        {"OPTIONS", HttpVerb::OPTIONS},
        {"PATCH", HttpVerb::PATCH}};

    HttpVerb string_to_verb(const std::string& s);

    std::string verb_to_string(const HttpVerb verb);

    /*****************************************************************************/
    /* REST PARAMS                                                               */
    /*****************************************************************************/

    struct RestParams
        : public std::vector<std::pair<std::string, std::string>> {
        RestParams()
        {
        }

        RestParams(std::initializer_list<std::pair<std::string, std::string>> l)
            : std::vector<std::pair<std::string, std::string>>(l.begin(), l.end())
        {
        }

        bool hasValue(const std::string& key) const;

        /** Return the value of the given key.  Throws an exception if it's not
        found.
    */
        std::string getValue(const std::string& key) const;

        std::string uriEscaped() const;
    };

    /*****************************************************************************/
    /* HTTP HEADER                                                               */
    /*****************************************************************************/

    /** Header for an HTTP request.  Just a place to dump the data. */

    struct HttpHeader {
        HttpHeader()
            : contentLength(-1)
            , isChunked(false)
        {
        }

        void swap(HttpHeader& other);

        void parse(const std::string& headerAndData, bool checkBodyLength = true);

        std::string verb;     // GET, PUT, etc
        HttpVerb enum_verb;   // Enum version of verb field
        std::string resource; // after the get
        std::string version;  // after the get
        std::string query;

        int responseCode() const; // for responses; parses it out of the "version" field

        RestParams queryParams; // Query parameters pulled out of the URL

        // These headers are automatically pulled out
        std::string contentType;
        int64_t contentLength;
        bool isChunked;

        // The rest of the headers are here
        std::map<std::string, std::string> headers;

        std::string getHeader(const std::string& key) const {
            auto it = headers.find(key);
            if (it == headers.end()) {
                throw ML::Exception("couldn't find header " + key);
            }
            return it->second;
        }

        std::string tryGetHeader(const std::string& key) const {
            auto it = headers.find(key);
            if (it == headers.end()) {
                return "";
            }
            return it->second;
        }

        // If some portion of the data is known, it's put in here
        std::string knownData;
    };

    std::ostream& operator<<(std::ostream& stream, const HttpHeader& header);

    /** Returns the reason phrase for the given code.
    See http://www.w3.org/Protocols/rfc2616/rfc2616-sec6.html
*/
    std::string getResponseReasonPhrase(int code);

} // namespace Datacratic
