/* passive_endpoint.cc
   Jeremy Barnes, 29 April 2011
   Copyright (c) 2011 Datacratic.  All rights reserved.

   Passive endpoint implementation.
*/

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/external_libs/datacratic/jml/arch/futex.h>
#include <yandex_io/external_libs/datacratic/soa/service/passive_endpoint.h>

#include <util/system/byteorder.h>

#include <unordered_map>

#include <poll.h>
#include <fcntl.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/prctl.h>

using namespace std;
using namespace ML;

namespace Datacratic {

    /*****************************************************************************/
    /* PASSIVE ENDPOINT                                                          */
    /*****************************************************************************/

    PassiveEndpoint::
        PassiveEndpoint(const std::string& name)
        : EndpointBase(name)
    {
    }

    PassiveEndpoint::
        ~PassiveEndpoint()
    {
        closePeer();
        shutdown();
    }

    int
    PassiveEndpoint::
        init(PortRange const& portRange, const std::string& hostname, int num_threads, bool synchronous,
             bool nameLookup, int backlog)
    {
        std::string threadName(32, '\0');
        prctl(PR_GET_NAME, threadName.data(), 0, 0, 0);
        threadName.resize(std::strlen(threadName.data()));
        threadName += "_QS";

        spinup(num_threads, synchronous, threadName);

        int port = listen(portRange, hostname, nameLookup, backlog);
        YIO_LOG_INFO(name() << " listening on hostname " << hostname << " port " << port);
        return port;
    }

    /*****************************************************************************/
    /* ACCEPTOR FOR SOCKETTRANSPORT                                              */
    /*****************************************************************************/

    INET_Addr::INET_Addr()
    {
        memset(&addr_, 0, sizeof(addr_));
        memset(&addr6_, 0, sizeof(addr6_));
    }

    INET_Addr::INET_Addr(int port, const char host[], int family)
        : family_(family)
        , port_(port)
    {
        set(port_, host, family_);
    }

    int INET_Addr::set(int port, const char host[], int family)
    {
        addrinfo hints;
        memset(&hints, 0, sizeof(hints));
        hints.ai_family = family;
        hints.ai_socktype = SOCK_STREAM;
        addrinfo* res = nullptr;
        int result = getaddrinfo(host, nullptr, &hints, &res);
        if (result != 0)
        {
            return -1;
        }

        if (family != AF_INET6)
        {
            memset(&addr_, 0, sizeof(addr_));
            addr_.sin_port = HostToInet<uint16_t>(port);
            addr_.sin_family = family;
            memcpy(&addr_.sin_addr, &(((struct sockaddr_in*)(res->ai_addr))->sin_addr), sizeof(addr_.sin_addr));
        } else {
            memset(&addr6_, 0, sizeof(addr6_));
            addr6_.sin6_port = HostToInet<uint16_t>(port);
            addr6_.sin6_family = family;
            memcpy(&addr6_.sin6_addr, &(((struct sockaddr_in6*)(res->ai_addr))->sin6_addr), sizeof(addr6_.sin6_addr));
        }
        freeaddrinfo(res);
        family_ = family;

        return 0;
    }

    void INET_Addr::set(const sockaddr_in* addr, size_t addrSize)
    {
        addr_ = *addr;
    }

    void* INET_Addr::get_addr() const {
        if (family_ != AF_INET6)
            return (void*)&addr_;
        else
            return (void*)&addr6_;
    }

    size_t INET_Addr::get_addr_size() const {
        if (family_ != AF_INET6)
            return sizeof(addr_);
        else
            return sizeof(addr6_);
    }
    int INET_Addr::get_port_number() const {
        return port_;
    }

    AcceptorT<SocketTransport>::
        AcceptorT()
        : fd(-1)
        , endpoint(0)
        , listening_(false)
    {
    }

    AcceptorT<SocketTransport>::
        ~AcceptorT()
    {
        closePeer();
    }

    int
    AcceptorT<SocketTransport>::
        listen(PortRange const& portRange,
               const std::string& hostname,
               PassiveEndpoint* endpoint,
               bool nameLookup,
               int backlog)
    {
        closePeer();

        this->endpoint = endpoint;
        this->nameLookup = nameLookup;

        fd = socket(endpoint->socketFamily_, SOCK_STREAM | SOCK_CLOEXEC, 0);

        // Avoid already bound messages for the minute after a server has exited
        int tr = 1;
        int res = setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &tr, sizeof(int));

        if (res == -1) {
            close(fd);
            fd = -1;
            throw Exception("error setsockopt SO_REUSEADDR: " + ML::strerror_r(errno));
        }

        const char* hostNameToUse = (hostname == "*" ? "0.0.0.0" : hostname.c_str());

        int port = portRange.bindPort([&](int port)
                                      {
                                          memset((void*)&addr, 0, sizeof(addr));
                                          addr = INET_Addr(port, hostNameToUse, endpoint->socketFamily_);

                                          // cerr << "port = " << port
                                          //      << " hostname = " << hostname
                                          //      << " addr = " << addr.get_host_name() << " "
                                          //      << addr.get_host_addr() << " "
                                          //      << addr.get_ip_address() << endl;

                                          int res = ::bind(fd,
                                                           reinterpret_cast<sockaddr*>(addr.get_addr()),
                                                           addr.get_addr_size());
                                          if (res == -1 && errno != EADDRINUSE)
                                              throw Exception("listen: bind returned " + ML::strerror_r(errno));
                                          return res == 0;
                                      });

        if (port == -1) {
            // fixme: remove it: DEVTOOLSSUPPORT-5730
            std::system("netstat -ltn");
            throw Exception("couldn't bind to any port in range [%d,%d]", portRange.first,
                            portRange.last);
        }

        // Avoid already bound messages for the minute after a server has exited
        res = setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &tr, sizeof(int));

        if (res == -1) {
            close(fd);
            fd = -1;
            throw Exception("error setsockopt SO_REUSEADDR: " + ML::strerror_r(errno));
        }

        res = ::listen(fd, backlog);

        if (res == -1) {
            close(fd);
            fd = -1;
            throw Exception("error on listen: " + std::to_string(errno) + " " + ML::strerror_r(errno));
        }

        if (port == 0) {
            sockaddr_in inAddr;
            socklen_t inAddrLen = sizeof(inAddr);
            res = ::getsockname(fd, (sockaddr*)&inAddr, &inAddrLen);
            port = InetToHost<uint16_t>(inAddr.sin_port);
            addr.set(&inAddr, inAddrLen);
        }

        listening_ = true;
        ML::futex_wake(listening_);

        shutdown = false;

        acceptThread.reset(new std::thread([=]() { this->runAcceptThread(); }));
        return port;
    }

    void
    AcceptorT<SocketTransport>::
        closePeer()
    {
        if (!acceptThread)
            return;
        shutdown = true;

        std::atomic_thread_fence(memory_order_seq_cst);

        wakeup.signal();

        close(fd);
        fd = -1;

        acceptThread->join();
        acceptThread.reset();
    }

    std::string
    AcceptorT<SocketTransport>::
        hostname() const {
        return "-";
    }

    int
    AcceptorT<SocketTransport>::
        port() const {
        return addr.get_port_number();
    }

    struct NameEntry {
        NameEntry(const std::string& name)
            : name_(name)
            , date_(Date::now())
        {
        }

        string name_;
        Date date_;
    };

    void
    AcceptorT<SocketTransport>::
        runAcceptThread()
    {
        // static const char *fName = "AcceptorT<SocketTransport>::runAcceptThread:";
        unordered_map<string, NameEntry> addr2Name;

        int res = fcntl(fd, F_SETFL, O_NONBLOCK);
        if (res != 0) {
            if (shutdown)
                return; // deal with race between start up and shut down
            throw ML::Exception(errno, "fcntl");
        }

        while (!shutdown) {
            sockaddr_in addr;
            socklen_t addr_len = sizeof(addr);
            // cerr << "accept on fd " << fd << endl;

            pollfd fds[2] = {
                {fd, POLLIN, 0},
                {wakeup.fd(), POLLIN, 0}};

            int res = ::poll(fds, 2, -1);

            // cerr << "accept poll returned " << res << endl;

            if (shutdown)
                return;

            if (res == -1 && errno == EINTR)
                continue;
            if (res == 0)
                throw ML::Exception("should not be no fds ready to accept");

            if (fds[1].revents) {
                wakeup.read();
            }

            if (!fds[0].revents)
                continue;

            res = accept4(fd, (sockaddr*)&addr, &addr_len, SOCK_CLOEXEC);

            // cerr << "accept returned " << res << endl;

            if (res == -1 && (errno == EWOULDBLOCK || errno == EAGAIN))
                continue;

            if (res == -1 && errno == EINTR)
                continue;
            if (-1 == res && errno == ECONNABORTED)
                continue;

            if (res == -1)
                endpoint->acceptError("accept: " + ML::strerror_r(errno));

#if 0
        union {
            char octets[4];
            uint32_t addr;
        } a;
        a.addr = addr.sin_addr;
#endif

            INET_Addr addr2;
            addr2.set(&addr, addr_len);

            std::shared_ptr<SocketTransport> newTransport(new SocketTransport(this->endpoint));

            newTransport->peer_ = SOCK_Stream(res);
            string peerName = "<unknown>";
            newTransport->peerName_ = peerName;
            endpoint->associateHandler(newTransport);

            /* cleanup name entries older than 5 seconds */
            Date now = Date::now();
            auto it = addr2Name.begin();
            while (it != addr2Name.end()) {
                const NameEntry& entry = it->second;
                if (entry.date_.plusSeconds(5) < now) {
                    it = addr2Name.erase(it);
                } else {
                    it++;
                }
            }
        }
    }

    void
    AcceptorT<SocketTransport>::
        waitListening()
            const {
        while (!listening_) {
            int oldListening = listening_;
            ML::futex_wait(listening_, oldListening);
        }
    }

} // namespace Datacratic
