#include "auth_provider.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/private/device_context.h>

#include <algorithm>
#include <ostream>

using namespace quasar;

namespace {

    int64_t makeTag() {
        return getNowTimestampMs();
    }

    class SdkContext: public AuthProvider::ISdkContext {
    public:
        SdkContext(const std::shared_ptr<ipc::IIpcFactory>& ipcFactory)
            : sdkContext_(ipcFactory, nullptr, false)
        {
        }

        void connectToUserAccountInfoChanged(std::function<void(const std::string&, const std::string&)> onUserAccountInfoChanged) override {
            sdkContext_.onUserAccountInfo = onUserAccountInfoChanged;
            sdkContext_.connectToSDK();
        }

        void fireOAuthTokenIsInvalid(const std::string& token) override {
            sdkContext_.fireOAuthTokenIsInvalid(token);
        }

    private:
        YandexIO::DeviceContext sdkContext_;
    };

    proto::AccountType userTypeToProto(quasar::UserType userType) {
        switch (userType) {
            case quasar::UserType::OWNER:
                return proto::AccountType::OWNER;
            case quasar::UserType::GUEST:
                return proto::AccountType::GUEST;
        };
    }

    quasar::UserType userTypeFromProto(const proto::AccountType& accountType) {
        switch (accountType) {
            case proto::AccountType::OWNER:
                return quasar::UserType::OWNER;
            case proto::AccountType::GUEST:
                return quasar::UserType::GUEST;
            case proto::AccountType::UNDEFINED:;
        }

        throw std::runtime_error("Invalid account type: " + PROTO_ENUM_TO_STRING(proto::AccountType, accountType));
    }

    quasar::AuthInfo2 authInfoFromProto(const proto::AuthEvent& event) {
        return AuthInfo2{
            .source = AuthInfo2::Source::AUTHD,
            .authToken = event.auth_token(),
            .passportUid = event.passport_uid(),
            .userType = userTypeFromProto(event.type()),
            .tag = event.tag()};
    }

    std::ostream& operator<<(std::ostream& os, quasar::AuthInfo2::Source source) {
        switch (source) {
            case quasar::AuthInfo2::Source::SDK: {
                os << "SDK";
                break;
            }
            case quasar::AuthInfo2::Source::AUTHD: {
                os << "AUTHD";
                break;
            }
            case quasar::AuthInfo2::Source::UNDEFINED: {
                os << "UNDEFINED";
                break;
            }
        }
        return os;
    }

    std::ostream& operator<<(std::ostream& os, quasar::UserType userType) {
        switch (userType) {
            case quasar::UserType::OWNER: {
                os << "OWNER";
                break;
            }
            case quasar::UserType::GUEST: {
                os << "GUEST";
                break;
            }
        }
        return os;
    }

    std::ostream& operator<<(std::ostream& os, const quasar::AuthInfo2& authInfo) {
        os << "uid: " << authInfo.passportUid << ", authToken: " << maskToken(authInfo.authToken)
           << ", userType: " << authInfo.userType << ", source: " << authInfo.source << ", tag: " << authInfo.tag;
        return os;
    }

    std::ostream& operator<<(std::ostream& os, const quasar::IAuthProvider::UsersAuthInfo& usersAuthInfo) {
        for (const auto& user : usersAuthInfo) {
            os << user << "; ";
        }

        return os;
    }

    AuthProvider::UsersAuthInfo::iterator findUser(AuthProvider::UsersAuthInfo& users, const AuthInfo2& authInfo) {
        return std::find_if(users.begin(), users.end(),
                            [&authInfo](const auto& el) {
                                return el.passportUid == authInfo.passportUid;
                            });
    }

} // namespace

AuthProvider::AuthProvider(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : AuthProvider(ipcFactory->createIpcConnector("authd"), std::make_shared<SdkContext>(ipcFactory))
{
}

AuthProvider::AuthProvider(std::shared_ptr<ipc::IConnector> authdConnector, std::shared_ptr<ISdkContext> sdkContext)
    : authdConnector_(std::move(authdConnector))
    , sdkContext_(sdkContext)
    , ownerInfo_(std::make_shared<AuthInfo2>())
    , allUsersInfo_(std::make_shared<UsersAuthInfo>())
{
    if (sdkContext) {
        sdkContext->connectToUserAccountInfoChanged(makeSafeCallback(
            [this](const std::string& authToken, const std::string& passportUid) {
                onUserAccountInfoChanged(authToken, passportUid);
            }, lifetime_));
    }

    if (authdConnector_) {
        authdConnector_->setMessageHandler(makeSafeCallback(
            [this](const auto& message) {
                onQuasarMessage(message);
            }, lifetime_));
        authdConnector_->connectToService();
    }
}

AuthProvider::~AuthProvider()
{
    lifetime_.die();
}

AuthProvider::IAuthInfo& AuthProvider::ownerAuthInfo()
{
    return ownerInfo_;
}

AuthProvider::IUsersAuthInfo& AuthProvider::usersAuthInfo()
{
    return allUsersInfo_;
}

AuthProvider::AddUserResponse AuthProvider::addUser(const std::string& authCode, UserType userType, bool withXToken, std::chrono::milliseconds timeout)
{
    authdConnector_->waitUntilConnected(std::chrono::seconds{5});
    auto addUserRequest = ipc::buildUniqueMessage([&authCode, userType, withXToken](auto& msg) {
        msg.mutable_add_user_request()->set_auth_code(TString(authCode));
        msg.mutable_add_user_request()->set_account_type(userTypeToProto(userType));
        msg.mutable_add_user_request()->set_with_xtoken(withXToken);
    });

    try {
        const auto authResponse = authdConnector_->sendRequestSync(std::move(addUserRequest), timeout);

        AddUserResponse addUserResponse;
        if (authResponse->has_add_user_response()) {
            switch (authResponse->add_user_response().status()) {
                case proto::AddUserResponse::OK:
                    addUserResponse.status = AddUserResponse::Status::OK;
                    break;
                case proto::AddUserResponse::NO_INTERNET:
                    addUserResponse.status = AddUserResponse::Status::NO_INTERNET;
                    break;
                case proto::AddUserResponse::CODE_EXPIRED:
                    addUserResponse.status = AddUserResponse::Status::CODE_EXPIRED;
                    break;
                case proto::AddUserResponse::CRYPTO_ERROR:
                    addUserResponse.status = AddUserResponse::Status::CRYPTO_ERROR;
                    break;
                case proto::AddUserResponse::INVALID_TOKEN:
                    addUserResponse.status = AddUserResponse::Status::INVALID_TOKEN;
                    break;
            }
            addUserResponse.authToken = authResponse->add_user_response().auth_token();
            addUserResponse.xToken = authResponse->add_user_response().xtoken();
            addUserResponse.id = authResponse->add_user_response().id();
            addUserResponse.tag = authResponse->add_user_response().tag();
        }

        return addUserResponse;
    } catch (const std::exception& e) {
        return AddUserResponse{
            .status = AddUserResponse::Status::TIMEOUT,
        };
    }
}

AuthProvider::DeleteUserResponse AuthProvider::deleteUser(int64_t id, std::chrono::milliseconds timeout)
{
    authdConnector_->waitUntilConnected(std::chrono::seconds{5});
    auto deleteUserRequest = ipc::buildUniqueMessage([&id](auto& msg) {
        msg.mutable_delete_user_request()->set_id(id);
    });

    try {
        const auto authResponse = authdConnector_->sendRequestSync(std::move(deleteUserRequest), timeout);

        DeleteUserResponse deleteUserResponse;
        if (authResponse->has_delete_user_response()) {
            switch (authResponse->delete_user_response().status()) {
                case proto::DeleteUserResponse::OK:
                    deleteUserResponse.status = DeleteUserResponse::Status::OK;
                    break;
                case proto::DeleteUserResponse::WRONG_USER:
                    deleteUserResponse.status = DeleteUserResponse::Status::WRONG_USER;
                    break;
            }
        }
        return deleteUserResponse;
    } catch (const std::exception& e) {
        return DeleteUserResponse{
            .status = DeleteUserResponse::Status::TIMEOUT,
        };
    }
}

AuthProvider::ChangeUserResponse AuthProvider::changeUser(int64_t id, std::chrono::milliseconds timeout)
{
    authdConnector_->waitUntilConnected(std::chrono::seconds{5});
    auto changeUserRequest = ipc::buildUniqueMessage([id](auto& msg) {
        msg.mutable_change_user_request()->set_id(id);
    });

    try {
        const auto authResponse = authdConnector_->sendRequestSync(std::move(changeUserRequest), timeout);

        return ChangeUserResponse{
            .status = ChangeUserResponse::Status::OK,
        };
    } catch (const std::exception& e) {
        return ChangeUserResponse{
            .status = ChangeUserResponse::Status::TIMEOUT,
        };
    }
}

void AuthProvider::requestAuthTokenUpdate(std::string_view source)
{
    YIO_LOG_INFO("requestAuthTokenUpdate " << source);
    if (sdkAuthInfo_.has_value()) {
        if (!sdkAuthInfo_->authToken.empty()) {
            sdkContext_->fireOAuthTokenIsInvalid(sdkAuthInfo_->authToken);
        }
    } else if (quasarAuthInfo_.has_value()) {
        auto request = ipc::buildUniqueMessage([this, source](auto& msg) {
            msg.mutable_auth_token_update_request()->set_auth_token(TString(quasarAuthInfo_->authToken));
            msg.mutable_auth_token_update_request()->set_source(TString(source));
        });
        authdConnector_->sendMessage(std::move(request));
    }
}

void AuthProvider::onQuasarMessage(const ipc::SharedMessage& message)
{
    if (sdkAuthInfo_.has_value()) {
        // Skip any message from authd if authInfo was set through sdk.
        return;
    }

    if (message->has_all_startup_info()) {
        onStartupInfo(message->all_startup_info());
    }

    if (message->has_add_user_event()) {
        onAddUser(message->add_user_event());
    }

    if (message->has_change_user_event()) {
        onChangeUser(message->change_user_event());
    }

    if (message->has_refresh_user_event()) {
        onRefreshUser(message->refresh_user_event());
    }

    if (message->has_change_token_event()) {
        onChangeToken(message->change_token_event());
    }

    if (message->has_delete_user_event()) {
        onDeleteUser(message->delete_user_event());
    }
}

void AuthProvider::onStartupInfo(const proto::AllStartupInfo& startupInfo)
{
    UsersAuthInfo usersAuthInfo;
    AuthInfo2 ownerAuthInfo{
        .source = AuthInfo2::Source::AUTHD,
        .authToken = "",
        .passportUid = "",
        .userType = UserType::OWNER,
        .tag = makeTag()};

    for (const auto& authEvent : startupInfo.accounts()) {
        auto authInfo = authInfoFromProto(authEvent);
        if (authInfo.userType == UserType::OWNER) {
            ownerAuthInfo = authInfo;
        }
        usersAuthInfo.push_back(std::move(authInfo));
    }

    std::scoped_lock lock{mutex_};
    updateOwnerInfo(ownerAuthInfo);
    if (usersAuthInfo.empty()) {
        usersAuthInfo.push_back(std::move(ownerAuthInfo));
    }

    updateAllUsers("StartupInfo", std::move(usersAuthInfo));
}

void AuthProvider::onAddUser(const proto::AuthEvent& event)
{
    std::scoped_lock lock{mutex_};

    auto authInfo = authInfoFromProto(event);

    YIO_LOG_INFO("Add user: " << authInfo);

    auto currentUsersAuthInfo = *allUsersInfo_.value();
    currentUsersAuthInfo.push_back(authInfo);

    updateAllUsers("AddUser", std::move(currentUsersAuthInfo));
}

void AuthProvider::onChangeUser(const proto::AuthEvent& event)
{
    std::scoped_lock lock{mutex_};
    changeUser(authInfoFromProto(event), "ChangeUser");
}

void AuthProvider::onRefreshUser(const proto::AuthEvent& event)
{
    std::scoped_lock lock{mutex_};

    auto authInfo = authInfoFromProto(event);

    YIO_LOG_INFO("Refresh user: " << authInfo);

    auto currentUsersAuthInfo = *allUsersInfo_.value();
    auto foundIt = findUser(currentUsersAuthInfo, authInfo);

    if (foundIt != currentUsersAuthInfo.end()) {
        *foundIt = authInfo;
    } else {
        currentUsersAuthInfo.push_back(authInfo);
    }

    if (authInfo.passportUid == ownerInfo_.value()->passportUid) {
        updateOwnerInfo(std::move(authInfo));
    }

    updateAllUsers("RefreshUser", std::move(currentUsersAuthInfo));
}

void AuthProvider::onChangeToken(const proto::AuthEvent& event)
{
    std::scoped_lock lock{mutex_};

    auto authInfo = authInfoFromProto(event);

    YIO_LOG_INFO("Change token: " << authInfo);

    auto currentUsersAuthInfo = *allUsersInfo_.value();
    auto foundIt = findUser(currentUsersAuthInfo, authInfo);

    if (foundIt != currentUsersAuthInfo.end()) {
        *foundIt = authInfo;
    } else {
        currentUsersAuthInfo.push_back(authInfo);
    }

    if (authInfo.passportUid == ownerInfo_.value()->passportUid) {
        updateOwnerInfo(std::move(authInfo));
    }

    updateAllUsers("ChangeToken", std::move(currentUsersAuthInfo));
}

void AuthProvider::onDeleteUser(const proto::AuthEvent& event)
{
    std::scoped_lock lock{mutex_};

    auto authInfo = authInfoFromProto(event);

    YIO_LOG_INFO("Delete user: " << authInfo);

    auto currentUsersAuthInfo = *allUsersInfo_.value();
    auto foundIt = findUser(currentUsersAuthInfo, authInfo);

    if (foundIt != currentUsersAuthInfo.end()) {
        if (authInfo.passportUid == ownerInfo_.value()->passportUid) {
            updateOwnerInfo(AuthInfo2{});
        }
        currentUsersAuthInfo.erase(foundIt);

        updateAllUsers("DeleteUser", std::move(currentUsersAuthInfo));
    }
}

void AuthProvider::onUserAccountInfoChanged(const std::string& authToken, const std::string& passportUid)
{
    std::scoped_lock lock(mutex_);

    auto authInfo = AuthInfo2{
        .source = AuthInfo2::Source::SDK,
        .authToken = authToken,
        .passportUid = passportUid,
        .userType = UserType::OWNER,
        .tag = makeTag()};

    changeUser(std::move(authInfo), "SDK AuthInfo changed");
}

void AuthProvider::changeUser(AuthInfo2 authInfo, std::string_view reason)
{
    YIO_LOG_INFO("Change user: " << authInfo);

    auto currentUsersAuthInfo = *allUsersInfo_.value();
    auto foundIt = findUser(currentUsersAuthInfo, authInfo);

    if (foundIt != currentUsersAuthInfo.end()) {
        *foundIt = authInfo;
    } else {
        currentUsersAuthInfo.push_back(authInfo);
    }

    updateOwnerInfo(std::move(authInfo));

    updateAllUsers(reason, std::move(currentUsersAuthInfo));
}

void AuthProvider::updateOwnerInfo(AuthInfo2 authInfo)
{
    if (authInfo.source == AuthInfo2::Source::SDK) {
        if (!sdkAuthInfo_.has_value() || *sdkAuthInfo_ != authInfo) {
            sdkAuthInfo_ = authInfo;
        }
    } else {
        if (!quasarAuthInfo_.has_value() || *quasarAuthInfo_ != authInfo) {
            quasarAuthInfo_ = authInfo;
        }
    }

    if (*ownerInfo_.value() != authInfo) {
        ownerInfo_ = std::make_shared<const AuthInfo2>(std::move(authInfo));
    }
}

void AuthProvider::updateAllUsers(std::string_view reason, UsersAuthInfo usersAuthInfo)
{
    YIO_LOG_INFO("Update users list: " << reason << ", " << usersAuthInfo);

    const auto& current = *allUsersInfo_.value();
    if (!std::equal(usersAuthInfo.cbegin(), usersAuthInfo.cend(), current.cbegin(), current.cend(),
                    [](const auto& newAI, const auto& oldAI) {
                        return newAI != oldAI;
                    }))
    {
        allUsersInfo_ = std::make_shared<const UsersAuthInfo>(std::move(usersAuthInfo));
    }
}
