#pragma once

#include <yandex_io/libs/signals/i_live_data.h>

#include <chrono>
#include <vector>

namespace quasar {

    enum class UserType {
        OWNER,
        GUEST,
    };

    struct AuthInfo2 {
        enum class Source {
            UNDEFINED,
            AUTHD,
            SDK,
        };
        Source source{Source::UNDEFINED};
        std::string authToken;
        std::string passportUid;
        UserType userType;

        /*
         * Some random number associated with this authorization attempt.
         * It is required by ActivationStateReceiver clients to be able to differentiate between re-registrations on our backend (see ALICE-7910).
         * It's not persisted anywhere and it will change in 2 cases:
         * 1. New reconfiguration attempt even with the same uid and token
         * 2. Device or daemons restart
         */
        int64_t tag{0};

        bool operator==(const AuthInfo2& other) const;
        bool operator!=(const AuthInfo2& other) const;
        bool isAuthorized() const;
        bool isSameAuth(const AuthInfo2& other) const;
    };

    class IAuthProvider {
    public:
        virtual ~IAuthProvider() = default;

        using IAuthInfo = ILiveData<std::shared_ptr<const AuthInfo2>>;
        virtual IAuthInfo& ownerAuthInfo() = 0;

        using UsersAuthInfo = std::vector<AuthInfo2>;
        using IUsersAuthInfo = ILiveData<std::shared_ptr<const UsersAuthInfo>>;
        virtual IUsersAuthInfo& usersAuthInfo() = 0;

        struct AddUserResponse {
            enum class Status {
                OK = 0,
                NO_INTERNET = 4,
                CODE_EXPIRED = 5,
                CRYPTO_ERROR = 6,
                INVALID_TOKEN = 7,
                TIMEOUT = 8,
                UNDEFINED = 1023
            };

            Status status{Status::UNDEFINED};
            std::string authToken;
            std::string xToken;
            int64_t id{0};
            int64_t tag{0};

            std::string statusName() const;
            static std::string statusName(Status status);
        };

        struct DeleteUserResponse {
            enum class Status {
                UNDEFINED = 0,
                OK = 1,
                WRONG_USER = 2,
                TIMEOUT = 3,
            };

            Status status{Status::UNDEFINED};

            std::string statusName() const;
            static std::string statusName(Status status);
        };

        struct ChangeUserResponse {
            enum class Status {
                UNDEFINED = 0,
                OK = 1,
                TIMEOUT = 2,
            };

            Status status{Status::UNDEFINED};

            std::string statusName() const;
            static std::string statusName(Status /*status*/);
        };

        virtual AddUserResponse addUser(const std::string& authCode, UserType userType, bool withXToken, std::chrono::milliseconds timeout) = 0;
        virtual DeleteUserResponse deleteUser(int64_t id, std::chrono::milliseconds timeout) = 0;

        virtual ChangeUserResponse changeUser(int64_t id, std::chrono::milliseconds timeout) = 0;
        virtual void requestAuthTokenUpdate(std::string_view source) = 0;
    };

} // namespace quasar
