#pragma once
#include <yandex_io/interfaces/auth/i_auth_provider.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/signals/live_data.h>

#include <string_view>

namespace quasar::mock {

    class AuthProvider: public IAuthProvider {
    public:
        AuthProvider()
            : owner_(std::make_shared<AuthInfo2>())
            , usersAuthInfo_(std::make_shared<UsersAuthInfo>())
        {
        }

        void setAddUser(std::function<AddUserResponse(const std::string&, UserType userType, bool withXToken, std::chrono::milliseconds)> f)
        {
            addUser_ = std::move(f);
        }

        void setDeleteUser(std::function<DeleteUserResponse(int64_t id, std::chrono::milliseconds)> f)
        {
            deleteUser_ = std::move(f);
        }

        void setChangeUser(std::function<ChangeUserResponse(int64_t, std::chrono::milliseconds)> f)
        {
            changeUser_ = std::move(f);
        }

        void setRequestAuthTokenUpdate(std::function<void(const std::string&)> f)
        {
            requestAuthTokenUpdate_ = std::move(f);
        }

    public:
        IAuthInfo& ownerAuthInfo() override {
            return owner_;
        }

        IUsersAuthInfo& usersAuthInfo() override {
            return usersAuthInfo_;
        }

        AddUserResponse addUser(const std::string& authCode, UserType userType, bool withXToken, std::chrono::milliseconds timeout) override {
            YIO_LOG_INFO("Call mock::AuthProvider.addOwner: authCode=" << authCode
                                                                       << ", userType=" << (userType == UserType::OWNER ? "OWNER" : "GUEST")
                                                                       << ", withXToken=" << withXToken << ", timeout=" << timeout.count());
            if (!addUser_) {
                YIO_LOG_WARN("Unexpected call of AuthProvider.addOwner");
                abort();
            }
            return addUser_(authCode, userType, withXToken, timeout);
        }

        DeleteUserResponse deleteUser(int64_t id, std::chrono::milliseconds timeout) override {
            YIO_LOG_INFO("Call mock::AuthProvider.deleteUser: id=" << id << ", timeout=" << timeout.count());
            if (!deleteUser_) {
                YIO_LOG_WARN("Unexpected call of AuthProvider.deleteUser");
                abort();
            }
            return deleteUser_(id, timeout);
        }

        ChangeUserResponse changeUser(int64_t id, std::chrono::milliseconds timeout) override {
            YIO_LOG_INFO("Call mock::AuthProvider.changeUser: id=" << id << ", timeout=" << timeout.count());
            if (!changeUser_) {
                YIO_LOG_WARN("Unexpected call of AuthProvider.changeUser");
                abort();
            }
            return changeUser_(id, timeout);
        }

        void requestAuthTokenUpdate(std::string_view source) override {
            YIO_LOG_INFO("Call mock::AuthProvider.requestAuthTokenUpdate " << source);
            if (!requestAuthTokenUpdate_) {
                YIO_LOG_WARN("Unexpected call of AuthProvider.requestAuthTokenUpdate");
                abort();
            }
            return requestAuthTokenUpdate_(owner_.value()->authToken);
        }

        void setOwner(AuthInfo2 authInfo) {
            authInfo.userType = UserType::OWNER;
            owner_ = std::make_shared<const AuthInfo2>(authInfo);
            addAuthInfo(std::move(authInfo));
        }

        void addGuest(AuthInfo2 authInfo) {
            authInfo.userType = UserType::GUEST;
            addAuthInfo(std::move(authInfo));
        }

        void clearGuests() {
            auto withoutGuests = UsersAuthInfo{};
            withoutGuests.push_back(*owner_.value());
            usersAuthInfo_ = std::make_shared<const UsersAuthInfo>(std::move(withoutGuests));
        }

        void clearAuthInfo() {
            usersAuthInfo_ = IUsersAuthInfo::ValueType{};
        }

    private:
        void addAuthInfo(AuthInfo2 authInfo) {
            UsersAuthInfo current = *usersAuthInfo_.value();
            current.push_back(std::move(authInfo));
            usersAuthInfo_ = std::make_shared<const UsersAuthInfo>(std::move(current));
        }

        LiveData<IAuthInfo> owner_;
        LiveData<IUsersAuthInfo> usersAuthInfo_;
        std::function<AddUserResponse(const std::string&, UserType, bool, std::chrono::milliseconds)> addUser_;
        std::function<DeleteUserResponse(int64_t, std::chrono::milliseconds)> deleteUser_;
        std::function<ChangeUserResponse(int64_t, std::chrono::milliseconds)> changeUser_;
        std::function<void(const std::string& token)> requestAuthTokenUpdate_;
    };

} // namespace quasar::mock
