#include "i_clock_tower_provider.h"
#include <yandex_io/libs/base/utils.h>

using namespace quasar;

namespace {
    class NullClock: public IClock {
    public:
        const std::string& deviceId() const noexcept override {
            return emptyString();
        }
        const std::string& host() const noexcept override {
            return emptyString();
        }
        int port() const noexcept override {
            return 0;
        }
        const std::string& clockId() const noexcept override {
            return emptyString();
        }
        std::chrono::nanoseconds now() const noexcept override {
            return std::chrono::steady_clock::now().time_since_epoch();
        }
        std::chrono::nanoseconds diff() const noexcept override {
            return std::chrono::nanoseconds{};
        }
        SyncLevel syncLevel() const noexcept override {
            return SyncLevel::NONE;
        }
        bool expired() const noexcept override {
            return true;
        }

    private:
        static const std::string& emptyString() {
            static std::string instance;
            return instance;
        };
    };
} // namespace

std::string IClock::peer() const {
    return makeString(host(), ":", port());
}

std::string IClock::syncLevelAsText() const {
    switch (syncLevel()) {
        case SyncLevel::NONE:
            return "NONE";
        case SyncLevel::WEAK:
            return "WEAK";
        case SyncLevel::STRONG:
            return "STRONG";
    }
    throw std::runtime_error("Unexpected sync level value");
}

bool ClockTowerState::operator==(const ClockTowerState& other) const {
    return localClock == other.localClock &&
           remoteClocks == other.remoteClocks;
}

bool ClockTowerState::operator!=(const ClockTowerState& other) const {
    return !(*this == other);
}

std::shared_ptr<const IClock> ClockTowerState::findClockById(std::string_view clockId) const {
    if (localClock && localClock->clockId() == clockId) {
        return localClock;
    }
    for (const auto& [_, clock] : remoteClocks) {
        if (clock->clockId() == clockId) {
            return clock;
        }
    }
    return nullptr;
}

std::shared_ptr<const IClock> ClockTowerState::findClockByDeviceId(std::string_view deviceId) const {
    if (localClock && localClock->deviceId() == deviceId) {
        return localClock;
    }
    for (const auto& [_, clock] : remoteClocks) {
        if (clock->deviceId() == deviceId) {
            return clock;
        }
    }
    return nullptr;
}

std::shared_ptr<const ClockTowerState> ClockTowerState::createDefault() {
    static const auto instance = [] {
        auto state = std::make_shared<ClockTowerState>();
        state->localClock = std::make_shared<NullClock>();
        return state;
    }();
    return instance;
}