#include "device_state_provider.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

namespace {

    DeviceState::Configuration convertEnum(proto::ConfigurationState configurationState)
    {
        switch (configurationState) {
            case proto::ConfigurationState::UNKNOWN_STATE:
                return DeviceState::Configuration::UNDEFINED;
            case proto::ConfigurationState::CONFIGURING:
                return DeviceState::Configuration::CONFIGURING;
            case proto::ConfigurationState::CONFIGURED:
                return DeviceState::Configuration::CONFIGURED;
        }
        throw std::logic_error("Unknown Configuration State: " + std::to_string(configurationState));
    }

    NetworkStatus::Type convertEnum(proto::ConnectionType connectionType)
    {
        switch (connectionType) {
            case proto::CONNECTION_TYPE_UNKNOWN:
                return NetworkStatus::Type::UNDEFINED;
            case proto::CONNECTION_TYPE_WIFI:
                return NetworkStatus::Type::WIFI;
            case proto::CONNECTION_TYPE_ETHERNET:
                return NetworkStatus::Type::ETHERNET;
        }
        throw std::logic_error("Unknown ConnectionType " + proto::ConnectionType_Name(connectionType));
    }

    NetworkStatus::Status convertEnum(proto::NetworkStatus::Status status)
    {
        switch (status) {
            case proto::NetworkStatus::NOT_CONNECTED:
                return NetworkStatus::Status::NOT_CONNECTED;
            case proto::NetworkStatus::CONNECTING:
                return NetworkStatus::Status::CONNECTING;
            case proto::NetworkStatus::CONNECTED_NO_INTERNET:
                return NetworkStatus::Status::CONNECTED_NO_INTERNET;
            case proto::NetworkStatus::CONNECTED:
                return NetworkStatus::Status::CONNECTED;
            case proto::NetworkStatus::NOT_CHOSEN:
                return NetworkStatus::Status::NOT_CHOSEN;
        }
        throw std::logic_error("Unknown NetworkType " + proto::NetworkStatus::Status_Name(status));
    }

} // namespace

DeviceStateProvider::DeviceStateProvider(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : DeviceStateProvider(
          ipcFactory->createIpcConnector("firstrund"),
          ipcFactory->createIpcConnector("networkd"),
          ipcFactory->createIpcConnector("updatesd"))
{
}

DeviceStateProvider::DeviceStateProvider(
    std::shared_ptr<ipc::IConnector> firstrundConnector,
    std::shared_ptr<ipc::IConnector> networkConnector,
    std::shared_ptr<ipc::IConnector> updatedConnector)
    : firstrundConnector_(std::move(firstrundConnector))
    , networkConnector_(std::move(networkConnector))
    , updatedConnector_(std::move(updatedConnector))
    , configurationChangedSignal_([this](bool /*onConnected*/) { return std::make_tuple(deviceState_.value()); }, lifetime_)
    , networkStatusChangedSignal_([this](bool /*onConnected*/) { return std::make_tuple(deviceState_.value()); }, lifetime_)
    , updateChangedSignal_([this](bool /*onConnected*/) { return std::make_tuple(deviceState_.value()); }, lifetime_)
    , deviceState_(std::make_shared<DeviceState>())
{
    if (firstrundConnector_) {
        firstrundConnector_->setMessageHandler(makeSafeCallback(
            [this](const auto& message) {
                onQuasarMessage(message);
            }, lifetime_));
        firstrundConnector_->connectToService();
    }
    if (networkConnector_) {
        networkConnector_->setMessageHandler(makeSafeCallback(
            [this](const auto& message) {
                onQuasarMessage(message);
            }, lifetime_));
        networkConnector_->connectToService();
    }
    if (updatedConnector_) {
        updatedConnector_->setMessageHandler(makeSafeCallback(
            [this](const auto& message) {
                onQuasarMessage(message);
            }, lifetime_));
        updatedConnector_->connectToService();
    }
}

DeviceStateProvider::~DeviceStateProvider()
{
    lifetime_.die();
}

DeviceStateProvider::IDeviceState& DeviceStateProvider::deviceState()
{
    return deviceState_;
}

DeviceStateProvider::IConfigurationChangedSignal& DeviceStateProvider::configurationChangedSignal()
{
    return configurationChangedSignal_;
}

DeviceStateProvider::INetworkStatusChangedSignal& DeviceStateProvider::networkStatusChangedSignal()
{
    return networkStatusChangedSignal_;
}

DeviceStateProvider::IUpdateChangedSignal& DeviceStateProvider::updateChangedSignal() {
    return updateChangedSignal_;
}

void DeviceStateProvider::onQuasarMessage(const ipc::SharedMessage& message)
{
    std::lock_guard ldLock(deviceState_); // signal about changes after release mutex_
    std::unique_lock mxLock(mutex_);

    auto oldDeviceState = deviceState_.value();
    auto newDeviceState = std::make_shared<DeviceState>(*oldDeviceState);
    std::vector<std::function<void()>> signals;

    if (message->has_configuration_state()) {
        auto configuration = convertEnum(message->configuration_state());
        if (oldDeviceState->configuration != configuration) {
            newDeviceState->configuration = configuration;
            signals.push_back([&] { configurationChangedSignal_.emit(); });
        }
    }

    if (message->has_network_status()) {
        auto type = convertEnum(message->network_status().type());
        auto status = (type != NetworkStatus::Type::UNDEFINED ? convertEnum(message->network_status().status()) : NetworkStatus::Status::NOT_CONNECTED);
        if (oldDeviceState->networkStatus.type != type || oldDeviceState->networkStatus.status != status) {
            newDeviceState->networkStatus.type = type;
            newDeviceState->networkStatus.status = status;
            signals.push_back([&] { networkStatusChangedSignal_.emit(); });
        }
    }

    if (message->has_update_state()) {
        auto update = DeviceState::Update::UNDEFINED;
        if (message->update_state().state() != proto::UpdateState::NONE && message->update_state().is_critical()) {
            update = DeviceState::Update::HAS_CRITICAL;
        } else {
            update = DeviceState::Update::NO_CRITICAL;
        }

        if (oldDeviceState->update != update) {
            newDeviceState->update = update;
            signals.push_back([&] { updateChangedSignal_.emit(); });
        }
    }
    mxLock.unlock();
    if (!signals.empty()) {
        deviceState_ = newDeviceState;
        for (auto& s : signals) {
            s();
        }
    }
}
