#include "glagol_cluster_provider.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/signals/live_data.h>
#include <yandex_io/protos/quasar_proto.pb.h>

namespace quasar {

    GlagolClusterProvider::GlagolClusterProvider(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
        : GlagolClusterProvider(ipcFactory->createIpcConnector("glagold"))
    {
    }

    GlagolClusterProvider::GlagolClusterProvider(std::shared_ptr<ipc::IConnector> glagoldConnector)
        : glagoldConnector_(std::move(glagoldConnector))
        , deviceList_(std::make_shared<const std::vector<std::string>>())
    {
        glagoldConnector_->setConnectHandler(
            makeSafeCallback([] {
                YIO_LOG_INFO("GlagolClusterProvider: Connection established.");
            }, lifetime_));
        glagoldConnector_->setMessageHandler(
            makeSafeCallback([this](const auto& quasarMessage) {
                if (quasarMessage->has_glagold_state()) {
                    auto oldDeviceList = deviceList_.value();
                    const auto& mDeviceList = quasarMessage->glagold_state().cluster_device_list();
                    if (!std::equal(oldDeviceList->begin(), oldDeviceList->end(), mDeviceList.begin(), mDeviceList.end())) {
                        auto newDeviceList = std::make_shared<std::vector<std::string>>();
                        newDeviceList->reserve(mDeviceList.size());
                        for (const auto& deviceId : mDeviceList) {
                            newDeviceList->push_back(std::move(deviceId));
                        }
                        deviceList_ = newDeviceList;
                    }

                    auto oldPublicIp = publicIp_.value();
                    if (oldPublicIp != quasarMessage->glagold_state().public_ip()) {
                        publicIp_ = quasarMessage->glagold_state().public_ip();
                    }
                }
            }, lifetime_));
        glagoldConnector_->connectToService();
    }

    void GlagolClusterProvider::send(Target target, std::string serviceName, const quasar::ipc::SharedMessage& message)
    {
        proto::QuasarMessage gcMessage;
        switch (target) {
            case Target::ALL:
                gcMessage.mutable_glagol_cluster_message()->mutable_target_device_all();
                gcMessage.mutable_glagol_cluster_message()->set_service_name(std::move(serviceName));
                gcMessage.mutable_glagol_cluster_message()->set_quasar_message_base64(encodeQuasarMessage(message));
                glagoldConnector_->sendMessage(std::move(gcMessage));
                return;
            case Target::REMOTE:
                gcMessage.mutable_glagol_cluster_message()->mutable_target_device_remote();
                gcMessage.mutable_glagol_cluster_message()->set_service_name(std::move(serviceName));
                gcMessage.mutable_glagol_cluster_message()->set_quasar_message_base64(encodeQuasarMessage(message));
                glagoldConnector_->sendMessage(std::move(gcMessage));
                return;
        }
        throw std::runtime_error("Unexpected target");
    }

    void GlagolClusterProvider::send(std::vector<std::string> targetDeviceIds, std::string serviceName, const quasar::ipc::SharedMessage& message)
    {
        proto::QuasarMessage gcMessage;
        for (auto& targetDeviceId : targetDeviceIds) {
            gcMessage.mutable_glagol_cluster_message()->add_target_device_ids(std::move(targetDeviceId));
        }
        gcMessage.mutable_glagol_cluster_message()->set_service_name(std::move(serviceName));
        gcMessage.mutable_glagol_cluster_message()->set_quasar_message_base64(encodeQuasarMessage(message));
        glagoldConnector_->sendMessage(std::move(gcMessage));
    }

    GlagolClusterProvider::IDeviceList& GlagolClusterProvider::deviceList()
    {
        return deviceList_;
    }

    GlagolClusterProvider::IPublicIp& GlagolClusterProvider::publicIp() {
        return publicIp_;
    }

    std::string GlagolClusterProvider::encodeQuasarMessage(const quasar::ipc::SharedMessage& message)
    {
        TString binaryMessage;
        Y_PROTOBUF_SUPPRESS_NODISCARD message->SerializeToString(&binaryMessage);
        return base64Encode(binaryMessage.c_str(), binaryMessage.length());
    }

} // namespace quasar
