#include "multiroom_provider.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/json.h>
#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

namespace {

    MultiroomState::Audio::State convertEnum(proto::AudioClientState state)
    {
        switch (state) {
            case proto::AudioClientState::IDLE:
                return MultiroomState::Audio::State::IDLE;
            case proto::AudioClientState::BUFFERING:
                return MultiroomState::Audio::State::BUFFERING;
            case proto::AudioClientState::PAUSED:
                return MultiroomState::Audio::State::PAUSED;
            case proto::AudioClientState::PLAYING:
                return MultiroomState::Audio::State::PLAYING;
            case proto::AudioClientState::STOPPED:
                return MultiroomState::Audio::State::STOPPED;
            case proto::AudioClientState::FINISHED:
                return MultiroomState::Audio::State::FINISHED;
            case proto::AudioClientState::FAILED:
                return MultiroomState::Audio::State::FAILED;
        }

        throw std::runtime_error("Unexpected proto::AudioClientState == " + proto::AudioClientState_Name(state));
    }

    MultiroomState::PlayingState convertEnum(proto::MultiroomBroadcast::State state)
    {
        switch (state) {
            case proto::MultiroomBroadcast::NONE:
                return MultiroomState::PlayingState::NONE;
            case proto::MultiroomBroadcast::PAUSED:
                return MultiroomState::PlayingState::PAUSED;
            case proto::MultiroomBroadcast::PLAYING:
                return MultiroomState::PlayingState::PLAYING;
        }
        throw std::runtime_error("Unexpected proto::MultiroomBroadcast::State == " + proto::MultiroomBroadcast::State_Name(state));
    }

    MultiroomState::Mode convertEnum(proto::MultiroomState::Mode mode)
    {
        switch (mode) {
            case proto::MultiroomState::NONE:
                return MultiroomState::Mode::NONE;
            case proto::MultiroomState::MASTER:
                return MultiroomState::Mode::MASTER;
            case proto::MultiroomState::SLAVE:
                return MultiroomState::Mode::SLAVE;
        }
        throw std::runtime_error("Unexpected proto::MultiroomState::Mode == " + proto::MultiroomState::Mode_Name(mode));
    }

    MultiroomState::SyncLevel convertEnum(proto::MultiroomState::SlaveSyncLevel syncLevel)
    {
        switch (syncLevel) {
            case proto::MultiroomState::NOSYNC:
                return MultiroomState::SyncLevel::NOSYNC;
            case proto::MultiroomState::WEAK:
                return MultiroomState::SyncLevel::WEAK;
            case proto::MultiroomState::STRONG:
                return MultiroomState::SyncLevel::STRONG;
        }
        throw std::runtime_error("Unexpected proto::MultiroomState::SlaveSyncLevel == " + proto::MultiroomState::SlaveSyncLevel_Name(syncLevel));
    }

    template <class T>
    bool compare(const std::vector<MultiroomState::Peer>& peers, const T& protoPeers)
    {
        if ((int)peers.size() != protoPeers.size()) {
            return false;
        }

        for (unsigned i = 0; i < peers.size(); ++i) {
            if (peers[i].deviceId != protoPeers[i].device_id() || peers[i].ipAddress != protoPeers[i].ip_address()) {
                return false;
            }
        }
        return true;
    }

    template <class T>
    std::vector<std::string> makeStringVector(const T& repeatedString)
    {
        std::vector<std::string> result;
        result.reserve(repeatedString.size());

        for (auto& p : repeatedString) {
            result.push_back(p);
        }
        return result;
    }

    template <class T>
    std::vector<MultiroomState::Peer> makeVector(const T& protoPeers)
    {
        std::vector<MultiroomState::Peer> result;
        result.reserve(protoPeers.size());

        for (auto& p : protoPeers) {
            result.push_back(
                MultiroomState::Peer{
                    .deviceId = p.device_id(),
                    .ipAddress = p.ip_address(),
                });
        }
        return result;
    }

    std::map<std::string, std::string> makeScenarioMeta(std::string_view context)
    {
        std::map<std::string, std::string> scenarioMeta;
        if (auto optJson = tryParseJson(context)) {
            const auto& json = *optJson;
            if (json.isMember("scenario_meta")) {
                const auto& jsm = json["scenario_meta"];
                if (jsm.isArray()) {
                    for (const auto& item : jsm) {
                        auto key = tryGetString(item, "key");
                        if (!key.empty()) {
                            scenarioMeta[std::move(key)] = tryGetString(item, "value");
                        }
                    }
                } else if (jsm.isObject()) {
                    for (const auto& key : jsm.getMemberNames()) {
                        scenarioMeta[key] = jsm[key].asString();
                    }
                }
            }
        }
        return scenarioMeta;
    }

} // namespace

MultiroomProvider::MultiroomProvider(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : MultiroomProvider(std::move(device), ipcFactory->createIpcConnector("multiroomd"), std::make_shared<NamedCallbackQueue>("MultiroomProvider"))
{
}

MultiroomProvider::MultiroomProvider(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IConnector> multiroomConnector, std::shared_ptr<ICallbackQueue> lifecycle)
    : device_(std::move(device))
    , lifecycle_(std::move(lifecycle))
    , multiroomConnector_(std::move(multiroomConnector))
    , multiroomState_(std::make_shared<MultiroomState>())
{
    Y_VERIFY(device_);
    Y_VERIFY(multiroomConnector_);
    Y_VERIFY(lifecycle_);

    lifecycle_->add(
        [this]() {
            multiroomConnector_->setConnectHandler(makeSafeCallback(
                [this] {
                    YIO_LOG_DEBUG("MultiroomProvider this=" << this << " connected to service");
                }, lifetime_));
            multiroomConnector_->setMessageHandler(makeSafeCallback(
                [this](const auto& message) {
                    lifecycle_->add(
                        [this, message]() mutable {
                            onQuasarMessage(message);
                        }, lifetime_);
                }, lifetime_));
            auto res = multiroomConnector_->tryConnectToService();
            if (res) {
                res = multiroomConnector_->waitUntilConnected(std::chrono::seconds(3));
            }
        }, lifetime_);
    YIO_LOG_DEBUG("Create MultiroomProvider this=" << this);
}

MultiroomProvider::~MultiroomProvider()
{
    YIO_LOG_DEBUG("Destroy MultiroomProvider this=" << this);
    lifetime_.die();
}

MultiroomProvider::IMultiroomState& MultiroomProvider::multiroomState()
{
    return multiroomState_;
}

void MultiroomProvider::onQuasarMessage(const ipc::SharedMessage& message)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (message->has_multiroom_state()) {
        std::lock_guard ldLock(multiroomState_);

        auto state = multiroomState_.value();
        auto broadcast = (state ? state->broadcast : nullptr);
        auto music = (broadcast ? broadcast->music : nullptr);
        auto audio = (broadcast ? broadcast->audio : nullptr);

        const auto protoState = &message->multiroom_state();
        const auto protoBroadcast = (protoState && protoState->has_multiroom_broadcast() ? &protoState->multiroom_broadcast() : nullptr);
        const auto protoParams = (protoBroadcast && protoBroadcast->has_multiroom_params() ? &protoBroadcast->multiroom_params() : nullptr);
        const auto protoMusic = (protoParams && protoParams->has_music_params() ? &protoParams->music_params() : nullptr);
        const auto protoAudio = (protoParams && protoParams->has_audio_params() ? &protoParams->audio_params() : nullptr);

        bool changed = false;

        if ((audio && !protoAudio) ||
            (!audio && protoAudio) ||
            (audio && protoAudio && (audio->offsetMs != protoAudio->audio().position_sec() * 1000 || audio->playedMs != protoAudio->audio().played_sec() * 1000 || audio->durationMs != protoAudio->audio().duration_sec() * 1000 || audio->lastPlayTimestamp != protoAudio->last_play_timestamp() || audio->lastStopTimestamp != protoAudio->last_stop_timestamp() || audio->state != convertEnum(protoAudio->state()) || audio->streamId != protoAudio->audio().id() || audio->streamType != protoAudio->audio().type() || audio->title != protoAudio->audio().metadata().title() || audio->subtitle != protoAudio->audio().metadata().subtitle()))) {
            if (protoAudio) {
                audio = std::make_shared<MultiroomState::Audio>(
                    MultiroomState::Audio{
                        .offsetMs = protoAudio->audio().position_sec() * 1000,
                        .playedMs = protoAudio->audio().played_sec() * 1000,
                        .durationMs = protoAudio->audio().duration_sec() * 1000,
                        .lastPlayTimestamp = protoAudio->last_play_timestamp(),
                        .lastStopTimestamp = protoAudio->last_stop_timestamp(),
                        .state = convertEnum(protoAudio->state()),
                        .streamId = protoAudio->audio().id(),
                        .streamType = protoAudio->audio().type(),
                        .title = protoAudio->audio().metadata().title(),
                        .subtitle = protoAudio->audio().metadata().subtitle(),
                        .scenarioMeta = makeScenarioMeta(protoAudio->audio().context()),
                    });
            } else {
                changed = true;
                audio = nullptr;
            }
        }

        if ((music && !protoMusic) ||
            (!music && protoMusic) ||
            (music && protoMusic && (music->trackId != protoMusic->current_track_id() || music->trackInfoJson != protoMusic->json_track_info() || music->uid != protoMusic->uid() || music->sessionId != protoMusic->session_id() || music->timestampMs != protoMusic->timestamp_ms() || music->paused != protoMusic->is_paused()))) {
            if (protoMusic) {
                changed = true;
                music = std::make_shared<MultiroomState::Music>(
                    MultiroomState::Music{
                        .trackId = protoMusic->current_track_id(),
                        .trackInfoJson = protoMusic->json_track_info(),
                        .uid = protoMusic->uid(),
                        .sessionId = protoMusic->session_id(),
                        .timestampMs = protoMusic->timestamp_ms(),
                        .paused = protoMusic->is_paused(),
                    });
            } else if (music) {
                changed = true;
                music = nullptr;
            }
        }

        auto multiroomTrackId = (protoBroadcast
                                     ? makeMultiroomTrackId(
                                           protoBroadcast->multiroom_params().url(),
                                           protoBroadcast->multiroom_params().basetime_ns(),
                                           protoBroadcast->multiroom_params().position_ns())
                                     : std::string{});
        if (changed ||
            (broadcast && !protoBroadcast) ||
            (!broadcast && protoBroadcast) ||
            (broadcast && protoBroadcast && (broadcast->masterDeviceId != protoBroadcast->device_id() || broadcast->sessionTimestampMs != protoBroadcast->session_timestamp_ms() || broadcast->multiroomToken != protoBroadcast->multiroom_token() || !std::equal(broadcast->clusterDeviceIds.begin(), broadcast->clusterDeviceIds.end(), protoBroadcast->room_device_ids().begin(), protoBroadcast->room_device_ids().end()) || broadcast->vinsRequestId != protoBroadcast->vins_request_id() || broadcast->playingState != convertEnum(protoBroadcast->state()) || broadcast->multiroomTrackId != multiroomTrackId))) {
            if (protoBroadcast) {
                changed = true;
                broadcast = std::make_shared<MultiroomState::Broadcast>(
                    MultiroomState::Broadcast{
                        .masterDeviceId = protoBroadcast->device_id(),
                        .sessionTimestampMs = protoBroadcast->session_timestamp_ms(),
                        .multiroomToken = protoBroadcast->multiroom_token(),
                        .clusterDeviceIds = makeStringVector(protoBroadcast->room_device_ids()),
                        .vinsRequestId = protoBroadcast->vins_request_id(),
                        .multiroomTrackId = std::move(multiroomTrackId),
                        .playingState = convertEnum(protoBroadcast->state()),
                        .audio = audio,
                        .music = music,
                    });
            } else if (broadcast) {
                changed = true;
                broadcast = nullptr;
            }
        }

        changed = changed || !std::equal(state->dialogDeviceIds.begin(), state->dialogDeviceIds.end(), protoState->dialog_device_ids().begin(), protoState->dialog_device_ids().end());
        changed = changed || !std::equal(state->legacyDeviceIds.begin(), state->legacyDeviceIds.end(), protoState->legacy_device_ids().begin(), protoState->legacy_device_ids().end());
        if (changed ||
            state->deviceId != protoState->device_id() ||
            state->ipAddress != protoState->ip_address() ||
            state->mode != convertEnum(protoState->mode()) ||
            state->slaveSyncLevel != convertEnum(protoState->slave_sync_level()) ||
            state->slaveClockSyncing != protoState->slave_clock_syncing() ||
            !compare(state->peers, protoState->peers())) {
            changed = true;
            state = std::make_shared<MultiroomState>(
                MultiroomState{
                    .deviceId = protoState->device_id(),
                    .ipAddress = protoState->ip_address(),
                    .mode = convertEnum(protoState->mode()),
                    .slaveSyncLevel = convertEnum(protoState->slave_sync_level()),
                    .slaveClockSyncing = protoState->slave_clock_syncing(),
                    .peers = makeVector(protoState->peers()),
                    .dialogDeviceIds = makeStringVector(protoState->dialog_device_ids()),
                    .broadcast = broadcast,
                    .seqnum = protoState->seqnum(),
                    .legacyDeviceIds = makeStringVector(protoState->legacy_device_ids()),
                });
        }

        if (changed) {
            multiroomState_ = state;
        }
    }
}

void MultiroomProvider::sendMultiroomDirective(std::function<void(proto::MultiroomDirective&)> filler) {
    lifecycle_->add(
        [this, filler{std::move(filler)}] {
            proto::QuasarMessage message;
            message.mutable_multiroom_directive()->set_sender_device_id(TString(device_->deviceId()));
            filler(*message.mutable_multiroom_directive());
            if (!message.mutable_multiroom_directive()->has_master_device_id()) {
                auto state = multiroomState_.value();
                if (state->broadcast) {
                    message.mutable_multiroom_directive()->set_master_device_id(TString(state->broadcast->masterDeviceId));
                    message.mutable_multiroom_directive()->set_session_timestamp_ms(state->broadcast->sessionTimestampMs);
                    message.mutable_multiroom_directive()->set_multiroom_token(TString(state->broadcast->multiroomToken));
                } else {
                    message.mutable_multiroom_directive()->set_master_device_id(TString(device_->deviceId()));
                }
            }
            YIO_LOG_INFO("{multiroom} MultiroomProvider send directive: " << convertMessageToDeepJsonString(message));
            multiroomConnector_->sendMessage(std::move(message));
        }, lifetime_);
}