#include "spectrogram_animation_provider.h"

#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/signals/live_data.h>
#include <yandex_io/libs/signals/signal.h>
#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

namespace {

    SpectrogramAnimationState convertSpectrogramAnimationState(const proto::SpectrogramAnimation::State& state)
    {
        return {
            .source =
                (state.has_source() && state.source() == proto::SpectrogramAnimation::State::EXTERNAL
                     ? SpectrogramAnimationState::Source::EXTERNAL
                     : SpectrogramAnimationState::Source::LOCAL),
            .configs = state.configs(),
            .current = state.current(),
            .extraData = state.extra_data(),
        };
    }

} // namespace

SpectrogramAnimationProvider::SpectrogramAnimationProvider(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : SpectrogramAnimationProvider(ipcFactory->createIpcConnector("spectrogram_animation"))
{
}

SpectrogramAnimationProvider::SpectrogramAnimationProvider(std::shared_ptr<ipc::IConnector> connector)
    : connector_(std::move(connector))
    , spectrogramAnimationState_(std::make_shared<SpectrogramAnimationState>())
{
    connector_->setMessageHandler(
        makeSafeCallback([this](const auto& message) {
            if (message->has_spectrogram_animation_message()) {
                const auto& sam = message->spectrogram_animation_message();
                if (sam.has_state()) {
                    auto oldState = spectrogramAnimationState_.value();
                    auto newState = convertSpectrogramAnimationState(sam.state());
                    if (*oldState != newState) {
                        spectrogramAnimationState_ = std::make_shared<SpectrogramAnimationState>(std::move(newState));
                    }
                }
            }
        }, lifetime_));
    connector_->tryConnectToService();
}

ISpectrogramAnimationState& SpectrogramAnimationProvider::spectrogramAnimationState()
{
    return spectrogramAnimationState_;
}

void SpectrogramAnimationProvider::setExternalPresets(const std::string& configs, const std::string& current, const std::string& extraData)
{
    proto::QuasarMessage message;
    message.mutable_spectrogram_animation_message()->mutable_set_external_presets_signal()->set_configs(TString(configs));
    message.mutable_spectrogram_animation_message()->mutable_set_external_presets_signal()->set_current(TString(current));
    message.mutable_spectrogram_animation_message()->mutable_set_external_presets_signal()->set_extra_data(TString(extraData));
    connector_->sendMessage(std::move(message));
}
