#include "stereo_pair_provider.h"

#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/signals/live_data.h>
#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

namespace {

    StereoPairState::Role convertRole(proto::StereoPair::Role role)
    {
        static_assert(static_cast<int>(proto::StereoPair::STANDALONE) == static_cast<int>(StereoPairState::Role::STANDALONE));
        static_assert(static_cast<int>(proto::StereoPair::LEADER) == static_cast<int>(StereoPairState::Role::LEADER));
        static_assert(static_cast<int>(proto::StereoPair::FOLLOWER) == static_cast<int>(StereoPairState::Role::FOLLOWER));

        switch (role) {
            case proto::StereoPair::STANDALONE:
                return StereoPairState::Role::STANDALONE;
            case proto::StereoPair::LEADER:
                return StereoPairState::Role::LEADER;
            case proto::StereoPair::FOLLOWER:
                return StereoPairState::Role::FOLLOWER;
        }
        throw std::runtime_error("Unexpected Role");
    }

    StereoPairState::Channel convertChannel(proto::StereoPair::Channel channel)
    {
        static_assert(static_cast<int>(proto::StereoPair::CH_UNDEFINED) == static_cast<int>(StereoPairState::Channel::UNDEFINED));
        static_assert(static_cast<int>(proto::StereoPair::CH_ALL) == static_cast<int>(StereoPairState::Channel::ALL));
        static_assert(static_cast<int>(proto::StereoPair::CH_RIGHT) == static_cast<int>(StereoPairState::Channel::RIGHT));
        static_assert(static_cast<int>(proto::StereoPair::CH_LEFT) == static_cast<int>(StereoPairState::Channel::LEFT));

        switch (channel) {
            case proto::StereoPair::CH_UNDEFINED:
                return StereoPairState::Channel::UNDEFINED;
            case proto::StereoPair::CH_ALL:
                return StereoPairState::Channel::ALL;
            case proto::StereoPair::CH_RIGHT:
                return StereoPairState::Channel::RIGHT;
            case proto::StereoPair::CH_LEFT:
                return StereoPairState::Channel::LEFT;
        }
        throw std::runtime_error("Unexpected Channel");
    }

    StereoPairState::Connectivity convertConnectivity(proto::StereoPair::Connectivity connectivity)
    {
        static_assert(static_cast<int>(proto::StereoPair::INAPPLICABLE) == static_cast<int>(StereoPairState::Connectivity::INAPPLICABLE));
        static_assert(static_cast<int>(proto::StereoPair::NO_CONNECTION) == static_cast<int>(StereoPairState::Connectivity::NO_CONNECTION));
        static_assert(static_cast<int>(proto::StereoPair::ONEWAY) == static_cast<int>(StereoPairState::Connectivity::ONEWAY));
        static_assert(static_cast<int>(proto::StereoPair::TWOWAY) == static_cast<int>(StereoPairState::Connectivity::TWOWAY));

        switch (connectivity) {
            case proto::StereoPair::INAPPLICABLE:
                return StereoPairState::Connectivity::INAPPLICABLE;
            case proto::StereoPair::NO_CONNECTION:
                return StereoPairState::Connectivity::NO_CONNECTION;
            case proto::StereoPair::ONEWAY:
                return StereoPairState::Connectivity::ONEWAY;
            case proto::StereoPair::TWOWAY:
                return StereoPairState::Connectivity::TWOWAY;
        }
        switch ((int)connectivity) {
            case 2:
                return StereoPairState::Connectivity::ONEWAY; // old value
            case 3:
                return StereoPairState::Connectivity::NO_CONNECTION; // old value
        }
        throw std::runtime_error("Unexpected Connectivity");
    }

    StereoPairState::StereoPlayerStatus convertStereoPlayerStatus(proto::StereoPair::StereoPlayerStatus stereoPlayerStatus)
    {
        static_assert(static_cast<int>(proto::StereoPair::PLAYER_UNDEFINED) == static_cast<int>(StereoPairState::StereoPlayerStatus::UNDEFINED));
        static_assert(static_cast<int>(proto::StereoPair::PLAYER_NO_SYNC) == static_cast<int>(StereoPairState::StereoPlayerStatus::NO_SYNC));
        static_assert(static_cast<int>(proto::StereoPair::PLAYER_PARTNER_NOT_READY) == static_cast<int>(StereoPairState::StereoPlayerStatus::PARTNER_NOT_READY));
        static_assert(static_cast<int>(proto::StereoPair::PLAYER_READY) == static_cast<int>(StereoPairState::StereoPlayerStatus::READY));

        switch (stereoPlayerStatus) {
            case proto::StereoPair::PLAYER_UNDEFINED:
                return StereoPairState::StereoPlayerStatus::UNDEFINED;
            case proto::StereoPair::PLAYER_NO_SYNC:
                return StereoPairState::StereoPlayerStatus::NO_SYNC;
            case proto::StereoPair::PLAYER_PARTNER_NOT_READY:
                return StereoPairState::StereoPlayerStatus::PARTNER_NOT_READY;
            case proto::StereoPair::PLAYER_READY:
                return StereoPairState::StereoPlayerStatus::READY;
        }
        throw std::runtime_error("Unexpected StereoPlayerStatus");
    }

    VolumeManagerState convertVolume(const proto::VolumeManager::State& vms)
    {
        return {
            .platformVolume = vms.platform_volume(),
            .aliceVolume = vms.alice_volume(),
            .isMuted = vms.is_muted(),
            .source = vms.source(),
            .setBtVolume = vms.set_bt_volume(),
        };
    }

    SpectrogramAnimationState convertSpectrogramAnimation(const proto::SpectrogramAnimation::State& state)
    {
        return {
            .source =
                (state.has_source() && state.source() == proto::SpectrogramAnimation::State::EXTERNAL
                     ? SpectrogramAnimationState::Source::EXTERNAL
                     : SpectrogramAnimationState::Source::LOCAL),
            .configs = state.configs(),
            .current = state.current(),
            .extraData = state.extra_data(),
        };
    }

    StereoPairState convertStereoPairState(const proto::StereoPair::State& state)
    {
        return {
            .role = (state.has_role() ? convertRole(state.role()) : StereoPairState::Role::UNDEFINED),
            .channel = convertChannel(state.channel()),
            .partnerDeviceId = state.partner_device_id(),
            .connectivity = (state.has_connectivity() ? convertConnectivity(state.connectivity()) : StereoPairState::Connectivity::UNDEFINED),
            .stereoPlayerStatus = convertStereoPlayerStatus(state.stereo_player_status()),
            .initialPairingTimepoint = std::chrono::system_clock::time_point{} + std::chrono::milliseconds(state.initial_pairing_time_ms()),
            .spectrogramAnimationState = convertSpectrogramAnimation(state.spectrogram()),
            .volumeManagerState = convertVolume(state.volume()),
        };
    }

} // namespace

StereoPairProvider::StereoPairProvider(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : StereoPairProvider(device->deviceId(), ipcFactory->createIpcConnector("stereo_pair"))
{
}

StereoPairProvider::StereoPairProvider(std::string myDeviceId, std::shared_ptr<ipc::IConnector> connector)
    : myDeviceId_(std::move(myDeviceId))
    , connector_(std::move(connector))
    , stereoPairState_(std::make_shared<StereoPairState>())
    , initialPairingSignal_(
          [this](bool /*onConnceted*/) {
              auto s = stereoPairState_.value();
              if (s->initialPairingTimepoint == std::chrono::system_clock::time_point{}) {
                  throw NoDataForSignal();
              }
              return std::make_tuple(
                  std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now() - s->initialPairingTimepoint), s);
          }, lifetime_)
{
    connector_->setConnectHandler(makeSafeCallback([this] { setReady(true); }, lifetime_));
    connector_->setMessageHandler(
        makeSafeCallback([this](const auto& message) {
            if (message->has_stereo_pair_message()) {
                const auto& spm = message->stereo_pair_message();
                if (spm.has_state()) {
                    updateStereoPairState(message->stereo_pair_message().state());
                }

                if (spm.has_user_event_signal() && !spm.user_event_signal().event_id().empty()) {
                    const auto& eventId = spm.user_event_signal().event_id();
                    const auto& payload = spm.user_event_signal().payload();
                    YIO_LOG_INFO("Emit user event signal: id=" << eventId << ", payload=" << payload);
                    userEventSignal_(eventId, tryParseJsonOrEmpty(payload));
                }
            }
        }, lifetime_));
    connector_->connectToService();
}

StereoPairProvider::IStereoPairState& StereoPairProvider::stereoPairState()
{
    return stereoPairState_;
}

StereoPairProvider::IInitialPairingSignal& StereoPairProvider::initialPairingSignal()
{
    return initialPairingSignal_;
}

StereoPairProvider::IUserEventSignal& StereoPairProvider::userEventSignal()
{
    return userEventSignal_;
}

void StereoPairProvider::userEvent(const std::string& id, const Json::Value& payload)
{
    proto::QuasarMessage message;
    auto& userEvent = *message.mutable_stereo_pair_message()->mutable_user_event_request();
    userEvent.set_event_id(TString(id));
    if (payload.isObject()) {
        userEvent.set_payload(jsonToString(payload, true));
    } else {
        userEvent.set_payload("{}");
    }
    connector_->sendMessage(std::move(message));
}

void StereoPairProvider::speakNotReadyNotification(NotReadyReason notReadyReason) {
    proto::QuasarMessage message;
    auto& ntf = *message.mutable_stereo_pair_message()->mutable_speak_not_ready_notification_request();
    switch (notReadyReason) {
        case NotReadyReason::NO_CONNECTION:
            ntf.set_reason(proto::StereoPair::SpeakNotReadyNotificationReqest::NO_CONNECTION);
            break;
        case NotReadyReason::PLAYER_NOT_READY:
            ntf.set_reason(proto::StereoPair::SpeakNotReadyNotificationReqest::PLAYER_NOT_READY);
            break;
    }
    connector_->sendMessage(std::move(message));
}

void StereoPairProvider::overrideChannel(StereoPairState::Channel channel)
{
    proto::QuasarMessage message;
    message.mutable_stereo_pair_message()->mutable_override_channel_request()->set_channel(StereoPairState::channelName(channel));
    connector_->sendMessage(std::move(message));
}

void StereoPairProvider::startConversationOnLeader()
{
    auto message = quasar::ipc::buildMessage([&](auto& msg) {
        msg.mutable_alice_message()->mutable_start_conversation_signal();
    });
    connector_->sendMessage(message);
}

void StereoPairProvider::stopConversationOnLeader()
{
    auto message = quasar::ipc::buildMessage([&](auto& msg) {
        msg.mutable_alice_message()->mutable_stop_conversation_signal();
    });
    connector_->sendMessage(message);
}

void StereoPairProvider::toggleConversationOnLeader()
{
    auto message = quasar::ipc::buildMessage([&](auto& msg) {
        msg.mutable_alice_message()->mutable_toggle_conversation_signal();
    });
    connector_->sendMessage(message);
}

void StereoPairProvider::finishConversationOnLeader()
{
    auto message = quasar::ipc::buildMessage([&](auto& msg) {
        msg.mutable_alice_message()->mutable_finish_conversation_signal();
    });
    connector_->sendMessage(message);
}

void StereoPairProvider::updateStereoPairState(const proto::StereoPair::State& state)
{
    auto oldStereoPairState = stereoPairState_.value();
    auto newStereoPairState = convertStereoPairState(state);

    if (*oldStereoPairState == newStereoPairState) {
        return;
    }

    bool initialPairing = (oldStereoPairState->initialPairingTimepoint < newStereoPairState.initialPairingTimepoint);
    stereoPairState_ = std::make_shared<StereoPairState>(std::move(newStereoPairState));

    if (initialPairing) {
        initialPairingSignal_.emit();
    }
}
