#pragma once

#include <yandex_io/interfaces/base/rpc_interface.h>
#include <yandex_io/interfaces/spectrogram_animation/i_spectrogram_animation_provider.h>
#include <yandex_io/interfaces/volume_manager/i_volume_manager_provider.h>

#include <yandex_io/libs/signals/i_live_data.h>
#include <yandex_io/libs/signals/i_signal.h>

#include <json/json.h>

namespace quasar {

    struct StereoPairState {
        enum class Role {
            UNDEFINED = 0,
            STANDALONE = 1,
            LEADER,
            FOLLOWER,
        };

        enum class Channel {
            UNDEFINED = 0,
            ALL = 1,
            RIGHT = 2,
            LEFT = 3,
        };

        enum class Connectivity {
            UNDEFINED = 0,
            INAPPLICABLE = 1, // for STANDALONE role
            NO_CONNECTION = 4,
            ONEWAY = 5,
            TWOWAY = 6,
        };

        enum class StereoPlayerStatus {
            UNDEFINED = 0,
            NO_SYNC = 1,
            PARTNER_NOT_READY = 2,
            READY = 3,
        };

        Role role{Role::UNDEFINED};
        Channel channel{Channel::UNDEFINED};
        std::string partnerDeviceId;
        Connectivity connectivity{Connectivity::UNDEFINED};
        StereoPlayerStatus stereoPlayerStatus{StereoPlayerStatus::UNDEFINED};
        std::chrono::system_clock::time_point initialPairingTimepoint;
        SpectrogramAnimationState spectrogramAnimationState;
        VolumeManagerState volumeManagerState;

        bool operator==(const StereoPairState& other) const;
        bool operator!=(const StereoPairState& other) const;

        bool isStereoPair() const;
        bool isFollower() const;

        static const char* roleName(Role role);
        static const char* channelName(Channel /*channel*/);
        static const char* connectivityName(Connectivity /*connectivity*/);
        static const char* stereoPlayerStatusName(StereoPlayerStatus /*stereoPlayerStatus*/);
        static Channel parseChannel(std::string_view /*text*/);
    };

    class IStereoPairProvider: public RpcInterface {
    public:
        virtual ~IStereoPairProvider();

        using IStereoPairState = ILiveData<std::shared_ptr<const StereoPairState>>;
        virtual IStereoPairState& stereoPairState() = 0;

        /**
         * @brief InitialPairingSignal is triggered when the follower device is first detected
         */
        using IInitialPairingSignal = ISignal<std::chrono::milliseconds, std::shared_ptr<const StereoPairState>>;
        virtual IInitialPairingSignal& initialPairingSignal() = 0;

        /**
         * @brief IUserEventSignal is triggered when method userEvent called from partner device
         */
        using IUserEventSignal = ISignal<std::string, Json::Value>;
        virtual IUserEventSignal& userEventSignal() = 0;
        virtual void userEvent(const std::string& id, const Json::Value& payload) = 0;

        /**
         * @brief Notify the user that the stereo pair is not ready
         */
        enum class NotReadyReason {
            NO_CONNECTION,
            PLAYER_NOT_READY,
        };
        virtual void speakNotReadyNotification(NotReadyReason) = 0;

        /**
         * @brief Set overrided sound channel for this device. UNDEFINED means reset overrided and use default
         */
        virtual void overrideChannel(StereoPairState::Channel) = 0;

        virtual void startConversationOnLeader() = 0;
        virtual void stopConversationOnLeader() = 0;
        virtual void toggleConversationOnLeader() = 0;
        virtual void finishConversationOnLeader() = 0;
    };

} // namespace quasar
