#include "updates_provider.h"

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>

namespace quasar {

    UpdatesProvider::UpdatesProvider(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
        : UpdatesProvider(ipcFactory->createIpcConnector("updatesd"), std::make_shared<NamedCallbackQueue>("UpdatesProvider"))
    {
    }

    UpdatesProvider::UpdatesProvider(std::shared_ptr<ipc::IConnector> updatesdConnector, std::shared_ptr<ICallbackQueue> callbackQueue)
        : updatesdConnector_(std::move(updatesdConnector))
        , callbackQueue_(std::move(callbackQueue))
        , readyApplyUpdateSignal_([this](bool /*onConnected*/) { return std::make_tuple(updatesState_.value()->readyToApplyUpdateFlag); }, lifetime_)
        , updatesState_(std::make_shared<UpdatesState2>())
    {
        if (updatesdConnector_) {
            updatesdConnector_->setMessageHandler(makeSafeCallback(
                [this](const ipc::SharedMessage& message) {
                    onQuasarMessage(message);
                }, lifetime_, callbackQueue_));
            updatesdConnector_->connectToService();
        }
    }

    UpdatesProvider::~UpdatesProvider()
    {
        lifetime_.die();
    }

    UpdatesProvider::IUpdatesState& UpdatesProvider::updatesState()
    {
        return updatesState_;
    }

    UpdatesProvider::IReadyApplyUpdateSignal& UpdatesProvider::readyApplyUpdateSignal()
    {
        return readyApplyUpdateSignal_;
    }

    void UpdatesProvider::checkUpdates()
    {
        proto::QuasarMessage msg;
        msg.mutable_check_updates();
        updatesdConnector_->sendMessage(std::move(msg));
    }

    void UpdatesProvider::confirmUpdateApply()
    {
        YIO_LOG_INFO("Confirm Update apply");
        proto::QuasarMessage msg;
        msg.mutable_confirm_update_apply();
        updatesdConnector_->sendMessage(std::move(msg));
    }

    void UpdatesProvider::postponeUpdateApply() {
        YIO_LOG_INFO("Postpone update apply");
        proto::QuasarMessage msg;
        msg.mutable_postpone_update_apply();
        updatesdConnector_->sendMessage(std::move(msg));
    }

    void UpdatesProvider::waitUpdateState(std::function<void(UpdatesState2::Critical)> callback, std::chrono::milliseconds timeout)
    {
        if (!callback) {
            return;
        }

        std::lock_guard lock(mutex_);
        awaitingCallbacks_.push_back(AwaitingCallbacks{std::move(callback), std::chrono::steady_clock::now() + timeout});
        callbackQueue_->addDelayed(
            [this]() {
                const auto now = std::chrono::steady_clock::now();
                std::list<AwaitingCallbacks> failedCallbacks_;
                std::unique_lock lock(mutex_);
                for (auto it = awaitingCallbacks_.begin(); it != awaitingCallbacks_.end();) {
                    if (it->expired < now) {
                        failedCallbacks_.push_back(std::move(*it));
                        it = awaitingCallbacks_.erase(it);
                    } else {
                        ++it;
                    }
                }
                lock.unlock();
                for (auto& fc : failedCallbacks_) {
                    fc.callback(UpdatesState2::Critical::UNDEFINED);
                }
            }, timeout, lifetime_);
    }

    void UpdatesProvider::onQuasarMessage(const ipc::SharedMessage& message)
    {
        std::lock_guard ldLock(updatesState_); // signal about changes after release mutex_
        std::unique_lock mxLock(mutex_);

        std::list<AwaitingCallbacks> awaitingCallbacks;
        bool notifyUpdateStatus = false;
        bool normalizeStatuses = false;
        auto currentUpdatesState = *updatesState_.value();
        auto newUpdatesState = currentUpdatesState;
        if (message->has_ready_to_apply_update()) {
            newUpdatesState.readyToApplyUpdateFlag = true;
            normalizeStatuses = true;
        }

        if (message->has_no_critical_update_found()) {
            newUpdatesState.critical = UpdatesState2::Critical::NO;
            notifyUpdateStatus = true;
            normalizeStatuses = true;
        }

        if (message->has_start_critical_update() || message->has_critical_update_progress()) {
            newUpdatesState.critical = UpdatesState2::Critical::YES;
            notifyUpdateStatus = true;
            normalizeStatuses = true;
        }

        if (message->has_update_state()) {
            switch (message->update_state().state()) {
                case proto::UpdateState_State::UpdateState_State_NONE:
                    newUpdatesState.progress = UpdatesState2::Progress::NONE;
                    break;
                case proto::UpdateState_State::UpdateState_State_DOWNLOADING:
                    newUpdatesState.progress = UpdatesState2::Progress::DOWNLOADING;
                    break;
                case proto::UpdateState_State::UpdateState_State_APPLYING:
                    newUpdatesState.progress = UpdatesState2::Progress::APPLYING;
                    break;
            }
            if (message->update_state().state() != proto::UpdateState_State::UpdateState_State_NONE && message->update_state().is_critical()) {
                newUpdatesState.critical = UpdatesState2::Critical::YES;
            } else {
                newUpdatesState.critical = UpdatesState2::Critical::NO;
            }
            newUpdatesState.readyToApplyUpdateFlag = false;
            notifyUpdateStatus = true;
            normalizeStatuses = true;
        }

        if (normalizeStatuses) {
            if (newUpdatesState.progress == UpdatesState2::Progress::UNDEFINED) {
                newUpdatesState.progress = UpdatesState2::Progress::NONE;
            }
            if (newUpdatesState.critical == UpdatesState2::Critical::UNDEFINED) {
                newUpdatesState.critical = UpdatesState2::Critical::NO;
            }
        }

        if (newUpdatesState != currentUpdatesState) {
            updatesState_ = std::make_shared<UpdatesState2>(newUpdatesState);
        }

        if (notifyUpdateStatus) {
            awaitingCallbacks_.swap(awaitingCallbacks);
        }

        mxLock.unlock();

        if (newUpdatesState.readyToApplyUpdateFlag != currentUpdatesState.readyToApplyUpdateFlag) {
            readyApplyUpdateSignal_.emit();
        }

        for (auto& ac : awaitingCallbacks) {
            ac.callback(newUpdatesState.critical);
        }
    }

} // namespace quasar
