#include "user_config_provider.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

namespace {
    const std::shared_ptr<Json::Value> EMPTY_JSON_VALUE = std::make_shared<Json::Value>();
    const std::map<IUserConfigProvider::ConfigScope, std::vector<IUserConfigProvider::ConfigScope>> SIGNAL_SCOPE_MAP =
        {
            {UserConfigProvider::ConfigScope::ACCOUNT, {UserConfigProvider::ConfigScope::ACCOUNT}},
            {UserConfigProvider::ConfigScope::DEVICE, {UserConfigProvider::ConfigScope::DEVICE}},
            {UserConfigProvider::ConfigScope::SYSTEM, {UserConfigProvider::ConfigScope::SYSTEM}},
            {UserConfigProvider::ConfigScope::AUXILIARY, {UserConfigProvider::ConfigScope::AUXILIARY}},
            {UserConfigProvider::ConfigScope::MERGED, {UserConfigProvider::ConfigScope::SYSTEM, UserConfigProvider::ConfigScope::DEVICE, UserConfigProvider::ConfigScope::ACCOUNT}},
    };

    const Json::Value& getScopeJson(IUserConfigProvider::ConfigScope scope, const std::shared_ptr<const UserConfig>& userConfig)
    {
        if (!userConfig) {
            return Json::Value::nullRef;
        }

        switch (scope) {
            case UserConfigProvider::ConfigScope::ACCOUNT:
                return userConfig->account;
            case UserConfigProvider::ConfigScope::DEVICE:
                return userConfig->device;
            case UserConfigProvider::ConfigScope::SYSTEM:
                return userConfig->system;
            case UserConfigProvider::ConfigScope::AUXILIARY:
                return userConfig->auxiliary;
            case UserConfigProvider::ConfigScope::MERGED:
                break;
        }
        return Json::Value::nullRef;
    }

} // namespace

UserConfigProvider::UserConfigProvider(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : UserConfigProvider(ipcFactory->createIpcConnector("syncd"))
{
}

UserConfigProvider::UserConfigProvider(std::shared_ptr<ipc::IConnector> connector)
    : syncdConnector_(std::move(connector))
    , userConfig_(nullptr) // no default config
    , accountDevicesChangedSignal_([this](bool onConnect) {
        auto userConfig = userConfig_.value();
        if (onConnect && (!userConfig || userConfig->auth == UserConfig::Auth::UNDEFINED)) {
            throw NoDataForSignal();
        }
        auto accountDevices = std::shared_ptr<const std::vector<AccountDevice>>(userConfig, &userConfig->accountDevices);
        return std::make_tuple(std::move(accountDevices));
    }, lifetime_)
{
    if (syncdConnector_) {
        syncdConnector_->setMessageHandler(makeSafeCallback([this](const auto& message) { onQuasarMessage(message); }, lifetime_));
        syncdConnector_->connectToService();
    }
}

UserConfigProvider::~UserConfigProvider()
{
    lifetime_.die();
}

UserConfigProvider::IUserConfig& UserConfigProvider::userConfig()
{
    return userConfig_;
}

UserConfigProvider::IJsonChangedSignal& UserConfigProvider::jsonChangedSignal(ConfigScope configScope, const std::string& xpath) {
    std::lock_guard mlock(mutex_);
    auto key = std::make_pair(configScope, xpath);
    auto it = xpathSignals_.find(key);
    if (it != xpathSignals_.end()) {
        return *it->second;
    }

    if (configScope == ConfigScope::MERGED) {
        struct Context {
            std::mutex mutex;
            std::shared_ptr<const Json::Value> json;
        };
        auto context = std::make_shared<Context>();
        auto signal = std::make_shared<SignalExternal<IJsonChangedSignal>>(
            [this, xpath, context](bool onConnect) {
                std::lock_guard lock(context->mutex);
                auto userConfig = userConfig_.value();
                if (!userConfig || userConfig->auth == UserConfig::Auth::UNDEFINED) {
                    context->json = EMPTY_JSON_VALUE;
                    throw NoDataForSignal();
                }
                if (userConfig->auth == UserConfig::Auth::FAILED) {
                    context->json = EMPTY_JSON_VALUE;
                } else if (!onConnect || !context->json) {
                    auto mergedJson = userConfig->merged(xpath);
                    context->json = (mergedJson.isNull() ? EMPTY_JSON_VALUE : std::make_shared<Json::Value>(std::move(mergedJson)));
                }
                return std::make_tuple(context->json);
            }, lifetime_);
        xpathSignals_.emplace(std::move(key), signal);
        return *signal;
    } else {
        auto signal = std::make_shared<SignalExternal<IJsonChangedSignal>>(
            [this, configScope, xpath](bool onConnect) {
                auto userConfig = userConfig_.value();
                if (onConnect && (!userConfig || userConfig->auth == UserConfig::Auth::UNDEFINED)) {
                    throw NoDataForSignal();
                }
                auto pSubValue = findXPath(getScopeJson(configScope, userConfig), xpath);
                auto initSubValue = (pSubValue
                                         ? std::shared_ptr<const Json::Value>(userConfig, pSubValue)
                                         : EMPTY_JSON_VALUE);
                return std::make_tuple(std::move(initSubValue));
            }, lifetime_);
        xpathSignals_.emplace(std::move(key), signal);
        return *signal;
    }
}

UserConfigProvider::IAccountDevicesChangedSignal& UserConfigProvider::accountDevicesChangedSignal() {
    return accountDevicesChangedSignal_;
}

bool UserConfigProvider::setUserConfig(std::shared_ptr<UserConfig> newUserConfig)
{
    auto oldUserConfig = userConfig_.value();
    if (!oldUserConfig || (*oldUserConfig != *newUserConfig)) {
        userConfig_ = newUserConfig;
        if (oldUserConfig) {
            notifyJsonChanges(oldUserConfig, newUserConfig);
        } else {
            notifyAllJsonChanges();
        }
        if (!oldUserConfig || oldUserConfig->accountDevices != newUserConfig->accountDevices) {
            accountDevicesChangedSignal_.emit();
        }
        return true;
    }
    return false;
}

void UserConfigProvider::onQuasarMessage(const ipc::SharedMessage& message)
{
    std::lock_guard userConfigLock(userConfig_);
    if (message->has_auth_failed()) {
        setUserConfig(std::make_shared<UserConfig>(UserConfig{.auth = UserConfig::Auth::FAILED}));
        return;
    }

    if (message->has_user_config_update() ||
        message->has_subscription_state() ||
        message->has_account_devices_list()) {
        auto oldUserConfig = userConfig_.value();
        auto newUserConfig = std::make_shared<UserConfig>(oldUserConfig ? *oldUserConfig : UserConfig{});
        if (message->has_user_config_update()) {
            if (auto optJson = tryParseJson(message->user_config_update().config())) {
                auto& json = *optJson;

                newUserConfig->auth = UserConfig::Auth::SUCCESS;
                newUserConfig->passportUid = message->user_config_update().passport_uid();
                newUserConfig->account = std::move(json["account_config"]);
                newUserConfig->device = std::move(json["device_config"]);
                newUserConfig->system = std::move(json["system_config"]);
                newUserConfig->auxiliary = std::move(json["auxiliary_device_config"]);
            }
        }
        if (message->has_subscription_state()) {
            if (newUserConfig->passportUid != message->subscription_state().passport_uid() ||
                newUserConfig->subscriptionInfo != message->subscription_state().subscription_info()) {
                newUserConfig->auth = UserConfig::Auth::SUCCESS;
                newUserConfig->passportUid = message->subscription_state().passport_uid();
                newUserConfig->subscriptionInfo = message->subscription_state().subscription_info();
                newUserConfig->updateTime = std::chrono::system_clock::time_point{} + std::chrono::seconds{message->subscription_state().last_update_time()};
            }
        }

        if (message->has_account_devices_list()) {
            newUserConfig->accountDevices.clear();
            newUserConfig->accountDevices.reserve(message->account_devices_list().account_devices().size());
            for (const auto& accountDevice : message->account_devices_list().account_devices()) {
                newUserConfig->accountDevices.emplace_back(
                    AccountDevice{
                        .deviceId = accountDevice.id(),
                        .platform = accountDevice.platform(),
                        .name = accountDevice.name(),
                        .serverCertificate = accountDevice.server_certificate(),
                        .serverPrivateKey = accountDevice.server_private_key(),
                    });
            }
        }
        setUserConfig(std::move(newUserConfig));
    }
}

void UserConfigProvider::notifyJsonChanges(const std::shared_ptr<const UserConfig>& oldUserConfig, const std::shared_ptr<const UserConfig>& newUserConfig)
{
    std::vector<std::shared_ptr<SignalExternal<IJsonChangedSignal>>> signals;

    {
        std::lock_guard mlock(mutex_);
        signals.reserve(xpathSignals_.size());
        for (const auto& p : xpathSignals_) {
            const auto& configScope = p.first.first;
            const auto& xpath = p.first.second;
            const auto& signal = p.second;

            for (const auto& checkConfigScope : SIGNAL_SCOPE_MAP.at(configScope)) {
                auto oldSubValue = findXPath(getScopeJson(checkConfigScope, oldUserConfig), xpath);
                auto newSubValue = findXPath(getScopeJson(checkConfigScope, newUserConfig), xpath);
                if (oldSubValue == nullptr && newSubValue == nullptr) {
                    continue;
                }
                if (oldSubValue == nullptr ||
                    newSubValue == nullptr ||
                    *oldSubValue != *newSubValue) {
                    signals.push_back(signal);
                    break;
                }
            }
        }
    }

    for (auto& signal : signals) {
        signal->emit();
    }
}

void UserConfigProvider::notifyAllJsonChanges()
{
    std::vector<std::shared_ptr<SignalExternal<IJsonChangedSignal>>> signals;
    {
        std::lock_guard mlock(mutex_);
        signals.reserve(xpathSignals_.size());
        for (const auto& [_, signal] : xpathSignals_) {
            signals.push_back(signal);
        }
    }
    for (auto& signal : signals) {
        signal->emit();
    }
}
