#include "i_user_config_provider.h"

#include <yandex_io/libs/json_utils/json_utils.h>

#include <set>

using namespace quasar;

namespace {
    struct IndirectComparer {
        bool operator()(const Json::Value* a, const Json::Value* b) const {
            return *a < *b;
        }
    };

    void jsonSmartMerge(const Json::Value& src, Json::Value& dst) {
        if (src.isNull()) {
            return;
        }

        if (dst.isNull() ||
            (src.isObject() && !dst.isObject()) ||
            (src.isArray() && !dst.isArray()) ||
            (!src.isObject() && dst.isObject()) ||
            (!src.isArray() && dst.isArray())) {
            dst = src;
            return;
        }

        if (src.isObject()) {
            for (const auto& key : src.getMemberNames()) {
                // will clean destination object if source is null
                if (src[key].isNull() && dst.isMember(key)) {
                    dst.removeMember(key);
                    continue;
                }
                jsonSmartMerge(src[key], dst[key]);
            }
        } else if (src.isArray()) {
            std::set<const Json::Value*, IndirectComparer> cache;
            for (const auto& item : dst) {
                cache.insert(&item);
            }
            for (const auto& item : src) {
                if (!cache.contains(&item)) {
                    dst.append(item);
                }
            }
        } else {
            dst = src;
        }
    }

} // namespace

bool AccountDevice::operator==(const AccountDevice& other) const {
    return deviceId == other.deviceId &&
           platform == other.platform &&
           name == other.name &&
           serverCertificate == other.serverCertificate &&
           serverPrivateKey == other.serverPrivateKey;
}

bool AccountDevice::operator!=(const AccountDevice& other) const {
    return !(*this == other);
}

bool UserConfig::operator==(const UserConfig& other) const {
    return auth == other.auth &&
           passportUid == other.passportUid &&
           subscriptionInfo == other.subscriptionInfo &&
           accountDevices == other.accountDevices &&
           account == other.account &&
           device == other.device &&
           system == other.system;
}

bool UserConfig::operator!=(const UserConfig& other) const {
    return !(*this == other);
}

Json::Value UserConfig::merged(const std::string& xpath) const {
    Json::Value result;

    if (auto subjson = findXPath(account, xpath)) {
        jsonSmartMerge(*subjson, result);
    }

    if (auto subjson = findXPath(device, xpath)) {
        jsonSmartMerge(*subjson, result);
    }

    if (auto subjson = findXPath(system, xpath)) {
        jsonSmartMerge(*subjson, result);
    }

    return result;
}
