#include "volume_manager_provider.h"

#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/signals/live_data.h>
#include <yandex_io/libs/signals/signal.h>
#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

namespace {
    VolumeManagerState convertVolumeManagerState(const proto::VolumeManager::State& vms)
    {
        return {
            .platformVolume = vms.platform_volume(),
            .aliceVolume = vms.alice_volume(),
            .isMuted = vms.is_muted(),
            .source = vms.source(),
            .setBtVolume = vms.set_bt_volume(),
        };
    }
} // namespace

VolumeManagerProvider::VolumeManagerProvider(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : VolumeManagerProvider(ipcFactory->createIpcConnector("volume_manager"))
{
}

VolumeManagerProvider::VolumeManagerProvider(std::shared_ptr<ipc::IConnector> connector)
    : connector_(std::move(connector))
    , volumeManagerState_(std::make_shared<VolumeManagerState>())
{
    connector_->setMessageHandler(
        makeSafeCallback([this](const auto& message) {
            if (message->has_volume_manager_message()) {
                const auto& vm = message->volume_manager_message();
                if (vm.has_state()) {
                    auto oldState = volumeManagerState_.value();
                    auto newState = convertVolumeManagerState(vm.state());
                    if (*oldState != newState) {
                        volumeManagerState_ = std::make_shared<VolumeManagerState>(std::move(newState));
                    }
                }
            }
        }, lifetime_));
    connector_->connectToService();
}

IVolumeManagerState& VolumeManagerProvider::volumeManagerState()
{
    return volumeManagerState_;
}

void VolumeManagerProvider::setPlatformVolume(int platformVolume, bool isMuted, const std::string& source) {
    proto::QuasarMessage message;
    message.mutable_volume_manager_message()->mutable_set_volume_signal()->set_platform_volume(platformVolume);
    message.mutable_volume_manager_message()->mutable_set_volume_signal()->set_is_muted(isMuted);
    message.mutable_volume_manager_message()->mutable_set_volume_signal()->set_source(TString(source));
    connector_->sendMessage(std::move(message));
}

void VolumeManagerProvider::setAliceVolume(int aliceVolume, bool isMuted, const std::string& source) {
    proto::QuasarMessage message;
    message.mutable_volume_manager_message()->mutable_set_volume_signal()->set_alice_volume(aliceVolume);
    message.mutable_volume_manager_message()->mutable_set_volume_signal()->set_is_muted(isMuted);
    message.mutable_volume_manager_message()->mutable_set_volume_signal()->set_source(TString(source));
    connector_->sendMessage(std::move(message));
}
