#include "activity_tracker.h"

#include <algorithm>

YIO_DEFINE_LOG_MODULE("activity_tracker");

namespace YandexIO {

    void ActivityTracker::clear()
    {
        currentFocusChannel_.reset();
        for (auto& activityQueue : channels_) {
            activityQueue.clear();
        }
    }

    bool ActivityTracker::addActivity(const IActivityPtr& activity) {
        YIO_LOG_INFO("addActivity (" << activity.get() << ")" << (activity == nullptr ? "" : activity->activityName()));

        if (activity == nullptr) {
            YIO_LOG_ERROR_EVENT("ActivityTracker.FailedAddNullActivity", "");
            return false;
        }
        const auto channel = activity->getAudioChannel();
        if (channel < quasar::proto::AudioChannel_MIN || channel > quasar::proto::AudioChannel_MAX) {
            YIO_LOG_ERROR_EVENT("ActivityTracker.AddActivityInvalidChannel", "channel " << channel << " is out of range");
            return false;
        }

        auto currentForegroundActivity = getFirstActivityStartingFrom(quasar::proto::AudioChannel_MAX);

        if (activity->isLocal()) {
            channels_[channel].push_front(activity);
        } else {
            channels_[channel].push_back(activity);
        }

        for (const auto& listener : listeners_) {
            listener->onActivityAdded(activity);
        }
        if (currentForegroundActivity == nullptr) {
            setForegroundActivity(activity);
        } else if (activity->isLocal() && currentForegroundActivity->getAudioChannel() <= activity->getAudioChannel()) {
            currentForegroundActivity->setBackground();
            setForegroundActivity(activity);
        } else if (!activity->isLocal() && currentForegroundActivity->getAudioChannel() < activity->getAudioChannel()) {
            currentForegroundActivity->setBackground();
            setForegroundActivity(activity);
        }

        return true;
    }

    bool ActivityTracker::removeActivity(const IActivityPtr& activity) {
        YIO_LOG_INFO("removeActivity (" << activity.get() << ")" << (activity == nullptr ? "" : activity->activityName()));

        if (activity == nullptr) {
            YIO_LOG_ERROR_EVENT("ActivityTracker.FailedRemoveNullActivity", "");
            return false;
        }
        const auto channel = activity->getAudioChannel();
        if (channel < quasar::proto::AudioChannel_MIN || channel > quasar::proto::AudioChannel_MAX) {
            YIO_LOG_ERROR_EVENT("ActivityTracker.RemoveActivityInvalidChannel", "channel " << channel << " is out of range");
            return false;
        }

        auto& activities = channels_[channel];
        auto activityIter = std::find(activities.begin(), activities.end(), activity);
        if (activityIter == activities.end()) {
            YIO_LOG_DEBUG("Failed to remove activity: activity " << activity.get() << " not found");
            return false;
        }

        const bool isFirstAndTopActivity = (activityIter == activities.begin()) && isTopPriorityChannel(channel);
        activity->setBackground();
        activities.erase(activityIter);
        for (const auto& listener : listeners_) {
            listener->onActivityRemoved(activity);
        }

        if (isFirstAndTopActivity) {
            std::optional<IActivityPtr> nextForegroundActivity = findNextForegroundActivity(channel);
            if (nextForegroundActivity.has_value()) {
                setForegroundActivity(nextForegroundActivity.value());
            }
        }

        return true;
    }

    std::optional<quasar::proto::AudioChannel> ActivityTracker::getCurrentFocusChannel() const {
        return currentFocusChannel_;
    }

    IActivityPtr ActivityTracker::findNextForegroundActivity(quasar::proto::AudioChannel channel) const {
        auto& activities = channels_[channel];

        if (!activities.empty()) {
            return *activities.begin();
        } else {
            if (channel > quasar::proto::AudioChannel_MIN) {
                return getFirstActivityStartingFrom(static_cast<quasar::proto::AudioChannel>(channel - 1));
            }
            return nullptr;
        }
    }

    bool ActivityTracker::isTopPriorityChannel(quasar::proto::AudioChannel channel) const {
        const auto activity = getFirstActivityStartingFrom(quasar::proto::AudioChannel_MAX);
        return activity != nullptr && activity->getAudioChannel() == channel;
    }

    void ActivityTracker::setForegroundActivity(const IActivityPtr& value) {
        YIO_LOG_INFO("setForegroundActivity (" << value.get() << ")" << (value == nullptr ? "" : value->activityName()));

        const auto prevFocusChannel = currentFocusChannel_;

        if (value == nullptr) {
            currentFocusChannel_.reset();
        } else {
            value->setForeground();
            currentFocusChannel_ = value->getAudioChannel();
        }

        if (prevFocusChannel != currentFocusChannel_) {
            YIO_LOG_INFO("ActivityFocus changed: " << formatCurrentFocusChannel());
        }
    }

    std::string ActivityTracker::formatCurrentFocusChannel() const {
        if (currentFocusChannel_.has_value()) {
            return quasar::proto::AudioChannel_Name(currentFocusChannel_.value());
        } else {
            return "none";
        }
    }

    IActivityPtr ActivityTracker::getFirstActivityStartingFrom(quasar::proto::AudioChannel channel) const {
        for (int i = channel; i >= quasar::proto::AudioChannel_MIN; --i) {
            if (!channels_[i].empty()) {
                return channels_[i].front();
            }
        }

        return nullptr;
    }

    void ActivityTracker::addListener(std::shared_ptr<Listener> listener) {
        if (listener) {
            listeners_.insert(listener);
        }
    }

    void ActivityTracker::removeListener(std::shared_ptr<Listener> listener) {
        listeners_.erase(listener);
    }
} // namespace YandexIO
