#include "app_metrica.h"

#include "app_metrica_client.h"
#include "app_metrica_sender.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/telemetry/telemetry.h>

#include <chrono>
#include <ctime>
#include <memory>
#include <string>

using std::string;

namespace quasar {

    MetricaMetadata MetricaMetadata::fromReportConfig(const ReportConfiguration& reportConfig) {
        MetricaMetadata metricaMetadata;
        metricaMetadata.UUID = reportConfig.getUuid();
        metricaMetadata.deviceID = reportConfig.getDeviceId();
        return metricaMetadata;
    }

    AppMetrica::AppMetrica(std::shared_ptr<quasar::EventsDatabase> db,
                           std::string sessionIdPersistentPath,
                           std::string sessionIdTemporaryPath,
                           std::shared_ptr<YandexIO::IDevice> device)
        : device_(std::move(device))
        , queue_(new blockingQueue(queueSize_, AppMetrica::onOverflow, AppMetrica::onOverflow))
        , threadStopped_(false)
        , db_(std::move(db))
    {
        periodicExecutor_ = std::make_unique<PeriodicExecutor>(
            std::bind(&AppMetrica::sendReports, this),
            reportSendPeriod_);
        dbSizeSender_ = std::make_unique<PeriodicExecutor>(
            std::bind(&AppMetrica::sendDbSize, this),
            std::chrono::minutes(30));
        metricaClient = std::make_unique<AppMetricaClient>(db_, queue_, std::move(sessionIdPersistentPath), std::move(sessionIdTemporaryPath));
    }

    AppMetrica::AppMetrica(std::chrono::milliseconds reportSendPeriod,
                           std::shared_ptr<ReportConfiguration> configuration,
                           std::shared_ptr<quasar::EventsDatabase> db,
                           std::string sessionIdPersistentPath,
                           std::string sessionIdTemporaryPath,
                           std::shared_ptr<YandexIO::IDevice> device)
        : device_(std::move(device))
        , reportSendPeriod_(reportSendPeriod)
        , queue_(new blockingQueue(queueSize_, AppMetrica::onOverflow, AppMetrica::onOverflow))
        , threadStopped_(false)
        , db_(std::move(db))
    {
        periodicExecutor_ = std::make_unique<PeriodicExecutor>(
            std::bind(&AppMetrica::sendReports, this),
            reportSendPeriod_);
        dbSizeSender_ = std::make_unique<PeriodicExecutor>(
            std::bind(&AppMetrica::sendDbSize, this),
            std::chrono::minutes(30));
        metricaClient = std::make_unique<AppMetricaClient>(db_, queue_, std::move(sessionIdPersistentPath), std::move(sessionIdTemporaryPath));

        std::lock_guard<std::mutex> guard(metricaSenderMutex_);
        metricaSender = std::make_unique<AppMetricaSender>(std::move(configuration), db_, queue_, device_);
    }

    AppMetrica::~AppMetrica() {
        threadStopped_ = true;
        /* Remove send reports periodic, so it won't access destructed metricaSender */
        if (initMetricaHttpClientAsyncThread_.joinable()) {
            initMetricaHttpClientAsyncThread_.join();
        }
        periodicExecutor_.reset(nullptr);
    }

    void AppMetrica::initMetricaHttpClientAsync(StartupConfiguration startupConfig) {
        initMetricaHttpClientAsyncThread_ = std::thread(&AppMetrica::initMetricaHttpClient, this, std::move(startupConfig));
    }

    MetricaMetadata AppMetrica::initMetricaHttpClient(StartupConfiguration startupConfig) {
        StartupClient startupClient(startupConfig, device_);
        std::shared_ptr<ReportConfiguration> reportConfig = startupClient.getReportConfig(threadStopped_);

        if (threadStopped_) {
            // Getting report config may take long time,
            // if thread is stopped when executing startupClient.getReportConfig(threadStopped_), we need to return
            return MetricaMetadata::fromReportConfig(*reportConfig);
        }

        {
            std::lock_guard<std::mutex> guard(metricaSenderMutex_);
            metricaSender = std::make_unique<AppMetricaSender>(reportConfig, db_, queue_, device_);
            if (!reportConfig->getRevision().empty()) {
                metricaClient->putEnvironmentVariable("revision", reportConfig->getRevision());
            }
            // report that SessionId are ordered and can be used to sort metrics
            metricaClient->putEnvironmentVariable("session_id_ordered", "1");
        }

        return MetricaMetadata::fromReportConfig(*reportConfig);
    }

    void AppMetrica::sendReports() {
        YIO_LOG_DEBUG("Sending AppMetrica report");
        if (metricaSenderInitialized()) {
            size_t reportsCount = metricaSender->sendReports();
            YIO_LOG_DEBUG("Sent " << reportsCount << " events");
        } else {
            YIO_LOG_WARN("Metrica sender does not exist yet");
        }
        std::lock_guard<std::mutex> guard(dbSizeMutex_);
        dbSize_.process(db_->getDatabaseSizeInBytes());
    }

    bool AppMetrica::metricaSenderInitialized() const {
        std::lock_guard<std::mutex> guard(metricaSenderMutex_); // barrier
        return bool(metricaSender);
    }

    void AppMetrica::sendDbSize() {
        std::lock_guard<std::mutex> guard(dbSizeMutex_);
        Json::Value dbSizeData;
        dbSizeData["value"]["min"] = dbSize_.getMin();
        dbSizeData["value"]["max"] = dbSize_.getMax();
        dbSizeData["value"]["mean"] = dbSize_.getMean();
        dbSizeData["value"]["last"] = dbSize_.getLast();
        dbSize_.reset();
        dbSizeData["filename"] = db_->getDbFilename();
        device_->telemetry()->reportEvent("sqliteDatabaseSize", jsonToString(dbSizeData), true);
    }

    void AppMetrica::onOverflow(size_t queueSize) {
        YIO_LOG_WARN("Queue overflowing with messages, current size is " << queueSize << " ");
    }

    void AppMetrica::processStats(Json::Value payload) const {
        std::string strValue;
        {
            Json::Value fullPayload = Json::objectValue;
            fullPayload["daemons"] = std::move(payload);
            fullPayload["client"] = metricaClient->getStatsPayload();
            if (metricaSenderInitialized()) {
                fullPayload["sender"] = metricaSender->getStatsPayload();
            }
            strValue = jsonToString(fullPayload);
        }
        YIO_LOG_INFO("appmetrica telemetryStats " << strValue);
        metricaClient->reportEvent("telemetryStats", strValue, false);
    }
} /* namespace quasar */
