#pragma once

#include "app_metrica_client.h"
#include "app_metrica_sender.h"
#include "metrica_metadata.h"
#include "startup_client.h"
#include "startup_configuration.h"

#include <yandex_io/libs/device/i_device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/metrica/base/events_database.h>
#include <yandex_io/libs/metrics_collector/numeric_stat.h>
#include <yandex_io/libs/threading/periodic_executor.h>

#include <yandex_io/protos/quasar_proto.pb.h>

#include <chrono>
#include <mutex>
#include <thread>

namespace quasar {
    class AppMetrica {
    public:
        AppMetrica(std::shared_ptr<quasar::EventsDatabase> db,
                   std::string sessionIdPersistentPath,
                   std::string sessionIdTemporaryPath,
                   std::shared_ptr<YandexIO::IDevice> device);

        // test only
        AppMetrica(std::chrono::milliseconds reportSendPeriod,
                   std::shared_ptr<ReportConfiguration> configuration,
                   std::shared_ptr<quasar::EventsDatabase> db,
                   std::string sessionIdPersistentPath,
                   std::string sessionIdTemporaryPath,
                   std::shared_ptr<YandexIO::IDevice> device);

        ~AppMetrica();

        MetricaMetadata initMetricaHttpClient(StartupConfiguration startupConfig);

        /**
         * @brief Starts initMetricaHttpClient in a background thread.
         *
         * Thread will be stopped (joined) in AppMetrica destructor
         */
        void initMetricaHttpClientAsync(StartupConfiguration startupConfig);

        bool metricaSenderInitialized() const;

        void processStats(Json::Value payload) const;

        std::unique_ptr<AppMetricaClient> metricaClient;
        std::unique_ptr<AppMetricaSender> metricaSender;

    private:
        void sendDbSize();
        void sendReports();
        static void onOverflow(size_t queueSize);

        constexpr static size_t queueSize_ = 20; // Queue size for immediate events like bugreport and minidump

        const std::shared_ptr<YandexIO::IDevice> device_;
        // Every this event is rather heavy, so we mustn't keep too many of these events in RAM
        std::chrono::milliseconds reportSendPeriod_ = std::chrono::seconds(10);
        std::shared_ptr<blockingQueue> queue_;

        std::atomic_bool threadStopped_;
        std::shared_ptr<quasar::EventsDatabase> db_;

        mutable std::mutex metricaSenderMutex_;

        std::thread initMetricaHttpClientAsyncThread_;

        /* Keep this periodic after metricaSender_ so it will be destructed first and won't use freed memory */
        std::unique_ptr<quasar::PeriodicExecutor> periodicExecutor_;

        std::mutex dbSizeMutex_;
        quasar::NumericStat<int> dbSize_;
        std::unique_ptr<quasar::PeriodicExecutor> dbSizeSender_;
    };

} // namespace quasar
