#include "app_metrica_client.h"

#include "app_metrica_utils.h"

#include <yandex_io/libs/appmetrica/proto/metrica.pb.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/counters/json_utils.h>

#include <json/json.h>

#include <cmath>
#include <ctime>
#include <fstream>
#include <iostream>
#include <utility>

using namespace quasar;
using namespace quasar::proto;
using quasar::proto::DatabaseMetricaEvent;

AppMetricaClient::AppMetricaClient(std::shared_ptr<quasar::EventsDatabase> db,
                                   std::shared_ptr<blockingQueue> immediateQueue,
                                   std::string sessionIdPersistentPath,
                                   std::string sessionIdTemporaryPath)
    : db_(std::move(db))
    , immediateQueue_(std::move(immediateQueue))
    , sessionProvider_(std::move(sessionIdPersistentPath), std::move(sessionIdTemporaryPath))
{
    startSession();
}

void AppMetricaClient::startSession()
{
    /* Call start session event immediately (through immediateQueue_) */
    auto session = sessionProvider_.generateNewSession();

    auto event = ReportMessage_Session_Event();
    event.set_type(ReportMessage_Session_Event_EventType::ReportMessage_Session_Event_EventType_EVENT_START);
    event.set_time(getNowInSeconds());
    event.set_number_in_session(session.eventNumber);
    event.mutable_network_info()->set_connection_type(toMetricaEventConnectionType(connectionType_.load()));

    if (!userId_.empty()) {
        auto account = event.mutable_account();
        account->set_id(TString(userId_));
        account->set_type("login");
    }
    std::scoped_lock<std::mutex> guard(envVariablesMapMutex_);
    if (!immediateQueue_->tryPush(ReportOneEvent(event, environmentVariables_, session))) {
        YIO_LOG_WARN("Queue overflowed with messages. New event has been dropped.");
    }
}

void AppMetricaClient::reportEventImmediate(const std::string& eventName, const std::string& eventValue, ReportMessage::Session::Event::EventType eventType) {
    const auto session = sessionProvider_.getAndIncrementSession();
    auto event = ReportMessage_Session_Event();
    event.set_type(eventType);
    event.set_time(getNowInSeconds());
    event.set_name(TString(eventName));
    event.set_value(TString(eventValue));
    event.set_number_in_session(session.eventNumber);
    event.mutable_network_info()->set_connection_type(toMetricaEventConnectionType(connectionType_.load()));

    if (!userId_.empty()) {
        auto account = event.mutable_account();
        account->set_id(TString(userId_));
        account->set_type("login");
    }
    std::scoped_lock<std::mutex> guard(envVariablesMapMutex_);
    if (!immediateQueue_->tryPush(ReportOneEvent(event, environmentVariables_, session))) {
        YIO_LOG_WARN("Queue overflowed with messages. New event has been dropped.");
        dailyStats_.increment({Counters::TOTAL, Counters::IMMEDIATE_DROP});
    } else {
        dailyStats_.increment(Counters::TOTAL);
    }
}

void AppMetricaClient::reportEventOverDb(const std::string& eventName, const std::string& eventValue, quasar::proto::DatabaseMetricaEvent::NewEvent::Type eventType) {
    const auto session = sessionProvider_.getAndIncrementSession();
    DatabaseMetricaEvent event;
    auto newEvent = event.mutable_new_event();
    newEvent->set_type(eventType);
    newEvent->set_timestamp(getNowInSeconds());
    newEvent->set_name(TString(eventName));
    newEvent->set_value(TString(eventValue));
    newEvent->set_serial_number(session.eventNumber);
    if (!userId_.empty()) {
        newEvent->set_account_type("login");
        newEvent->set_account_id(TString(userId_));
    }
    newEvent->set_session_id(session.id);
    newEvent->set_session_start_time(session.startTime);
    newEvent->set_connection_type(connectionType_.load());

    if (db_->addEvent(event)) {
        dailyStats_.increment(Counters::TOTAL);
    } else {
        dailyStats_.increment({Counters::TOTAL, Counters::DBFAIL_DROP});
    }
}

void AppMetricaClient::reportEvent(const std::string& eventName, const std::string& eventValue, bool skipDatabase) {
    if (!shouldProcessEvent(eventName)) {
        dailyStats_.increment({Counters::TOTAL, Counters::BLACKLIST});
        return;
    }

    if (skipDatabase) {
        reportEventImmediate(eventName, eventValue, ReportMessage::Session::Event::EVENT_CLIENT);
    } else {
        reportEventOverDb(eventName, eventValue, DatabaseMetricaEvent::NewEvent::CLIENT);
    }
}

void AppMetricaClient::putEnvironmentVariable(const std::string& variableName, const std::string& variableValue) {
    std::scoped_lock<std::mutex> guard(envVariablesMapMutex_);
    const auto iterator = environmentVariables_.find(variableName);
    if (iterator != environmentVariables_.end() && iterator->second == variableValue) {
        return;
    }
    environmentVariables_[variableName] = variableValue;
    DatabaseMetricaEvent event;
    auto environmentValues = event.mutable_new_environment()->mutable_environment_values();
    for (auto const& item : environmentVariables_) {
        (*environmentValues)[item.first] = item.second;
    }
    db_->addEvent(event);
}

void AppMetricaClient::deleteEnvironmentVariable(const std::string& variableName) {
    std::scoped_lock<std::mutex> guard(envVariablesMapMutex_);
    if (environmentVariables_.erase(variableName) > 0) {
        DatabaseMetricaEvent event;
        auto environmentValues = event.mutable_new_environment()->mutable_environment_values();
        for (auto const& item : environmentVariables_) {
            (*environmentValues)[item.first] = item.second;
        }
        db_->addEvent(event);
    }
}

void AppMetricaClient::reportError(const std::string& errorEventName, bool skipDatabase, const std::string& errorValue) {
    if (!shouldProcessEvent(errorEventName)) {
        dailyStats_.increment({Counters::TOTAL, Counters::BLACKLIST});
        return;
    }

    if (skipDatabase) {
        reportEventImmediate(errorEventName, errorValue, ReportMessage::Session::Event::EVENT_PROTOBUF_ERROR);
    } else {
        reportEventOverDb(errorEventName, errorValue, DatabaseMetricaEvent::NewEvent::ERROR);
    }
}

void AppMetricaClient::setUserInfo(const std::string& userId) {
    userId_ = userId;
}

void AppMetricaClient::setConnectionType(quasar::proto::ConnectionType newValue) {
    connectionType_.store(newValue);
}

uint64_t AppMetricaClient::getNowInSeconds() {
    const auto now = std::chrono::system_clock::now();
    auto timeInSec = std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count();
    return static_cast<uint64_t>(timeInSec);
}

bool AppMetricaClient::shouldProcessEvent(const std::string& eventName) {
    std::shared_lock lock(eventBlacklistMutex_);

    if (!eventBlacklist_.empty()) {
        const bool isBlacklisted = eventBlacklist_.find(eventName) != eventBlacklist_.end();
        return !isBlacklisted;
    } else {
        return true;
    }
}

void AppMetricaClient::updateEventBlacklist(const Json::Value& eventBlacklist) {
    std::unique_lock lock(eventBlacklistMutex_);

    eventBlacklist_.clear();

    if (eventBlacklist.isArray()) {
        for (size_t i = 0; i != eventBlacklist.size(); ++i) {
            auto eventName = getArrayElement(eventBlacklist, i).asString();
            eventBlacklist_.insert(std::move(eventName));
        }
    }
}

Json::Value AppMetricaClient::getStatsPayload() {
    return EnumCountersToJson(dailyStats_.getUpdatedCounters(), {"total", "immediate_drop", "dbfail_drop", "blacklist"});
}
