#include "app_metrica_sender.h"

#include "app_metrica_utils.h"

#include <yandex_io/libs/appmetrica/proto/metrica.pb.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/metrica/base/utils.h>
#include <yandex_io/libs/protobuf_utils/proto_trace.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/counters/json_utils.h>

#include <algorithm>
#include <chrono>
#include <ctime>
#include <fstream>
#include <iostream>
#include <map>
#include <string>
#include <unordered_map>

using namespace quasar;
using quasar::proto::DatabaseMetricaEvent;

AppMetricaSender::AppMetricaSender(std::shared_ptr<ReportConfiguration> reportConfig,
                                   std::shared_ptr<quasar::EventsDatabase> db,
                                   std::shared_ptr<blockingQueue> immediateQueue,
                                   std::shared_ptr<YandexIO::IDevice> device)
    : config_(std::move(reportConfig))
    , realSender_(std::move(device))
    , db_(std::move(db))
    , immediateQueue_(std::move(immediateQueue))
{
    immediateSend_ = std::thread(&AppMetricaSender::immediateSendThread, this);
    YIO_LOG_DEBUG("AppMetricaSender started, stopped_=" << stopped_ << "");
}

AppMetricaSender::~AppMetricaSender() {
    stopped_ = true;
    conditionVariable_.notify_all();
    immediateQueue_->push(StopQueue());
    immediateSend_.join();
}

size_t AppMetricaSender::sendReports() {
    size_t sentEvents = 0;
    size_t lastSent;
    // We send reports until there's something to send
    while ((lastSent = sendReport())) {
        dailyStats_.increment(StatsCounter(Counters::SENT, unsigned(lastSent)));
        sentEvents += lastSent;
    }
    return sentEvents;
}

size_t AppMetricaSender::makePersistent() {
    auto eventImportantList = (std::lock_guard(reportEnvironmentMutex_), eventImportantList_);
    if (eventImportantList && !eventImportantList->empty()) {
        auto res = db_->makePersistent(
            [&](auto /*id*/, const auto& event, bool /*saved*/) {
                return event.has_new_environment() || eventImportantList->count(event.new_event().name());
            }, maxEventId_);
        if (res.eventMaxId) {
            maxEventId_ = std::max(res.eventMaxId, maxEventId_);
        }
        if (res.eventCount) {
            YIO_LOG_INFO("To prevent the loss of " << res.eventCount << " events (" << res.eventBytes << " bytes) these saved into persistent database");
        }
        return res.eventCount;
    }
    return 0;
}
size_t AppMetricaSender::sendReport() {
    ReportMessage_Session_Event_NetworkInfo networkInfo;
    {
        /* Copy networkInfo_ once per report. So once access mutex */
        std::lock_guard<std::mutex> guard(reportEnvironmentMutex_);
        networkInfo.CopyFrom(networkInfo_);
    }

    auto report = ReportMessage();
    std::unordered_map<uint64_t, ReportMessage_Session*> sessions;
    size_t i = 0;
    std::vector<uint64_t> eventsIds;
    std::vector<uint64_t> skipIds;
    std::unique_ptr<EventsDatabase::Event> dbEvent;

    uint64_t dbEnvironmentValueId = 0;
    std::map<std::string, std::string> environmentValues;
    bool reachedNewEnvValues = false;
    unsigned reportEventsCount = 0;

    while ((dbEvent = db_->getEarliestEvent(skipIds))) {
        if (dbEvent->databaseMetricaEvent.has_new_environment()) {
            // We get new environment. If we already have at least one event, we break loop and start sending
            // If we haven't taken even 1 event from database so we don't have to keep this environment value -- it doesn't have events, delete it from database
            if (i == 0) {
                if (dbEnvironmentValueId != 0) {
                    // Previous environment values didn't have events to send, just delete it from database
                    db_->deleteEvents({dbEnvironmentValueId});
                }
                environmentValues.clear();
                for (const auto& item : dbEvent->databaseMetricaEvent.new_environment().environment_values()) {
                    environmentValues[item.first] = item.second;
                }
                dbEnvironmentValueId = dbEvent->id;
                skipIds.push_back(dbEnvironmentValueId);
                continue;
            } else {
                reachedNewEnvValues = true;
                break;
            }
        }

        // create new session if necessary
        auto sessionId = dbEvent->databaseMetricaEvent.new_event().session_id();
        auto sessionStartTime = dbEvent->databaseMetricaEvent.new_event().session_start_time();
        if (sessions.find(sessionId) == sessions.end()) {
            sessions[sessionId] = addSession(report, sessionId, sessionStartTime);
        }
        auto sessionEvents = sessions[sessionId]->mutable_events();

        // If we get here we get event, not environment value, just add it to report
        ReportMessage_Session_Event eventToSend;
        eventToSend.set_name(dbEvent->databaseMetricaEvent.new_event().name());
        eventToSend.set_time(dbEvent->databaseMetricaEvent.new_event().timestamp() - sessionStartTime);

        // Account deprecated
        if (dbEvent->databaseMetricaEvent.new_event().has_account_type()) {
            eventToSend.mutable_account()->set_type(dbEvent->databaseMetricaEvent.new_event().account_type());
        }
        if (dbEvent->databaseMetricaEvent.new_event().has_account_id()) {
            eventToSend.mutable_account()->set_id(dbEvent->databaseMetricaEvent.new_event().account_id());
        }

        if (dbEvent->databaseMetricaEvent.new_event().type() == DatabaseMetricaEvent::NewEvent::ERROR) {
            eventToSend.set_type(ReportMessage_Session_Event_EventType::ReportMessage_Session_Event_EventType_EVENT_PROTOBUF_ERROR);
            eventToSend.set_value(dbEvent->databaseMetricaEvent.new_event().value());
        } else {
            eventToSend.set_type(ReportMessage_Session_Event_EventType::ReportMessage_Session_Event_EventType_EVENT_CLIENT);
            eventToSend.set_value(dbEvent->databaseMetricaEvent.new_event().value());
        }
        if (dbEvent->databaseMetricaEvent.new_event().has_serial_number()) {
            eventToSend.set_number_in_session(dbEvent->databaseMetricaEvent.new_event().serial_number());
        } else {
            // backward compatibility, in future should simply drop events without number
            eventToSend.set_number_in_session(i);
        }

        /* Set up scanned wifi networks and network type (wifi) for each event */
        eventToSend.mutable_network_info()->CopyFrom(networkInfo);
        eventToSend.mutable_network_info()->set_connection_type(toMetricaEventConnectionType(dbEvent->databaseMetricaEvent.new_event().connection_type()));

        auto eventSize = eventToSend.SerializeAsString().size();
        if (eventSize > MAX_REPORT_SIZE) {
            dailyStats_.increment(Counters::TOO_LARGE);
            YIO_LOG_WARN("Event " << eventToSend.name() << "is too large (" << eventSize << "bytes), don't send it and remove from database");
            db_->deleteEvents({dbEvent->id});
            continue;
        }
        unsigned long reportSize = report.SerializeAsString().size();
        if (reportSize + eventSize > MAX_REPORT_SIZE) {
            reportEventsCount = countEvents(report);
            YIO_LOG_DEBUG("Current report length. Events:" << reportEventsCount << " Bytes: " << reportSize);
            YIO_LOG_INFO("Current report is larger than " << (MAX_REPORT_SIZE / 1024) << "Kb with things left over in event queue, sending it");
            break;
        }
        *sessionEvents->Add() = eventToSend;
        eventsIds.push_back(dbEvent->id);
        skipIds.push_back(dbEvent->id);
        ++i;
    }

    if (!reportEventsCount) {
        reportEventsCount = countEvents(report);
    }

    if (sessions.empty()) {
        YIO_LOG_DEBUG("No events in report, don't send anything");
        return 0;
    }

    {
        std::lock_guard<std::mutex> guard(reportEnvironmentMutex_);

        for (const auto& env : environmentValues) {
            if (
                envKeysBlacklist_.empty() || envKeysBlacklist_.find(env.first) == envKeysBlacklist_.end()) {
                auto report_env = report.add_app_environment();
                report_env->set_name(TString(env.first));
                report_env->set_value(TString(env.second));
            }
        }
    }

    auto sendTime = report.mutable_send_time();
    const auto now = std::chrono::system_clock::now();
    auto timestampInSec = std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count();
    sendTime->set_timestamp(static_cast<uint64_t>(timestampInSec));
    sendTime->set_time_zone(report.sessions(0).session_desc().start_time().time_zone());
    __PROTOTRACE("Metrica", report);
    const auto serializedReport = report.SerializeAsString();
    const bool reportSent = realSender_.trySendReport(std::move(report), config_->reportRequestURLS());
    if (reportSent) {
        auto maxIt = std::max_element(eventsIds.begin(), eventsIds.end());
        maxEventId_ = std::max(maxEventId_, (maxIt != eventsIds.end() ? *maxIt : 0));
        // Deleting events from database
        db_->deleteEvents(eventsIds);
        if (reachedNewEnvValues) {
            db_->deleteEvents({dbEnvironmentValueId});
        }

        return reportEventsCount;
    } else {
        YIO_LOG_WARN("Unable to send report to any host");
        makePersistent();
        return 0;
    }
}

ReportMessage_Session* AppMetricaSender::addSession(ReportMessage& report, uint64_t sessionId, uint64_t sessionStartTime) {
    auto session = report.add_sessions();
    session->set_id(sessionId);

    auto sessionDesc = session->mutable_session_desc();
    sessionDesc->set_session_type(ReportMessage_Session::SessionDesc::SESSION_FOREGROUND);
    sessionDesc->set_locale("ru-RU");

    auto session_start_time = sessionDesc->mutable_start_time();
    session_start_time->set_timestamp(sessionStartTime);

    // Has to be set because required field, default is Moscow (geolocation nmodule will override it anyway)
    session_start_time->set_time_zone(3 * 60 * 60 /* 3 hours */);
    {
        /* Have only 1 session -> set up location and timezone for it */
        std::lock_guard<std::mutex> guard(reportEnvironmentMutex_);
        if (locationMessage_.has_lat() && locationMessage_.has_lon()) {
            sessionDesc->mutable_location()->CopyFrom(locationMessage_);
        }
        /* Set up timezone offset sec */
        if (time_.has_time_zone()) {
            session_start_time->set_time_zone(time_.time_zone());
        }
    }

    return session;
}

size_t AppMetricaSender::countEvents(const ReportMessage& report) {
    size_t eventsCount = 0;
    for (auto& session : report.sessions()) {
        eventsCount += session.events().size();
    }
    return eventsCount;
}

void AppMetricaSender::immediateSendThread() {
    while (!stopped_) {
        queueElement element;
        immediateQueue_->pop(element);
        ReportOneEvent event_from_client;
        try {
            event_from_client = std::get<ReportOneEvent>(element);
        } catch (const std::bad_variant_access& e) {
            YIO_LOG_INFO("Queue got stop event, quitting");
            break;
        }
        auto report = ReportMessage();
        auto session = addSession(report, event_from_client.session.id, event_from_client.session.startTime);
        auto events = session->mutable_events();
        auto event = event_from_client.event;

        session->mutable_session_desc()->set_session_type(ReportMessage_Session::SessionDesc::SESSION_FOREGROUND);
        event.set_time(event.time() - event_from_client.session.startTime);

        {
            std::lock_guard<std::mutex> guard(reportEnvironmentMutex_);
            /* Set up scanned wifi list to each event */
            event.mutable_network_info()->CopyFrom(networkInfo_);
        }

        *events->Add() = event;

        for (const auto& env : event_from_client.environmentVariables) {
            auto report_env = report.add_app_environment();
            report_env->set_name(TString(env.first));
            report_env->set_value(TString(env.second));
        }

        auto send_time = report.mutable_send_time();
        const auto now = std::chrono::system_clock::now();
        auto timestampInSec = std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count();
        send_time->set_timestamp(static_cast<unsigned long>(timestampInSec));
        send_time->set_time_zone(session->session_desc().start_time().time_zone());

        bool reportSent = false;
        while (!reportSent && !stopped_) {
            reportSent = realSender_.trySendReport(report, config_->reportRequestURLS());
            if (!reportSent) {
                YIO_LOG_WARN("Unable to send report to any host");
                std::unique_lock<std::mutex> lock(mutex_);
                conditionVariable_.wait_for(lock, std::chrono::seconds(RETRY_PERIOD_SEC_),
                                            [&]() { return stopped_.load(); });
            }
        }
    }
}

void AppMetricaSender::setLocation(double lat, double lon) {
    std::lock_guard<std::mutex> guard(reportEnvironmentMutex_);
    locationMessage_.set_lat(lat);
    locationMessage_.set_lon(lon);
}

void AppMetricaSender::setTimezoneOffsetSec(int32_t offsetSec) {
    std::lock_guard<std::mutex> guard(reportEnvironmentMutex_);
    time_.set_time_zone(offsetSec);
}

void AppMetricaSender::setNetworkInfo(const ReportMessage_Session_Event_NetworkInfo& networkInfo) {
    std::lock_guard<std::mutex> guard(reportEnvironmentMutex_);
    networkInfo_.CopyFrom(networkInfo);
}

void AppMetricaSender::setEventImportantList(std::shared_ptr<const std::unordered_set<std::string>> eventImportantList) {
    std::lock_guard<std::mutex> guard(reportEnvironmentMutex_);
    eventImportantList_ = std::move(eventImportantList);
}

void AppMetricaSender::setEnvKeysBlacklist(std::unordered_set<std::string> envKeysBlacklist) {
    std::lock_guard<std::mutex> guard(reportEnvironmentMutex_);
    envKeysBlacklist_ = std::move(envKeysBlacklist);
}

Json::Value AppMetricaSender::getStatsPayload() {
    return EnumCountersToJson(dailyStats_.getUpdatedCounters(), {"too_large", "sent"});
}
