#pragma once

#include "report_configuration.h"
#include "app_metrica_sender_base.h"
#include "typedefs.h"

#include <yandex_io/libs/appmetrica/proto/metrica.pb.h>

#include <yandex_io/libs/device/i_device.h>
#include <yandex_io/libs/metrica/base/events_database.h>
#include <yandex_io/libs/threading/blocking_queue.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>
#include <yandex_io/libs/counters/default_daily_counter.h>

#include <json/json.h>

#include <atomic>
#include <memory>
#include <mutex>
#include <random>
#include <utility>

class AppMetricaSender final {
public:
    AppMetricaSender(std::shared_ptr<ReportConfiguration> reportConfig, std::shared_ptr<quasar::EventsDatabase> db,
                     std::shared_ptr<blockingQueue> immediateQueue, std::shared_ptr<YandexIO::IDevice> device);
    ~AppMetricaSender();

    size_t sendReports();
    void setLocation(double lat, double lon);

    void setTimezoneOffsetSec(int32_t offsetSec);

    void setNetworkInfo(const ReportMessage_Session_Event_NetworkInfo& networkInfo);

    void setEventImportantList(std::shared_ptr<const std::unordered_set<std::string>> eventImportantList);
    void setEnvKeysBlacklist(std::unordered_set<std::string> envKeysBlacklist);

    Json::Value getStatsPayload();

private:
    const unsigned int RETRY_PERIOD_SEC_ = 10;
    std::atomic_bool stopped_{false};
    std::mutex mutex_;
    quasar::SteadyConditionVariable conditionVariable_;
    size_t makePersistent();
    size_t sendReport();

    static constexpr int MAX_REPORT_SIZE = 1024 * 200; // We don't send reports with size more than 200 Kb
    std::shared_ptr<ReportConfiguration> config_;
    AppMetricaSenderBase realSender_;
    ReportMessage_Session* addSession(ReportMessage& report, uint64_t sessionId, uint64_t sessionStartTime);
    static size_t countEvents(const ReportMessage& report);

    std::shared_ptr<quasar::EventsDatabase> db_;
    uint64_t maxEventId_{0};

    std::shared_ptr<blockingQueue> immediateQueue_;
    std::thread immediateSend_;
    void immediateSendThread();

    /**
     * @brief reportEnvironmentMutex_ protects data, that should be set up for each Report or Event:
     *        Location, Time (timezone), NetworkInfo
     */
    std::mutex reportEnvironmentMutex_;
    ReportMessage_Session_SessionDesc_Location locationMessage_; /* for each report */
    /* Save timezone in Time (use has_time_zone) to check that it was set up */
    Time time_; /* for each report */
    std::shared_ptr<const std::unordered_set<std::string>> eventImportantList_;
    std::unordered_set<std::string> envKeysBlacklist_;

    /**
     * @brief networkInfo_ holds information about current network connection as well as all recently scanned
     * wifi networks.
     *          networkInfo_ should be set up directly by setNetworkInfo call, otherwise empty will be used
     */
    ReportMessage_Session_Event_NetworkInfo networkInfo_;

    enum Counters {
        TOO_LARGE,
        SENT,
        MAX_
    };

    using DailyCounter = quasar::DefaultDailyCounter<Counters>;
    using StatsCounter = typename DailyCounter::CounterType;
    DailyCounter dailyStats_;
};
