#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/metrica/base/utils.h>

#include "app_metrica_sender_base.h"

namespace {
    std::string serializeAndCompress(ReportMessage report) {
        return compress_string(report.SerializeAsString());
    }
} // namespace

using namespace quasar;

AppMetricaSenderBase::AppMetricaSenderBase(std::shared_ptr<YandexIO::IDevice> device, bool keepAlive)
    : httpClient_("appmetrica", std::move(device))
{
    httpClient_.setFollowRedirect(true);
    httpClient_.setConnectionTimeout(std::chrono::seconds(3));
    httpClient_.setTimeout(std::chrono::seconds(30));
    // disable http_request metrics to avoid infinite loop when metrica sender
    // triggers new metrica to send
    httpClient_.disableHttpRequestMetrics();
    httpClient_.setReuse(keepAlive);
}

AppMetricaSenderBase::AppMetricaSenderBase(std::shared_ptr<YandexIO::IDevice> device)
    : AppMetricaSenderBase(device, false)
{
}

bool AppMetricaSenderBase::trySendReport(ReportMessage report, const std::vector<std::string>& hosts) {
    const auto compressed = serializeAndCompress(std::move(report));
    for (const auto& url : hosts) {
        YIO_LOG_DEBUG("Trying to send report to " << url);
        if (sendPostRequest(url, compressed)) {
            return true;
        }
    }
    return false;
}

bool AppMetricaSenderBase::sendPostRequest(const std::string& url, const std::string& payload) {
    HttpClient::Headers headers = {
        {"Accept", "application/json"},
        {"Accept-Encoding", "gzip"},
        {"Content-Encoding", "gzip"},
        {"Send-Timestamp", std::to_string(std::time(nullptr))}};
    try {
        /* Send Wifi List to Server and get current location */
        HttpClient::HttpResponse response = httpClient_.post("report", url, payload, headers);
        if (response.responseCode == 200) {
            return true;
        }
        YIO_LOG_INFO("Unable to send report, Non-200 response (" + response.body + ") from " + url + ": " + to_string(response.responseCode));
    } catch (const std::runtime_error& e) {
        YIO_LOG_WARN("Unable to send report: " << e.what());
    }
    return false;
}
