#include "metrica_metadata.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/base/persistent_file.h>

#include <fstream>

YIO_DEFINE_LOG_MODULE("appmetrica");

using namespace quasar;

Json::Value MetricaMetadata::toJson() const {
    Json::Value json;
    json["UUID"] = UUID;
    json["deviceID"] = deviceID;
    return json;
}

MetricaMetadata MetricaMetadata::fromJson(const Json::Value& json) {
    if (!json.isMember("UUID")) {
        throw std::logic_error(jsonToString(json));
    }

    MetricaMetadata result;
    result.UUID = json["UUID"].asString();

    if (json.isMember("deviceID")) {
        result.deviceID = json["deviceID"].asString();
    }

    return result;
}

MetricaMetadata MetricaMetadata::loadMetadata(const std::string& filepath) {
    if (!fileExists(filepath)) {
        return MetricaMetadata{};
    }

    std::ifstream file{filepath};
    if (!file.is_open()) {
        YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.LoadMetadata.OpenFailed", "Failed to open " << filepath);
        return MetricaMetadata{};
    }

    try {
        return MetricaMetadata::fromJson(parseJson(file));
    } catch (const Json::Exception& e) {
        YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.LoadMetadata.InvalidJson", "Failed to read from " << filepath << ", JSON is broken (" << e.what() << ")");
    } catch (const std::logic_error& e) {
        YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.LoadMetadata.ReadJsonFailed", "Failed to read from JSON. " << e.what());
    }

    return MetricaMetadata{};
}

void MetricaMetadata::saveMetadata(const std::string& filepath) const {
    const auto json{toJson()};

    TransactionFile file{filepath};
    try {
        if (!file.write(jsonToString(json))) {
            YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.SaveMetadata.WriteFailed", "Failed to write to temporary file.");
            return;
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.SaveMetadata.WriteException", "Failed to write to temporary file. " << e.what());
        return;
    }

    if (!file.commit()) {
        YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.SaveMetadata.CommitFailed", "Failed to write to persistent file.");
    }
}
