#include "startup_client.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/metrica/base/utils.h>

#include <json/json.h>

#include <algorithm>
#include <chrono>
#include <memory>
#include <thread>

YIO_DEFINE_LOG_MODULE("appmetrica");

namespace {
    void initHttpClient(quasar::HttpClient& httpClient) {
        httpClient.setTimeout(std::chrono::seconds{30});
        httpClient.setRetriesCount(0);
        httpClient.setFollowRedirect(true);
        httpClient.setConnectionTimeout(std::chrono::milliseconds(3000));
    }
} // namespace

StartupClient::StartupClient(StartupConfiguration config, std::shared_ptr<YandexIO::IDevice> device)
    : device_(std::move(device))
    , config_(std::move(config))
{
}

std::optional<ResponseData> StartupClient::tryGetBootstrapConfig(std::shared_ptr<YandexIO::IDevice> device, const std::string& startupUrl) {
    quasar::HttpClient httpClient("appmetricaStartup", device);
    initHttpClient(httpClient);
    httpClient.setRetriesCount(0);

    auto response = httpClient.get("startup", startupUrl, {{"Accept", "application/json"}});
    if (response.responseCode == 200) {
        return parseStartupResponse(response.body);
    } else {
        YIO_LOG_WARN("Non-200 response (" << response.body + ") from " << startupUrl << ": " << to_string(response.responseCode));
    }
    return std::nullopt;
}

ResponseData StartupClient::getStartupConfig(std::atomic_bool& threadStopped) {
    const std::string startupUrl = config_.startupRequestURL(config_.startupHost);
    YIO_LOG_DEBUG("Sending startup request to url: " << startupUrl);
    /* Make retries manually */

    bool isFirstRetry = true;
    while (!threadStopped) {
        try {
            auto result = tryGetBootstrapConfig(device_, startupUrl);
            if (result) {
                return *result;
            }
            isFirstRetry = false;
        } catch (const std::exception& e) {
            YIO_LOG_WARN("Can't get startup config: " << e.what());
        }
        std::unique_lock<std::mutex> lock(mutex_);
        // poor man's exponential backoff, prevent ddosing startup.mobile
        conditionVariable_.wait_for(lock, std::chrono::seconds(isFirstRetry ? FIRST_RETRY_PERIOD_SEC_ : RETRY_PERIOD_SEC_),
                                    [&]() { return threadStopped.load(); });
    }

    // User thread is stopped. Return empty config
    return ResponseData();
}

std::unique_ptr<ReportConfiguration> StartupClient::getReportConfigFromBootstrap(std::shared_ptr<YandexIO::IDevice> device, const ResponseData& bootstrap, const StartupConfiguration& config) {
    quasar::HttpClient httpClient("appmetricaStartup", device);
    initHttpClient(httpClient);
    /* Make only one retry, do not ddos metrica with retries */
    httpClient.setRetriesCount(1);
    std::vector<std::string> reportHosts = bootstrap.reportHosts;

    // startup config contains urls to other startup config with different report urls
    // we have to save all report urls from all startup configs
    for (const auto& startupHost : bootstrap.startupHosts) {
        const std::string startupUrl = config.startupRequestURL(startupHost);
        YIO_LOG_DEBUG("Sending startup request to url: " << startupUrl);
        try {
            auto response = httpClient.get("startup", startupUrl, {{"Accept", "application/json"}});
            if (response.responseCode == 200) {
                auto responseData = parseStartupResponse(response.body);
                for (const auto& host : responseData.reportHosts) {
                    // different startup configs may have the same report hosts
                    // to avoid host duplicating, check if host wasn't added yet
                    if (std::find(reportHosts.cbegin(), reportHosts.cend(), host) == reportHosts.cend()) {
                        reportHosts.push_back(host);
                    }
                }
            } else {
                YIO_LOG_WARN("Non-200 response (" << response.body << ") from " << startupHost << ": " << to_string(response.responseCode));
            }
        } catch (const std::exception& e) {
            YIO_LOG_WARN("Can't get report config: " << e.what());
        }
    }
    const std::string& uuid = bootstrap.UUID.empty() ? config.UUID : bootstrap.UUID;
    return std::make_unique<ReportConfiguration>(std::time(nullptr), reportHosts, config, uuid, config.deviceID);
}

std::unique_ptr<ReportConfiguration> StartupClient::getReportConfig(std::atomic_bool& threadStopped) {
    const auto startupConfig = getStartupConfig(threadStopped);

    if (threadStopped) {
        /* User thread is stopped. Return empty config */
        return std::make_unique<ReportConfiguration>(std::time(nullptr), std::vector<std::string>{}, *device_);
    }
    return getReportConfigFromBootstrap(device_, startupConfig, config_);
}

ResponseData StartupClient::parseStartupResponse(const std::string& httpResponse) {
    Json::Value parsedResponse;
    Json::Reader reader;
    ResponseData responseData{};
    YIO_LOG_DEBUG("Raw startup response: " << httpResponse);

    reader.parse(httpResponse, parsedResponse);

    try {
        if (parsedResponse.isMember("uuid")) {
            responseData.UUID = parsedResponse["uuid"]["value"].asString();
        }
    } catch (const Json::Exception& exception) {
        YIO_LOG_ERROR_EVENT("StartupClient.ErrorParseUUID", "Cannot parse uuid value. " << exception.what() << ". Raw startup response: " << httpResponse);
    }

    try {
        for (const auto& reportUrl : parsedResponse["query_hosts"]["list"]["report"]["urls"]) {
            responseData.reportHosts.push_back(reportUrl.asString());
        }
    } catch (const Json::Exception& exception) {
        YIO_LOG_ERROR_EVENT("StartupClient.ErrorParseReportUrls", "Cannot parse report urls. " << exception.what() << ". Raw startup response: " << httpResponse);
    }

    try {
        for (const auto& startupUrl : parsedResponse["query_hosts"]["list"]["startup"]["urls"]) {
            responseData.startupHosts.push_back(startupUrl.asString());
        }
    } catch (const Json::Exception& exception) {
        YIO_LOG_ERROR_EVENT("StartupClient.ErrorParseStartupUrls", "Cannot parse startup urls. " << exception.what() << ". Raw startup response: " << httpResponse);
    }

    return responseData;
}
