#include "hour_range.h"

#include <yandex_io/libs/errno/errno_exception.h>

#include <cerrno>
#include <cstring>
#include <ctime>
#include <stdexcept>

using namespace quasar;

HourRange::HourRange(int minHour, int maxHour, int utcTimezoneShiftHours)
    : minHour_(minHour)
    , maxHour_(maxHour)
    , utcTimezoneShiftHours_(utcTimezoneShiftHours)
{
    // No operations.
}

HourRange::HourRange(int minHour, int maxHour)
    : minHour_(minHour)
    , maxHour_(maxHour)
    , useSystemTime_(true)
{
    // No operations.
}

void HourRange::setTimezoneShiftHours(const int shiftHours)
{
    useSystemTime_ = false;
    utcTimezoneShiftHours_ = shiftHours;
}

bool HourRange::containsUtcHour(int utcHour) const {
    const int utcMinHour = getShiftedUtcMinHour();
    const int utcMaxHour = getShiftedUtcMaxHour();

    if (utcMinHour <= utcMaxHour) // 15:00 - 20:00 for example
    {
        return (utcHour >= utcMinHour) && (utcHour <= utcMaxHour);
    } else // 23:00 - 07:00 for example
    {
        return (utcHour >= utcMinHour) || (utcHour <= utcMaxHour);
    }
    return true;
}

bool HourRange::containsTimestamp(TimePoint timestamp) const {
    std::tm dateTime;
    time_t t = std::chrono::system_clock::to_time_t(timestamp);
    if (!gmtime_r(&t, &dateTime)) {
        throw quasar::ErrnoException(errno, "problem with gmtime_r");
    }
    return containsUtcHour(dateTime.tm_hour);
}

int HourRange::getTimezoneShiftHours() const {
    if (useSystemTime_)
    {
        time_t t = time(nullptr);
        struct tm lt;
        memset(&lt, 0, sizeof(lt));
        localtime_r(&t, &lt);

        return int(lt.tm_gmtoff / 3600);
    } else {
        return utcTimezoneShiftHours_;
    }
}

int HourRange::getShiftedUtcMinHour() const {
    int utcMinHour = minHour_ - getTimezoneShiftHours();
    if (utcMinHour < 0) {
        utcMinHour += 24;
    }
    if (utcMinHour > 24) {
        utcMinHour -= 24;
    }

    return utcMinHour;
}

int HourRange::getShiftedUtcMaxHour() const {
    int utcMaxHour = maxHour_ - getTimezoneShiftHours();
    if (utcMaxHour < 0) {
        utcMaxHour += 24;
    }

    if (utcMaxHour > 24) {
        utcMaxHour -= 24;
    }

    return utcMaxHour;
}
