#include "linux_timer.h"

#include <yandex_io/libs/errno/errno_exception.h>

#include <cstring>
#include <errno.h>
#include <time.h>

using namespace quasar;

LinuxTimer::LinuxTimer(std::function<void()> function)
    : function_(std::move(function))
{
    struct sigevent sigevent;
    memset(&sigevent, 0, sizeof(sigevent));

    sigevent.sigev_notify = SIGEV_THREAD;
    sigevent.sigev_notify_function = LinuxTimer::callback;
    sigevent.sigev_value.sival_ptr = this;

    if (timer_create(CLOCK_MONOTONIC, &sigevent, &timerId_)) {
        throw ErrnoException(errno, "Can't create timer");
    }
}

void LinuxTimer::start(std::chrono::milliseconds time) {
    struct itimerspec itimerspec;
    const int64_t ticksMs = time.count();
    itimerspec.it_value.tv_sec = ticksMs / 1000;
    itimerspec.it_value.tv_nsec = NANO_IN_MILLI * (ticksMs % 1000);
    itimerspec.it_interval.tv_sec = 0;
    itimerspec.it_interval.tv_nsec = 0;
    if (timer_settime(timerId_, 0, &itimerspec, nullptr)) {
        throw ErrnoException(errno, "Can't start timer");
    }
}

void LinuxTimer::startPeriodic(std::chrono::milliseconds time) {
    struct itimerspec itimerspec;
    const int64_t ticksMs = time.count();
    itimerspec.it_value.tv_sec = ticksMs / 1000;
    itimerspec.it_value.tv_nsec = NANO_IN_MILLI * (ticksMs % 1000);
    itimerspec.it_interval.tv_sec = ticksMs / 1000;
    itimerspec.it_interval.tv_nsec = NANO_IN_MILLI * (ticksMs % 1000);
    if (timer_settime(timerId_, 0, &itimerspec, nullptr)) {
        throw ErrnoException(errno, "Can't start timer");
    }
}

void LinuxTimer::startPeriodic(std::chrono::milliseconds firstRunTime, std::chrono::milliseconds time) {
    struct itimerspec itimerspec;

    const int64_t firstTicksMs = firstRunTime.count();
    const int64_t ticksMs = time.count();

    itimerspec.it_value.tv_sec = firstTicksMs / 1000;
    itimerspec.it_value.tv_nsec = NANO_IN_MILLI * (firstTicksMs % 1000);
    itimerspec.it_interval.tv_sec = ticksMs / 1000;
    itimerspec.it_interval.tv_nsec = NANO_IN_MILLI * (ticksMs % 1000);
    if (timer_settime(timerId_, 0, &itimerspec, nullptr)) {
        throw ErrnoException(errno, "Can't start timer");
    }
}

void LinuxTimer::stop() {
    struct itimerspec itimerspec;
    itimerspec.it_value.tv_sec = 0;
    itimerspec.it_value.tv_nsec = 0;
    itimerspec.it_interval.tv_sec = 0;
    itimerspec.it_interval.tv_nsec = 0;
    if (timer_settime(timerId_, 0, &itimerspec, nullptr)) {
        throw ErrnoException(errno, "Can't stop timer");
    }
}

void LinuxTimer::callback(union sigval data) noexcept {
    auto timerPtr = (LinuxTimer*)(data.sival_ptr);
    timerPtr->function_();
}

LinuxTimer::~LinuxTimer() {
    timer_delete(timerId_);
}
