#pragma once

#include <chrono>
#include <functional>
#include <signal.h>

namespace quasar {

    /**
     * This is C++ wrapper of linux timer
     */
    class LinuxTimer {
    public:
        /**
         * Creates a timer object, which will execute a function after timeout after calling start() method.
         * @throw quasar::ErrnoException - error when calling timer_create.
         * @param function - function that will be executed.
         */
        explicit LinuxTimer(std::function<void()> function);

        LinuxTimer(const LinuxTimer&) = delete;
        LinuxTimer(LinuxTimer&&) = delete;
        LinuxTimer& operator=(const LinuxTimer&) = delete;
        LinuxTimer& operator=(LinuxTimer&&) = delete;

        /**
         * Starts an alarm. If alarm was already started it just restarts timer WITHOUT running a callback.
         * @throw quasar::ErrnoException - error when calling timer_settime.
         * @param time - timeout after which callback is called.
         */
        void start(std::chrono::milliseconds time);

        /**
         * Starts periodic alarm. This alarm executes callback not once, but until is stopped.
         * If it was previously started (no matter with start() or startPeriodic()) it just resets it.
         * @throw quasar::ErrnoException - error when calling timer_settime.
         * @param time - period of execution.
         */
        void startPeriodic(std::chrono::milliseconds time);
        /**
         * Starts periodic alarm. This alarm executes callback not once, but until is stopped.
         * If it was previously started (no matter with start() or startPeriodic()) it just resets it.
         * @throw quasar::ErrnoException - error when calling timer_settime.
         * @param firstRunTime - first callback will be called after this time.
         * @param time - period of execution of later callbacks.
         */
        void startPeriodic(std::chrono::milliseconds firstRunTime, std::chrono::milliseconds time);

        /**
         * Stops timer if it's running now. Has no effect if timer is not running now.
         * @throw quasar::ErrnoException - error when calling timer_settime.
         */
        void stop();
        /**
         * Deletes timer before destroying.
         * @throw quasar::ErrnoException - error when calling timer_delete.
         */
        ~LinuxTimer();

    private:
        std::function<void()> function_;

        static void callback(union sigval data) noexcept;

        timer_t timerId_;

        static constexpr int NANO_IN_MILLI = 1000000; // 1 ms = one million ns
    };

} // namespace quasar
