#include "persistent_file.h"
#include "utils.h"

#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

#include <openssl/md5.h>

#include <cerrno>
#include <chrono>
#include <cstddef>
#include <cstring>
#include <string>
#include <utility>

#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

YIO_DEFINE_LOG_MODULE("base");

using namespace quasar;

PersistentFile::PersistentFile(const std::string& filename, PersistentFile::Mode mode)
    : filename_(filename)
    , mode_(mode)
{
    open();
}

PersistentFile::~PersistentFile() {
    close();
}

void PersistentFile::open() {
    Y_VERIFY(!isOpen());

    /* rw-rw-rw- */
    const mode_t openMode{S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH};
    int flags{O_WRONLY | O_CREAT | O_SYNC};

    switch (mode_) {
        case Mode::APPEND:
            flags |= O_APPEND;
            break;

        case Mode::TRUNCATE:
            flags |= O_TRUNC;
            break;
    }

    fd_ = ::open(filename_.c_str(), flags, openMode);
    if (fd_ < 0) {
        YIO_LOG_ERROR_EVENT("PersistentFile.OpenFail", "Error opening " << filename_ << ": " << strError(errno));
        throw ErrnoException(errno, "Error opening file '" + filename_ + "'");
    }
}

void PersistentFile::close() {
    Y_VERIFY(isOpen());

    if (::close(fd_) < 0) {
        YIO_LOG_ERROR_EVENT("PersistentFile.CloseFail", "Error closing file: " << strError(errno));
    }

    fd_ = -1;
}

bool PersistentFile::isOpen() const {
    return fd_ >= 0;
}

bool PersistentFile::write(const std::string& buf) const {
    return write(buf.data(), buf.size());
}

bool PersistentFile::write(const void* buf, size_t size) const {
    Y_VERIFY(isOpen());

    ssize_t ret = ::write(fd_, buf, size);

    if (ret < 0 || (size_t)ret < size) {
        YIO_LOG_ERROR_EVENT("PersistentFile.WriteFail", "Writing failed: " << ret << " of " << size << " bytes : " << strError(errno));
        return false;
    }

    return true;
}

int PersistentFile::fd() const {
    return fd_;
}

std::string PersistentFile::filename() const {
    return filename_;
}

AtomicFile::AtomicFile(const std::string& filename)
    : filename_(filename)
    , tmpFilename_(filename + "_tmp")
{
    /* Crate temp file to write data */
    filePtr_ = std::make_unique<PersistentFile>(tmpFilename_, PersistentFile::Mode::TRUNCATE);
}

bool AtomicFile::write(const std::string& buf) const {
    return write(buf.data(), buf.size());
}

bool AtomicFile::write(const void* buf, size_t size) const {
    return filePtr_->write(buf, size);
}

AtomicFile::~AtomicFile()
{
    filePtr_.reset(nullptr);
    if (std::rename(tmpFilename_.c_str(), filename_.c_str()) != 0) {
        std::remove(tmpFilename_.c_str());
    }
}

std::atomic<int> TransactionFile::seqCounter_{0};
TransactionFile::TransactionFile(std::string filename)
    : TransactionFile(filename, makeUniqueName(filename))
{
}

TransactionFile::TransactionFile(std::string filename, std::string tmpFilename)
    : filename_(std::move(filename))
    , tmpFilename_(std::move(tmpFilename))
    , filePtr_(std::make_unique<PersistentFile>(tmpFilename_, PersistentFile::Mode::TRUNCATE))
{
}

TransactionFile::~TransactionFile()
{
    rollback();
}

bool TransactionFile::write(const std::string& buf) const {
    return write(buf.data(), buf.size());
}

bool TransactionFile::write(const void* buf, size_t size) const {
    if (!filePtr_) {
        throw std::runtime_error("File transaction closed");
    }
    return filePtr_->write(buf, size);
}

bool TransactionFile::commit()
{
    if (filePtr_) {
        if (std::rename(tmpFilename_.c_str(), filename_.c_str()) != 0) {
            return false;
        }

        filePtr_.reset();

        const auto dirName = getDirectoryName(filename_);
        int dirFd = ::open(dirName.c_str(), O_DIRECTORY | O_RDONLY);

        if (dirFd >= 0) {
            ::fsync(dirFd);
            ::close(dirFd);
        }

        return true;
    }
    return false;
}

void TransactionFile::rollback()
{
    if (filePtr_) {
        filePtr_.reset();
        std::remove(tmpFilename_.c_str());
    }
}

std::string TransactionFile::temporaryFilename() const {
    return tmpFilename_;
}

std::string TransactionFile::makeUniqueName(const std::string& filename)
{
    uint8_t hash[MD5_DIGEST_LENGTH];
    MD5_CTX md5ctx;
    MD5_Init(&md5ctx);
    auto now = std::chrono::steady_clock::now().time_since_epoch().count();
    int seq = ++seqCounter_;
    MD5_Update(&md5ctx, &now, sizeof(now));
    MD5_Update(&md5ctx, &seq, sizeof(seq));
    MD5_Final(hash, &md5ctx);
    std::string result;
    result.reserve(filename.size() + MD5_DIGEST_LENGTH + 1);
    result = filename;
    result += "_";

    const char* symbols = "0123456789abcdef";
    for (size_t i = 0; i < MD5_DIGEST_LENGTH; ++i) {
        result += symbols[hash[i] >> 4];
        result += symbols[hash[i] & 0x0F];
    }
    return result;
}
