#pragma once

#include <cstddef>
#include <memory>
#include <string>

namespace quasar {

    class PersistentFile {
    public:
        enum class Mode {
            APPEND,
            TRUNCATE,
        };

        PersistentFile(const std::string& filename, Mode mode);
        virtual ~PersistentFile();

        PersistentFile(const PersistentFile&) = delete;
        PersistentFile& operator=(const PersistentFile&) = delete;

    public:
        bool write(const std::string& buf) const;
        bool write(const void* buf, size_t size) const;

        int fd() const;
        std::string filename() const;

    protected:
        void open();
        void close();

        bool isOpen() const;

    protected:
        int fd_{-1};
        std::string filename_;
        Mode mode_;
    };

    /**
     * @brief While AtomicFile exist it write data into tmp file. In destructor AtomicFile move this file to target filename.
     *        This approach guarantee that other threads or processes will read file with all content written
     *        AtomicFile always rewrite file fully
     */
    class AtomicFile {
    public:
        AtomicFile(const std::string& filename);
        ~AtomicFile();

        bool write(const std::string& buf) const;
        bool write(const void* buf, size_t size) const;

    private:
        std::unique_ptr<PersistentFile> filePtr_;
        const std::string filename_;
        const std::string tmpFilename_;
    };

    class TransactionFile {
    public:
        TransactionFile(std::string filename);
        TransactionFile(std::string filename, std::string tmpFilename);
        ~TransactionFile();

        bool write(const std::string& buf) const;
        bool write(const void* buf, size_t size) const;
        bool commit();
        void rollback();
        std::string temporaryFilename() const;

    private:
        static std::string makeUniqueName(const std::string& filename);

    private:
        static std::atomic<int> seqCounter_;
        const std::string filename_;
        const std::string tmpFilename_;
        std::unique_ptr<PersistentFile> filePtr_;
    };

} // namespace quasar
