#include "utils.h"
#include "crc32.h"

#include "persistent_file.h"

#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/logging/logging.h>

#include <boost/algorithm/string.hpp>

#include <contrib/libs/uuid/include/uuid/uuid.h>

#include <json/json.h>

#include <util/folder/path.h>
#include <util/generic/scope.h>
#include <util/stream/mem.h>
#include <util/stream/str.h>
#include <util/stream/zlib.h>
#include <util/system/shellcommand.h>

#include <climits>
#include <cstdlib>
#include <fstream>
#include <random>
#include <regex>
#include <thread>
#include <unordered_map>

#include <dirent.h>
#include <poll.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <zlib.h>

YIO_DEFINE_LOG_MODULE("base");

using namespace quasar;

namespace quasar {

    std::string trim(const std::string& s) {
        int start = 0;
        int len = s.size();
        int end = s.size() - 1;
        while (start < len && std::isspace(s[start])) {
            ++start;
        }
        while (end >= 0 && std::isspace(s[end])) {
            --end;
        }
        return s.substr(start, end - start + 1);
    }

    bool fileExists(const std::string& fileName)
    {
        struct stat buffer {};
        return (stat(fileName.c_str(), &buffer) == 0);
    }

    std::string getPermissionsString(const mode_t& permissions) {
        std::string result(9, '-');
        result[0] = (permissions & S_IRUSR) ? 'r' : '-';
        result[1] = (permissions & S_IWUSR) ? 'w' : '-';
        result[2] = (permissions & S_IXUSR) ? 'x' : '-';
        result[3] = (permissions & S_IRGRP) ? 'r' : '-';
        result[4] = (permissions & S_IWGRP) ? 'w' : '-';
        result[5] = (permissions & S_IXGRP) ? 'x' : '-';
        result[6] = (permissions & S_IROTH) ? 'r' : '-';
        result[7] = (permissions & S_IWOTH) ? 'w' : '-';
        result[8] = (permissions & S_IXOTH) ? 'x' : '-';

        return result;
    }

    std::string getFileContent(const std::string& fileName)
    {
        std::ifstream file(fileName);
        if (!file.good()) {
            throw std::runtime_error("Cannot open file '" + fileName + "'.");
        }
        std::string content((std::istreambuf_iterator<char>(file)), std::istreambuf_iterator<char>());
        return content;
    }

    std::string getFileTail(const std::string& fileName, int64_t tailSize)
    {
        std::ifstream file(fileName);
        if (!file.good()) {
            throw std::runtime_error("Cannot open file '" + fileName + "'.");
        }
        file.seekg(-tailSize, std::ifstream::end);
        std::string content((std::istreambuf_iterator<char>(file)), std::istreambuf_iterator<char>());

        if ((int64_t)content.size() > tailSize) {
            // somehow this is happening -- probably file grows while we read it
            content.resize(tailSize);
        }

        Y_VERIFY((int64_t)content.size() <= tailSize);

        return content;
    }

    std::string makeUUID()
    {
        uuid_t uuid;
        uuid_generate_random(uuid);
        char cuuid[37];
        uuid_unparse_lower(uuid, cuuid);
        return std::string(cuuid);
    }

    bool isUUID(const std::string& str)
    {
        if (str.length() != 36) {
            return false;
        }
        for (size_t i = 0; i < str.length(); ++i)
        {
            if (8 == i || 13 == i || 18 == i || 23 == i)
            {
                if ('-' != str[i]) {
                    return false;
                }
                continue;
            }
            if (!std::isxdigit(str[i])) {
                return false;
            }
        }

        return true;
    }

    namespace {
        const std::string base64Chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                                        "abcdefghijklmnopqrstuvwxyz"
                                        "0123456789+/";

        inline bool isBase64(unsigned char c)
        {
            return (isalnum(c) || (c == '+') || (c == '/'));
        }

        std::vector<size_t> getBase64CharacterMap()
        {
            std::vector<size_t> result;
            result.resize(256);
            for (size_t i = 0; i < base64Chars.size(); ++i) {
                result[base64Chars[i]] = i;
            }

            return result;
        }

    } // anonymous namespace

    std::string base64Encode(const char* bytesToEncode, unsigned int inLen)
    {
        std::string ret;
        int i = 0;
        unsigned char charArray3[3];
        unsigned char charArray4[4];

        ret.reserve((inLen * 4) / 3 + 5);
        while (inLen--)
        {
            charArray3[i++] = *(bytesToEncode++);
            if (3 == i)
            {
                charArray4[0] = (charArray3[0] & 0xfc) >> 2;
                charArray4[1] = ((charArray3[0] & 0x03) << 4) + ((charArray3[1] & 0xf0) >> 4);
                charArray4[2] = ((charArray3[1] & 0x0f) << 2) + ((charArray3[2] & 0xc0) >> 6);
                charArray4[3] = charArray3[2] & 0x3f;

                for (i = 0; (i < 4); i++) {
                    ret += base64Chars[charArray4[i]];
                }
                i = 0;
            }
        }

        if (i != 0)
        {
            for (int j = i; j < 3; j++) {
                charArray3[j] = '\0';
            }

            charArray4[0] = (charArray3[0] & 0xfc) >> 2;
            charArray4[1] = ((charArray3[0] & 0x03) << 4) + ((charArray3[1] & 0xf0) >> 4);
            charArray4[2] = ((charArray3[1] & 0x0f) << 2) + ((charArray3[2] & 0xc0) >> 6);
            charArray4[3] = charArray3[2] & 0x3f;

            for (int j = 0; (j < i + 1); j++) {
                ret += base64Chars[charArray4[j]];
            }

            while ((i++ < 3)) {
                ret += '=';
            }
        }

        return ret;
    }

    std::string base64EncodeFile(const std::string& path) {
        const std::string content = quasar::getFileContent(path);
        return quasar::base64Encode(content.c_str(), content.length());
    }

    std::string base64Decode(std::string_view encodedString)
    {
        static std::vector<size_t> characterMap = getBase64CharacterMap();

        int inLen = encodedString.size();
        int i = 0;
        int in_ = 0;
        unsigned char charArray4[4];
        unsigned char charArray3[3];
        std::string ret;
        ret.reserve((inLen * 3) / 4 + 6);

        while (inLen-- && (encodedString[in_] != '=') && isBase64(encodedString[in_]))
        {
            charArray4[i++] = encodedString[in_];
            in_++;
            if (4 == i)
            {
                for (i = 0; i < 4; i++) {
                    charArray4[i] = characterMap[charArray4[i]]; // base64Chars.find(charArray4[i]);
                }

                charArray3[0] = (charArray4[0] << 2) + ((charArray4[1] & 0x30) >> 4);
                charArray3[1] = ((charArray4[1] & 0xf) << 4) + ((charArray4[2] & 0x3c) >> 2);
                charArray3[2] = ((charArray4[2] & 0x3) << 6) + charArray4[3];

                for (i = 0; (i < 3); i++) {
                    ret += charArray3[i];
                }
                i = 0;
            }
        }

        if (i != 0)
        {
            for (int j = i; j < 4; j++) {
                charArray4[j] = 0;
            }

            for (int j = 0; j < 4; j++) {
                charArray4[j] = characterMap[charArray4[j]]; //  base64Chars.find(charArray4[j]);
            }

            charArray3[0] = (charArray4[0] << 2) + ((charArray4[1] & 0x30) >> 4);
            charArray3[1] = ((charArray4[1] & 0xf) << 4) + ((charArray4[2] & 0x3c) >> 2);
            charArray3[2] = ((charArray4[2] & 0x3) << 6) + charArray4[3];

            for (int j = 0; (j < i - 1); j++) {
                ret += charArray3[j];
            }
        }

        return ret;
    }

    namespace {

        constexpr unsigned int KB = (1 << 10);
        constexpr unsigned int MB = (KB << 10);
        constexpr unsigned int GB = (MB << 10);

        std::string char2hex(char dec)
        {
            char dig1 = (dec & 0xF0) >> 4;
            char dig2 = (dec & 0x0F);
            if (0 <= dig1 && dig1 <= 9) {
                dig1 += 48; // 0,48 in ascii
            }
            if (10 <= dig1 && dig1 <= 15) {
                dig1 += 65 - 10; // A,65 in ascii
            }
            if (0 <= dig2 && dig2 <= 9) {
                dig2 += 48;
            }
            if (10 <= dig2 && dig2 <= 15) {
                dig2 += 65 - 10;
            }

            std::string r;
            r.append(&dig1, 1);
            r.append(&dig2, 1);
            return r;
        }

    } // anonymous namespace

    std::string urlEncode(const std::string& c)
    {
        std::string escaped;
        int max = c.length();
        for (int i = 0; i < max; i++)
        {
            if ((48 <= c[i] && c[i] <= 57) ||  // 0-9
                (65 <= c[i] && c[i] <= 90) ||  // ABC...XYZ
                (97 <= c[i] && c[i] <= 122) || // abc...xyz
                (c[i] == '~' || c[i] == '-' || c[i] == '_' || c[i] == '.'))
            {
                escaped.append(&c[i], 1);
            } else {
                escaped.append("%");
                escaped.append(char2hex(c[i])); // converts char 255 to string "FF"
            }
        }
        return escaped;
    }

    std::string urlDecode(const std::string& in)
    {
        std::string out;
        out.reserve(in.size());
        for (std::size_t i = 0; i < in.size(); ++i)
        {
            if (in[i] == '%')
            {
                if (i + 3 <= in.size())
                {
                    int value = 0;
                    std::istringstream is(in.substr(i + 1, 2));
                    if (is >> std::hex >> value)
                    {
                        out += static_cast<char>(value);
                        i += 2;
                    } else {
                        throw std::runtime_error(
                            "Cannot URL decode string '" + in + "'. Cannot read hex at " + std::to_string(i + 1) + " position.");
                    }
                } else {
                    throw std::runtime_error(
                        "Cannot URL decode string '" + in + "'. Unexpected end of string.");
                }
            } else if (in[i] == '+')
            {
                out += ' ';
            } else {
                out += in[i];
            }
        }

        return out;
    }

    void directoryForEach(const std::string& directory, std::function<void(std::string_view, unsigned char)> callback) {
        DIR* dp = opendir(directory.c_str());
        if (nullptr == dp)
        {
            throw ErrnoException(errno, "Cannot open directory '" + directory + "'");
        }
        Y_DEFER {
            closedir(dp);
        };

        struct dirent* ep;

        while ((ep = readdir(dp)))
        {
            std::string_view name = ep->d_name;
            if (name != "." && name != "..") {
                callback(name, ep->d_type);
            }
        }
    }

    std::vector<std::string> getDirectoryFileList(const std::string& directory)
    {
        std::vector<std::string> result;
        directoryForEach(directory, [&result](auto name, auto /*type*/) {
            result.emplace_back(name);
        });
        return result;
    }

    std::vector<std::string> getDirectoryFileListRecursive(const std::string& directory)
    {
        std::vector<std::string> result;
        directoryForEach(directory, [&result, &directory](auto name, auto type) {
            std::string fullPath = directory + "/";
            fullPath += name;
            if (type == DT_REG) {
                result.emplace_back(fullPath);
            } else if (type == DT_LNK) {
                struct stat st;
                stat(fullPath.c_str(), &st);
                if ((st.st_mode & S_IFMT) == S_IFREG) {
                    result.emplace_back(fullPath);
                } else {
                    YIO_LOG_WARN("Symbolic link \"" << fullPath << "\" is not a regular file, skipping");
                }
            } else if (type == DT_DIR) {
                auto recursiveResult = getDirectoryFileListRecursive(fullPath);
                result.insert(result.end(),
                              std::make_move_iterator(recursiveResult.begin()),
                              std::make_move_iterator(recursiveResult.end()));
            }
        });

        return result;
    }

    std::string urlToFilePath(const std::string& url) {
        std::string output;
        output = std::regex_replace(url, std::regex(R"(,)"), "  ");
        output = std::regex_replace(output, std::regex(R"(/)"), ",");
        output = std::regex_replace(output, std::regex(R"(\?)"), ", 1");
        output = std::regex_replace(output, std::regex(R"(\*)"), ", 2");
        return output;
    }

    std::string getFileName(const std::string& path)
    {
        std::vector<std::string> components;
        boost::iter_split(components, path, boost::first_finder("/"));
        if (components.empty())
        {
            throw std::runtime_error("Cannot get file name from path " + path);
        }

        return components.back();
    }

    std::string getDirectoryName(const std::string& path)
    {
        std::vector<std::string> components;

        boost::iter_split(components, path, boost::first_finder("/"));
        if (components.empty()) {
            throw std::runtime_error("Cannot get file name from path " + path);
        }

        components.resize(components.size() - 1);

        return boost::algorithm::join(components, "/");
    }

    std::vector<std::string> split(const std::string& string, const std::string& delimiter, size_t maxParts)
    {
        std::vector<std::string> result;
        if (maxParts == 1) {
            result.push_back(string);
            return result;
        }
        size_t pos = 0;
        size_t lastPos = 0;
        while ((pos = string.find(delimiter, pos)) != std::string::npos)
        {
            result.push_back(string.substr(lastPos, pos - lastPos));
            pos += delimiter.size();
            lastPos = pos;
            if (maxParts && result.size() >= maxParts - 1) {
                break;
            }
        }
        if (lastPos == string.size()) {
            result.emplace_back("");
        } else {
            result.push_back(string.substr(lastPos, string.size() - lastPos));
        }
        return result;
    }

    std::unordered_map<std::string, std::string> getUrlParams(const std::string& url) {
        std::unordered_map<std::string, std::string> result;

        auto startPos = url.find('?');
        if (startPos == std::string::npos || startPos == url.size() - 1) {
            return result;
        }
        auto paramsStrings = split(url.substr(startPos + 1, url.size() - startPos - 1), "&"); // TODO: this will break when anchor ("#") will be in url
        for (const auto& paramString : paramsStrings)
        {
            auto keyVal = split(paramString, "=");
            result[urlDecode(keyVal[0])] = urlDecode(keyVal[1]);
        }
        return result;
    }

    std::string addGETParam(const std::string& url, const std::string& key, const std::string& value, bool replaceIfExists /* = true */)
    {
        // NOTE: url may contain params with '?' started and '&' as delimiter, and also can contain anchor '#', so we have some variants
        // NOTE: we assume, that anchor is always after the params, otherwise we will break

        auto paramsStartPos = url.find('?');
        auto anchorStartPos = url.find('#');
        auto alreadyExistsPos = url.find(key + "=");

        if (alreadyExistsPos != std::string::npos) {
            if (!replaceIfExists) {
                return url;
            }

            auto delimiterPos = url.find('&', alreadyExistsPos);
            if (delimiterPos == std::string::npos) {
                delimiterPos = anchorStartPos;
            }
            return url.substr(0, alreadyExistsPos) + key + "=" + urlEncode(value) + (delimiterPos != std::string::npos ? url.substr(delimiterPos, url.size() - delimiterPos) : "");
        }

        auto paramsDelimiter = paramsStartPos == std::string::npos ? "?" : "&";

        if (anchorStartPos == std::string::npos)
        {
            return url + paramsDelimiter + key + "=" + urlEncode(value);
        } else {
            return url.substr(0, anchorStartPos) + paramsDelimiter + key + "=" + urlEncode(value) + url.substr(anchorStartPos, url.size() - anchorStartPos);
        }
    }

    int64_t getFileSize(const std::string& fileName)
    {
        std::ifstream in(fileName, std::ifstream::ate | std::ifstream::binary);
        if (!in)
        {
            throw ErrnoException(errno, "Cannot open file '" + fileName + "'");
        }
        return in.tellg();
    }

    bool isSuccessHttpCode(int code)
    {
        return (code >= 200 && code <= 299);
    }

    void convertFPSamplesToInt(std::vector<int16_t>& out, std::span<const float> processed, uint32_t outFreqDivider, float scale) {
        // (processed.size() / outFreqDivider) rounded up
        out.resize((processed.size() + outFreqDivider - 1) / outFreqDivider);
        float sampleMaxValue = 1 << (sizeof(int16_t) * 8 - 1);
        for (size_t i = 0; i < out.size(); i++) {
            float tmp = processed[i * outFreqDivider] * scale;
            out[i] = (int16_t)std::clamp(tmp, -sampleMaxValue, sampleMaxValue - 1);
        }
    }

    std::wstring utf8_to_utf16(const std::string& utf8)
    {
        std::vector<unsigned long> unicode;
        size_t i = 0;
        while (i < utf8.size())
        {
            unsigned long uni;
            size_t todo;
            unsigned char ch = utf8[i++];
            if (ch <= 0x7F)
            {
                uni = ch;
                todo = 0;
            } else if (ch <= 0xBF)
            {
                throw std::logic_error("not a UTF-8 string");
            } else if (ch <= 0xDF)
            {
                uni = ch & 0x1F;
                todo = 1;
            } else if (ch <= 0xEF)
            {
                uni = ch & 0x0F;
                todo = 2;
            } else if (ch <= 0xF7)
            {
                uni = ch & 0x07;
                todo = 3;
            } else {
                throw std::logic_error("not a UTF-8 string");
            }
            for (size_t j = 0; j < todo; ++j)
            {
                if (i == utf8.size()) {
                    throw std::logic_error("not a UTF-8 string");
                }
                unsigned char ch = utf8[i++];
                if (ch < 0x80 || ch > 0xBF) {
                    throw std::logic_error("not a UTF-8 string");
                }
                uni <<= 6;
                uni += ch & 0x3F;
            }
            if (uni >= 0xD800 && uni <= 0xDFFF) {
                throw std::logic_error("not a UTF-8 string");
            }
            if (uni > 0x10FFFF) {
                throw std::logic_error("not a UTF-8 string");
            }
            unicode.push_back(uni);
        }
        std::wstring utf16;
        for (size_t i = 0; i < unicode.size(); ++i)
        {
            unsigned long uni = unicode[i];
            if (uni <= 0xFFFF)
            {
                utf16 += (wchar_t)uni;
            } else {
                uni -= 0x10000;
                utf16 += (wchar_t)((uni >> 10) + 0xD800);
                utf16 += (wchar_t)((uni & 0x3FF) + 0xDC00);
            }
        }
        return utf16;
    }

    std::string gzipCompress(const std::string& input)
    {
        constexpr unsigned int gzipMaxBytes = 2 * GB;
        constexpr int windowBits = 15 + 16; /* gzip with windowbits of 15 */
        constexpr int memLevel = 8;

        const size_t size = input.length();

        std::string output;
        z_stream stream;

        if (size > std::numeric_limits<decltype(stream.avail_in)>::max()) {
            throw std::runtime_error("Size arg is too large to fit into unsigned int type");
        }

        if (size > gzipMaxBytes) {
            throw std::runtime_error("Size may use more memory than intended when decompressing");
        }

        stream.zalloc = Z_NULL;
        stream.zfree = Z_NULL;
        stream.opaque = Z_NULL;
        stream.avail_in = 0;
        stream.next_in = Z_NULL;

        if (deflateInit2(&stream,
                         Z_DEFAULT_COMPRESSION,
                         Z_DEFLATED,
                         windowBits,
                         memLevel,
                         Z_DEFAULT_STRATEGY) != Z_OK)
        {
            throw std::runtime_error("Deflate init failed");
        }

        stream.next_in = (Bytef*)(input.data());
        stream.avail_in = static_cast<decltype(stream.avail_in)>(size);

        size_t sizeCompressed = 0;
        do {
            size_t increase = size / 2 + 1024;

            if (output.size() < (sizeCompressed + increase)) {
                output.resize(sizeCompressed + increase);
            }

            stream.avail_out = static_cast<unsigned int>(increase);
            stream.next_out = reinterpret_cast<Bytef*>((&output[0] + sizeCompressed));

            deflate(&stream, Z_FINISH);
            sizeCompressed += (increase - stream.avail_out);
        } while (stream.avail_out == 0);

        deflateEnd(&stream);
        output.resize(sizeCompressed);

        return output;
    }

    std::string gzipDecompress(const std::string& input) {
        TMemoryInput in(input.data(), input.size());
        TZLibDecompress d(&in);

        TString output;
        TStringOutput out(output);

        TransferData(&d, &out);

        return output;
    }

    std::string executeWithOutput(const char* command)
    {
        std::string result;
        errno = 0;
        std::unique_ptr<FILE, int (*)(FILE*)> pipe(::popen(command, "r"), &pclose);
        if (pipe) {
            char buf[1024];
            size_t readChars;
            while ((readChars = fread(buf, sizeof(char), 1024, pipe.get()))) {
                result.append(buf, readChars);
            }
        } else {
            if (errno) {
                throw quasar::ErrnoException(errno, std::string("executeWithOutput(") + command + ")");
            } else {
                throw std::runtime_error(std::string("couldn't open pipe when executing command ") + command);
            }
        }
        return result;
    }

    bool tryUntilSuccess(std::function<bool()> f, std::chrono::duration<float, std::milli> baseSleepFor, std::chrono::duration<float, std::milli> maxSleepFor, unsigned int maxRetriesCount, float backoffFactor) {
        bool result = f();
        auto sleepTime = baseSleepFor;
        unsigned int i = 0;
        while (!result && i < maxRetriesCount) {
            std::this_thread::sleep_for(sleepTime);
            const auto newSleepFor = sleepTime * backoffFactor;
            sleepTime = newSleepFor > maxSleepFor ? maxSleepFor : newSleepFor;
            result = f();
            i++;
        }
        return result;
    }

    void runOncePerBoot(std::function<bool()> f, const std::string& guardFilePath) {
        if (!fileExists(guardFilePath)) {
            f();
            try {
                PersistentFile file(guardFilePath, PersistentFile::Mode::TRUNCATE);
            } catch (const std::runtime_error& e) {
                YIO_LOG_ERROR_EVENT("RunOncePerBoot.CreateGuardFail", "Unable to create guardFile " << guardFilePath);
            }
        }
    }

    std::string maskToken(std::string_view input) {
        /* https://wiki.yandex-team.ru/security/for/web-developers/#logiichtovnixnelzjapisat */
        const size_t halfLength = input.size() / 2;
        const auto leaveStr = input.substr(0, halfLength);
        const std::string asterisks(input.size() - halfLength, '*');
        return std::string(leaveStr) + asterisks;
    }

    void extractTargzArchive(const std::string& archivePath, const std::string& destPath) {
        TFsPath(destPath).MkDirs();
        const std::string command = "zcat " + archivePath + " | tar -xvf - -C" + destPath;
        if (std::system(command.c_str())) {
            throw std::runtime_error("Failed to extract spotter archive " + archivePath + " to " + destPath);
        }
    }

    int64_t getNowTimestampMs()
    {
        const auto now = std::chrono::system_clock::now();
        return static_cast<int64_t>(std::chrono::duration_cast<std::chrono::milliseconds>(now.time_since_epoch()).count());
    }

    int16_t javaStyleStringHash(const std::string& s)
    {
        auto sUtf8 = utf8_to_utf16(s);
        int64_t h = 0;
        for (auto c : sUtf8)
        {
            h = 31 * h + c;
            h &= 0xFFFFFFFF;
        }
        return (int16_t)((int32_t)h % std::numeric_limits<int16_t>::max());
    }

    std::string bytesToString(const std::vector<unsigned char>& bytes, int offset, int length)
    {
        if (length == -1)
        {
            length = bytes.size();
        }
        std::string result;
        for (int i = 0; i < length; ++i)
        {
            result += (char)bytes[offset + i];
        }
        return result;
    }

    std::vector<unsigned char> stringToBytes(const std::string& s)
    {
        return std::vector<unsigned char>(s.begin(), s.end());
    }

    std::string rndMark(size_t N) {
        constexpr char table[] = "0123456789QWERTYUIOPASDFGHJKLZXCVBNMqwertyuiopasdfghjklzxcvbnm";
        constexpr auto size = sizeof(table) - 1;
        static std::mt19937 generator((std::random_device())());
        std::uniform_int_distribution<size_t> d(0, size - 1);
        std::ostringstream stream;
        for (size_t i = 0; i < N; ++i) {
            stream << table[d(generator)];
        }
        return stream.str();
    }

    uint32_t getRandomSeed(const std::string& deviceId) {
        const auto nowSinceEpoch = std::chrono::system_clock::now().time_since_epoch();
        const uint64_t sinceEpochMs = std::chrono::duration_cast<std::chrono::milliseconds>(nowSinceEpoch).count();

        return getCrc32(deviceId) + sinceEpochMs;
    }
} // namespace quasar
