#pragma once

#include <chrono>
#include <functional>
#include <iomanip>
#include <memory>
#include <optional>
#include <sstream>
#include <string>
#include <string_view>
#include <unordered_map>
#include <vector>

#include <span>

#include <util/string/hex.h>

namespace quasar {

    bool fileExists(const std::string& fileName);
    std::string getPermissionsString(const mode_t& permissions);

    std::string trim(const std::string& s);

    std::string getFileContent(const std::string& fileName);
    std::string getFileTail(const std::string& fileName, int64_t tailSize);

    std::string gzipCompress(const std::string& input);
    std::string gzipDecompress(const std::string& input);

    std::string makeUUID();
    bool isUUID(const std::string& str);

    std::string base64Encode(const char* bytesToEncode, unsigned int inLen);
    std::string base64Decode(std::string_view encodedString);
    std::string base64EncodeFile(const std::string& path);

    std::string urlEncode(const std::string& c);
    std::string urlDecode(const std::string& in);

    /**
     * execute callback for every item in a directory.
     * Second parameter of callback is 'dirent::d_type'
     * It can be checked for DT_DIR, DT_REG etc
     * This function will exclude '.' and '..' entries
     */
    void directoryForEach(const std::string& directory, std::function<void(std::string_view, unsigned char)> callback);

    std::vector<std::string> getDirectoryFileList(const std::string& directory);
    /**
     * Returns list of all files in directory, including files in subdirectories.
     * The difference from getDirectoryFileList is that all file names returned here start with directory name.
     * It returns only regular file names, without directories, soft links, device files etc.
     * NOTE: it doesn't follow directories, which are symbolic links.
     * Only symlinks which are supposed to be in observed directories are links to regular files.
     * @param directory directory from which files are read.
     * @return files list in directory
     */
    std::vector<std::string> getDirectoryFileListRecursive(const std::string& directory);

    std::string urlToFilePath(const std::string& url);

    std::string getFileName(const std::string& path);

    std::string getDirectoryName(const std::string& path);

    std::vector<std::string> split(const std::string& string, const std::string& delimiter = " ", size_t maxParts = 0);

    template <class T>
    std::string join(const T& args, const std::string& delimiter = "") {
        return join(args, delimiter, [](const decltype(*args.end())& v) -> const decltype(*args.end())& { return v; });
    }

    template <class T, class F>
    std::string join(const T& args, const std::string& delimiter, const F& trasform) {
        int index = 0;
        std::ostringstream result;
        for (const auto& arg : args) {
            if (index++) {
                result << delimiter;
            }
            result << trasform(arg);
        }
        return result.str();
    }

    std::unordered_map<std::string, std::string> getUrlParams(const std::string& url);
    std::string addGETParam(const std::string& url, const std::string& key, const std::string& value, bool replaceIfExists = true);

    int64_t getFileSize(const std::string& fileName);

    bool isSuccessHttpCode(int code);

    std::wstring utf8_to_utf16(const std::string& utf8);

    std::string executeWithOutput(const char* command);
    bool tryUntilSuccess(std::function<bool()> f, std::chrono::duration<float, std::milli> baseSleepFor, std::chrono::duration<float, std::milli> maxSleepFor, unsigned int maxRetriesCount, float backoffFactor);
    void runOncePerBoot(std::function<bool()> f, const std::string& guardFilePath);

    void convertFPSamplesToInt(std::vector<int16_t>& out, std::span<const float> processed, uint32_t outFreqDivider, float scale);

    std::string maskToken(std::string_view input);

    /**
     * Extracts .tar.gz to specified path
     * @param archivePath Archive path
     * @param destPath Destination path
     */
    void extractTargzArchive(const std::string& archivePath, const std::string& destPath);

    constexpr bool stringsEqual(char const* a, char const* b)
    {
        /* Found on stack overflow. Check that strings are equal in compile time */
        return *a == *b && (*a == '\0' || stringsEqual(a + 1, b + 1));
    }

    template <typename T>
    void clear(T& container) {
        T().swap(container);
    }

    int16_t javaStyleStringHash(const std::string& s);
    std::string bytesToString(const std::vector<unsigned char>& bytes, int offset = 0, int length = -1);
    std::vector<unsigned char> stringToBytes(const std::string& s);

    int64_t getNowTimestampMs();

    inline void normalize(std::vector<float>& samples, int sampleByteSize) {
        const float factor = (1LL << (sampleByteSize * 8 - 1));
        for (auto& elem : samples) {
            elem /= factor;
        }
    }

    template <typename It>
    void printRangeInStream(std::ostream& out, It begin, It end, const std::string& delimiter = ", ", const std::string& terminator = ".") {
        for (auto it = begin; it != end; ++it) {
            out << *it << (std::next(it) == end ? terminator : delimiter);
        }
    }

    template <typename T>
    class Hex {
    public:
        Hex(const T& value)
            : value_(value)
        {
        }

        template <typename U = T, std::enable_if_t<std::is_integral_v<std::decay_t<U>>, int> = 0>
        void print(std::ostream& os) const {
            const auto flags = os.flags();

            os << "0x" << std::hex << std::setfill('0') << std::setw(sizeof(U) * 2)
               << static_cast<std::uintmax_t>(static_cast<std::make_unsigned_t<U>>(value_));

            os.flags(flags);
        }

        template <typename U = T, std::enable_if_t<std::is_same_v<std::decay_t<decltype(*std::declval<U>().data())>, uint8_t>, int> = 0>
        void print(std::ostream& os) const {
            os << "0x" << HexEncode(value_.data(), value_.size());
        }

    private:
        const T& value_;
    };

    std::string rndMark(size_t N = 8);
    uint32_t getRandomSeed(const std::string& deviceId);

} // namespace quasar

template <typename T>
std::ostream& operator<<(std::ostream& out, const std::optional<T>& data)
{
    if (data) {
        out << *data;
    } else {
        out << "none";
    }
    return out;
}

template <typename T>
std::ostream& operator<<(std::ostream& os, const quasar::Hex<T>& hex) {
    hex.print(os);

    return os;
}

namespace quasar {
    template <typename... Args>
    std::string makeString(Args const&... args)
    {
        // auto s = makeString("Hello", ' ', std::string{"world"}, '!', " x=", 125);
        // result: s == "Hello world! x=125
        std::ostringstream stream;
        (void)(int[]){0, ((stream << args), 0)...};

        return stream.str();
    }
} // namespace quasar
