#include "bluetooth.h"

#include <sstream>

Bluetooth::Bluetooth(const std::string& btName)
    : name_(btName)
    , threadExists_(true)
{
    eventSenderThread_ = std::thread(&Bluetooth::eventSenderThread, this);
};

Bluetooth::~Bluetooth() {
    std::unique_lock<std::mutex> lock(eventQueueMutex_);
    threadExists_ = false;
    condVar_.notify_one();
    lock.unlock();
    eventSenderThread_.join();
}

void Bluetooth::registerEventListener(std::weak_ptr<BluetoothEventListener> listener) {
    std::scoped_lock guard(listenersMutex_);
    listeners_.emplace_back(listener);
}

void Bluetooth::sourceEventCallbackLocked(SourceEvent ev, EventResult res) {
    std::scoped_lock guard(eventQueueMutex_);
    sourceEventQueue_.push(std::make_pair(ev, std::move(res)));
    condVar_.notify_one();
}

void Bluetooth::sinkEventCallbackLocked(SinkEvent ev, EventResult res) {
    std::scoped_lock guard(eventQueueMutex_);
    sinkEventQueue_.push(std::make_pair(ev, std::move(res)));
    condVar_.notify_one();
}

void Bluetooth::baseEventCallbackLocked(BaseEvent ev, EventResult res) {
    std::scoped_lock guard(eventQueueMutex_);
    baseEventQueue_.push(std::make_pair(ev, std::move(res)));
    condVar_.notify_one();
}

void Bluetooth::eventSenderThread() {
    std::unique_lock<std::mutex> lock(eventQueueMutex_);
    while (threadExists_) {
        condVar_.wait(lock, [this]() -> bool {
            if (!baseEventQueue_.empty()) {
                return true;
            }
            if (!sourceEventQueue_.empty()) {
                return true;
            }
            if (!sinkEventQueue_.empty()) {
                return true;
            }
            return !threadExists_;
        });
        if (!threadExists_) {
            /* Thread should ends up*/
            break;
        }
        std::scoped_lock guard(listenersMutex_);

        while (!baseEventQueue_.empty()) {
            const auto eventPair = baseEventQueue_.front();
            baseEventQueue_.pop();
            lock.unlock();
            for (auto i = listeners_.begin(); i != listeners_.end();) {
                auto listenerPtr = i->lock();
                if (listenerPtr == nullptr) {
                    i = listeners_.erase(i);
                } else {
                    listenerPtr->onBaseEvent(eventPair.first, eventPair.second);
                    ++i;
                }
            }
            lock.lock();
        }

        while (!sourceEventQueue_.empty()) {
            const auto eventPair = sourceEventQueue_.front();
            sourceEventQueue_.pop();
            lock.unlock();
            for (auto i = listeners_.begin(); i != listeners_.end();) {
                auto listenerPtr = i->lock();
                if (listenerPtr == nullptr) {
                    i = listeners_.erase(i);
                } else {
                    listenerPtr->onSourceEvent(eventPair.first, eventPair.second);
                    ++i;
                }
            }
            lock.lock();
        }
        while (!sinkEventQueue_.empty()) {
            const auto eventPair = sinkEventQueue_.front();
            sinkEventQueue_.pop();
            lock.unlock();
            for (auto i = listeners_.begin(); i != listeners_.end();) {
                auto listenerPtr = i->lock();
                if (listenerPtr == nullptr) {
                    i = listeners_.erase(i);
                } else {
                    listenerPtr->onSinkEvent(eventPair.first, eventPair.second);
                    ++i;
                }
            }
            lock.lock();
        }
    } /* while(threadExists_) */
}

Json::Value Bluetooth::TrackInfo::toJson() const {
    Json::Value json(Json::objectValue);
    if (!title.empty()) {
        json["title"] = title;
    }
    if (!artist.empty()) {
        json["artist"] = artist;
    }
    if (!album.empty()) {
        json["album"] = album;
    }
    if (!genre.empty()) {
        json["genre"] = genre;
    }
    if (songLenMs != -1) {
        json["song_len_ms"] = songLenMs;
    }
    if (currPosMs != -1) {
        json["curr_pos_ms"] = currPosMs;
    }
    return json;
}
