#pragma once

#include <json/json.h>

#include <atomic>
#include <condition_variable>
#include <functional>
#include <list>
#include <mutex>
#include <queue>
#include <set>
#include <string>
#include <thread>
#include <utility>

/**
 * @brief Bluetooth is an interface for bluetooth implementation
 *        All interface methods should use callbacks to notify user about Success or Error of called
 *        operation. For each interface method there is a callback.
 *        There are 3 types of callbacks: BaseEvent's, SinkEvent's, SourceEvent's
 *        BaseEvent's   - are callbacks for base events like scanning, or disconnect_all
 *        SinkEvent's   - are callbacks (or requests) that should be used when local device is
 *                        working as a SINK bt device (stream in) and any
 *                        sink event/request happens. i.e.: Connected, disconnected, play, pause ...
 *        SourceEvent's - are callback for Source events. Local device is working as a Source (stream out)
 *                        bluetooth device. i.e.: paired, disconnected
 *
 *        It is possible that Bluetooth Device is working as a Sink and Source device at the same time.
 */
class Bluetooth {
public:
    /**
     * @brief Bluetooth constructor. Should initialize bluetooth. If error happens -> throw exception
     * @param btName The name of bluetooth device. This name should be visible for another BT devices
     */
    explicit Bluetooth(const std::string& btName);

    virtual ~Bluetooth();

    enum class BtRole {
        SINK,   /*!< SINK device - means that device is waiting for stream (input) */
        SOURCE, /*!< SOURCE device - means that device is stream out device */
        ALL,    /*!< And Sink and Source */
        UNKNOWN /*!< UNKNOWN - unknown device role */
    };
    /**
     * @brief BtNetwork struct defines the Bluetooth Network: Role, name , MAC address
     */
    struct BtNetwork {
        BtNetwork() = default;
        BtNetwork(std::string _name, std::string _addr, BtRole _role)
            : name(std::move(_name))
            , addr(std::move(_addr))
            , role(_role)
        {
        }
        std::string name;
        std::string addr;
        BtRole role{BtRole::UNKNOWN};

        /* std::set comparator */
        friend bool operator<(const BtNetwork& lhs, const BtNetwork& rhs) {
            return lhs.addr < rhs.addr;
        }

        friend bool operator==(const BtNetwork& lhs, const BtNetwork& rhs) {
            return lhs.addr == rhs.addr;
        }
    };

    /**
     * @brief request to change Bluetooth name (that should be visible by other bluetooth devices).
     * @param name - new name to set
     * @return - request result. At success return  0. Function should return value < 0 if error occurred
     */
    virtual int setName(const std::string& name) = 0;

    /**
     * @brief request for bluetooth networks scan. The bluetooth scan result should be
     *        returned by using baseEventCallback (baseEventCallbackLocked(BaseEvent::SCANNED, result), where
     *        result is the EventResult struct with result = EventResult::Result::OK and scanResult = set
     *        scanned networks. If error occurred should be baseEventCallback used with BaseEvent::SCANNED
     *        and EventResult::Result::ERROR.
     * @return request result. If error occurred return value < 0 and duplicate it with baseEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int scanNetworks() = 0;

    /**
     * @brief request to stop current bluetooth scan
     * @return request result. If error occurred return value < 0 and duplicate it with baseEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int stopScanNetworks() = 0;

    /**
     * @brief base request. Disconnect all SINK and SOURCE connections
     *          BaseEvent::DISCONNECT_ALL should be used to indicate the operation result
     *          Also SinkEvent::DISCONNECTED and SourceEvent::DISCONNECTED should be used to
     *          notify about each disconnection
     * @param[in] type of devices to disconnect. SINK - disconnect all sink devices. SOURCE - disconnect all connected
     *          source devices. ALL - disconnect all currently connected devices
     * @return request result. If error occurred return value < 0 and duplicate it with baseEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    /*@FIXME: Currently DISCONNECT_ALL and each Sink/Source Event::DISCONNECTED duplicates
     *        the information
     */
    virtual int disconnectAll(BtRole role) = 0;

    /**
     * @brief source request for pairing. When this command is used bluetooth should connect to
     *        introduces Sink device and start streaming (all sounds should be streamed out to Sink device
     *        using A2DP bluetooth protocol). SourceEvent::PAIRED should be used to indicate the operation
     *        result
     * @param network the Sink device connect to
     * @return request result. If error occurred return value < 0 and duplicate it with sourceEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int pairWithSink(const BtNetwork& network) = 0;

    /**
     * @brief sink request. Set local bluetooth device visibility
     * @param isDiscoverable indicates can device be visible for another bt devices or not
     * @param isConnectable  indicates can another device connect to local bt device
     * @return request result. If error occurred return value < 0 and duplicate it with sinkEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int setVisibility(bool isDiscoverable, bool isConnectable) = 0;

    /**
     * @brief sink request. Send PLAY_NEXT request to connected source device.
     *        SinkEvent::PLAY_NEXT should be used to return operation result
     * @param network target connected source device. (Few source device might be connected at the same time)
     * @return request result. If error occurred return value < 0 and duplicate it with sinkEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int asSinkPlayNext(const BtNetwork& network) = 0;

    /**
     * @brief sink request. Send PLAY_PREV request to connected source device.
     *        SinkEvent::PLAY_PREV should be used to return operation result
     * @param network target connected source device. (Few source device might be connected at the same time)
     * @return request result. If error occurred return value < 0 and duplicate it with sinkEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int asSinkPlayPrev(const BtNetwork& network) = 0;

    /**
     * @brief sink request. Send PLAY_PAUSE request to connected source device.
     *        SinkEvent::PLAY_PAUSE should be used to return operation result
     * @param network target connected source device. (Few source device might be connected at the same time)
     * @return request result. If error occurred return value < 0 and duplicate it with sinkEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int asSinkPlayPause(const BtNetwork& network) = 0;

    /**
     * @brief sink request. Send PLAY_START request to connected source device.
     *        SinkEvent::PLAY_START should be used to return operation result
     * @param network target connected source device. (Few source device might be connected at the same time)
     * @return request result. If error occurred return value < 0 and duplicate it with sinkEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int asSinkPlayStart(const BtNetwork& network) = 0;

    /**
     * @brief sink request.  Request volume change to ALL connected source device.
     *        SinkEvent::CHANGE_VOLUME_ABS should be used to return operation result
     * @param volume volume value: [0..127] (IPhone does use value from 0 to 127).
     * @return request result. If error occurred return value < 0 and duplicate it with sinkEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int asSinkSetVolumeAbs(int volume) = 0;

    /**
     * @brief Power on Bluetooth module
     * @return request result. If error occurred return value < 0 and duplicate it with baseEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int powerOn() = 0;

    /**
     * @brief Power off Bluetooth module
     * @return request result. If error occurred return value < 0 and duplicate it with baseEventCallbackLocked
     *         with error result. At success return 0.
     *         @note: it is not a operation result (operation result is async)
     */
    virtual int powerOff() = 0;

    /**
     * @brief Some one want to capture Audio Focus. Need to volume Down current a2dp stream (as Sink)
     */
    virtual void freeAudioFocus() = 0;

    /**
     * @brief Focus returned to BT Stream. Need return volume to default
     */
    virtual void takeAudioFocus() = 0;

    /**
     * @brief Describes current bluetooth module power state
     */
    enum class PowerState {
        OFF, /*<! Bluetooth Module is powered OFF */
        ON   /*<! Bluetooth Module is powered ON */
    };

    /**
     * @brief Return current power state (actual or simulated. Depend on realisation. But when user call
     *        for powerOff - state should OFF, after powerOn - ON)
     * @return Current Power State
     */
    virtual PowerState getPowerState() const = 0;

    /**
     * @brief Get state of local sink player (is it playing or not)
     * @return true - currently playing as Sink device
     */
    virtual bool isAsSinkPlaying() const = 0;

    /**
     * @brief Restore bt stack data to factory state (remove saved networks, saved local bt name, etc...)
     */
    virtual void factoryReset() = 0;
    /**
     * @brief AVRCP Events that can be sent/received from/to SINK/SOURCE device
     */
    enum class AVRCP {
        NONE = 0,          /*!< Not an AVRCP event. Just a stub for default value */
        PLAY_START,        /*!< Source device started playing (stream in works) */
        PLAY_PAUSE,        /*!< The request result of appropriate called method */
        PLAY_STOP,         /*!< Source device stopped playing (stream in stopped works) */
        PLAY_NEXT,         /*!< The request result of appropriate called method */
        PLAY_PREV,         /*!< The request result of appropriate called method */
        CHANGE_VOLUME_ABS, /*!< Abs volume change req from smartphone */
        TRACK_META_INFO,   /*!< Send/Received track meta information (title, artist, etc.) */
        LAST_ENUM
    };

    struct TrackInfo {
        std::string title;
        std::string artist;
        std::string album;
        std::string genre;
        int songLenMs{-1};
        int currPosMs{-1};

        Json::Value toJson() const;
    };

    /**
     * @brief Operation result or Event Description. If error occurred -> set up result as Result::ERROR,
     *        otherwise Result::OK network field should be used to specify the target network if it is needed
     *        In example: SinkEvent::CONNECTED should set up network field with connected Source device
              scanResult should be used for scanBluetoothNetworks method only.
     */
    struct EventResult {
        enum class Result {
            OK,
            ERROR
        };
        Result result{Result::OK};      /*!< Result for all operations */
        BtNetwork network;              /*!< Network the Event works with (connect to, disconnect from, etc...) */
        std::set<BtNetwork> scanResult; /*!< Result or Progress of BT Networks Scan */
        unsigned volumeAbs{0};          /*!< value from 0 to 127. IPhone send volume using values from 0 to 127 */
        AVRCP avrcpEvent{AVRCP::NONE};  /*!< AVRCP Event description. It's set only with AVRCP_IN and AVRCP_OUT events*/
        TrackInfo trackInfo;            /*!< TrackInfo. This filed is set up with AVRCP::TRACK_META_INFO */
    };

    /**
     * @brief As Source Events
     */
    enum class SourceEvent {
        PAIRED,       /*!< paired with sink device. Device should connect to sink device and start stream out */
        DISCONNECTED, /*!< disconnect from paired sink device */
        AVRCP_IN,     /*!< Sink device send AVRCP event to local device (SOURCE) */
        AVRCP_OUT,    /*!< Result of AVRCP event send to Sink device (local device is a Source) */
    };

    /**
     * @brief Sink Events
     */
    enum class SinkEvent {
        CONNECTED,                /*!< Source device connected to local sink device. Can happen without command  */
        DISCONNECTED,             /*!< Source device disconnected form local sink device. Can to happen without command */
        DISCOVERABLE,             /*!< Device set up as Discoverable */
        CONNECTABLE,              /*!< Device set up as Connectable */
        DISCOVERABLE_CONNECTABLE, /*!< Device set up as Connectable and Discoverable */
        NON_VISIBLE,              /*!< Device set up as Not Visible */
        AVRCP_IN,                 /*!< Source device send AVRCP event to local device (SINK) */
        AVRCP_OUT,                /*!< Result of AVRCP event send to Source device (local device is a Sink) */
    };

    /**
     * @brief Base Events
     */
    enum class BaseEvent {
        SCANNING,       /*!< Current Scan Networks Result event. When scan ends should be used SCANNED event */
        SCANNED,        /*!< Scan networks result event */
        SCAN_CANCELED,  /*!< Stop scan network result event */
        DISCONNECT_ALL, /*!< All devices disconnected event (should be used for disconnectAll method only) */
        FACTORY_RESET,  /*!< Restore bluetooth to factory settings */
        POWER_ON,       /*!< Bluetooth module is powered On */
        POWER_OFF       /*!< Bluetooth module is powered Off */
    };

    /**
     * @brief Bluetooth Events Listener.
     */
    class BluetoothEventListener {
    public:
        virtual void onBaseEvent(Bluetooth::BaseEvent ev, const Bluetooth::EventResult& res) = 0;

        virtual void onSourceEvent(Bluetooth::SourceEvent ev, const Bluetooth::EventResult& res) = 0;

        virtual void onSinkEvent(Bluetooth::SinkEvent ev, const Bluetooth::EventResult& res) = 0;

        virtual ~BluetoothEventListener() = default;
    };

    /**
     * @brief Register listener for bluetooth events
     * @param listener bluetooth event listener
     */
    void registerEventListener(std::weak_ptr<BluetoothEventListener> listener);

protected:
    std::string name_;

    /**
     * @brief Callback that should be used when Source Event happened. This callback
     * is thread safe
     * @param ev SourceEvent value
     * @param res Result value
     */
    void sourceEventCallbackLocked(SourceEvent ev, EventResult res);

    /**
     * @brief Callback that should be used when Sink Event happened. This callback
     * is thread safe
     * @param ev SinkEvent value
     * @param res Result value
     */
    void sinkEventCallbackLocked(SinkEvent ev, EventResult res);

    /**
     * @brief Callback that should be used when Base Event happened. This callback
     * is thread safe
     * @param ev BaseEvent value
     * @param res Result value
     */
    void baseEventCallbackLocked(BaseEvent ev, EventResult res);

private:
    std::condition_variable condVar_;
    mutable std::mutex eventQueueMutex_;
    std::thread eventSenderThread_;

    std::atomic_bool threadExists_;

    std::queue<std::pair<BaseEvent, EventResult>> baseEventQueue_;
    std::queue<std::pair<SourceEvent, EventResult>> sourceEventQueue_;
    std::queue<std::pair<SinkEvent, EventResult>> sinkEventQueue_;

    std::mutex listenersMutex_;
    std::list<std::weak_ptr<BluetoothEventListener>> listeners_;

    /**
     * @brief It is thread safe way to send user callbacks. When BluetoothImpl
     *        use *EventCallbackLocked method the task will be added into
     *        queue and eventSenderThread woke up and send callback.
     *        This is a way to prevent deadlocks when user callback are used
     */
    void eventSenderThread();
};

namespace std {
    template <>
    struct hash<Bluetooth::BtRole> {
        using argument_type = Bluetooth::BtRole;
        using underlying_type = std::underlying_type<argument_type>::type;
        using result_type = std::hash<underlying_type>::result_type;
        result_type operator()(const argument_type& arg) const {
            std::hash<underlying_type> hasher;
            return hasher(static_cast<underlying_type>(arg));
        }
    };

    template <>
    struct hash<Bluetooth::BtNetwork> {
        size_t operator()(const Bluetooth::BtNetwork& n) const {
            // Compute individual hash values for first, second and third
            // http://stackoverflow.com/a/1646913/126995
            size_t res = 17;
            res = res * 31 + hash<string>()(n.name);
            res = res * 31 + hash<string>()(n.addr);
            res = res * 31 + hash<Bluetooth::BtRole>()(n.role);
            return res;
        }
    };
} // namespace std
