#include "bluez_ble_impl.h"

#include "bluetooth_adapter.h"
#include "common.h"
#include "object_manager.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <fstream>

YIO_DEFINE_LOG_MODULE("bluez");

using namespace quasar;
using namespace bluez_impl;

namespace {

    constexpr uint32_t ADVERTISING_INTERVAL = 0x20; // 20ms

} // namespace

const sdbus::ObjectPath BluezBLEImpl::ORG_PATH = "/ru/yandex/quasar";
const sdbus::ObjectPath BluezBLEImpl::PROFILE_PATH = ORG_PATH + "/profile";
const sdbus::ObjectPath BluezBLEImpl::ADVERT_PATH = ORG_PATH + "/advertising";
const sdbus::ObjectPath BluezBLEImpl::GATT_APP_PATH = ORG_PATH + "/gatt";

BluezBLEImpl::BluezBLEImpl()
    : adapterPath_(ObjectManager::getBluezObjectPath(BluetoothAdapter::INTERFACE_NAME))
    , systemBusConnection_(sdbus::createSystemBusConnection(QUASAR_BUS))
    , connectionMonitor_(*systemBusConnection_)
    , gattApplication_(*systemBusConnection_, GATT_APP_PATH)
{
    setAdvertisingInterval(ADVERTISING_INTERVAL, ADVERTISING_INTERVAL);

    advertisingManager_ = std::make_unique<LEAdvertisingManager>(*systemBusConnection_, adapterPath_);
    gattManager_ = std::make_unique<GattManager>(*systemBusConnection_, adapterPath_);
    YIO_LOG_INFO("BluezBLEImpl created");
}

BluezBLEImpl::~BluezBLEImpl()
{
    callbackQueue_.destroy();
}

void BluezBLEImpl::setManufacturerData(ManufacturerData manufacturerData)
{
    callbackQueue_.add([this, manufacturerData{std::move(manufacturerData)}]() mutable {
        manufacturerData_ = std::move(manufacturerData);
    });
}

void BluezBLEImpl::addGattService(std::shared_ptr<IGattService> service)
{
    callbackQueue_.add([this, service{std::move(service)}]() mutable {
        gattApplication_.addGattService(std::move(service));
    });
}

void BluezBLEImpl::addGattCharacteristic(const std::string& gattServiceName, std::shared_ptr<IGattCharacteristic> characteristic)
{
    callbackQueue_.add([this, gattServiceName, characteristic{std::move(characteristic)}]() mutable {
        gattApplication_.addGattCharacteristic(gattServiceName, std::move(characteristic));
    });
}

void BluezBLEImpl::startAdvertising()
{
    YIO_LOG_INFO("Starting advertising...");

    callbackQueue_.add([this] {
        acquireAdapter();

        gattManager_->RegisterApplication(gattApplication_.getObjectPath(), {});

        try {
            advertisingReenabler_ = std::make_unique<LEAdvertisingReenabler>();
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("BluezBLEImpl.StartAdvertising", "Failed to start LE advertising reenabler: " << e.what());
        }

        advertisement_ = std::make_unique<LEAdvertisement>(
            *systemBusConnection_, ADVERT_PATH, manufacturerData_.id, manufacturerData_.data, gattApplication_.getServiceUUIDs());
        advertisingManager_->RegisterAdvertisement(advertisement_->getObjectPath(), {});
    });
}

void BluezBLEImpl::stopAdvertising()
{
    YIO_LOG_INFO("Stop advertising");

    callbackQueue_.add([this] {
        advertisingReenabler_.reset();

        advertisingManager_->UnregisterAdvertisement(advertisement_->getObjectPath());
        advertisement_.reset();

        gattManager_->UnregisterApplication(gattApplication_.getObjectPath());

        releaseAdapter();
    });
}

void BluezBLEImpl::acquireAdapter()
{
    BluetoothAdapter adapter(adapterPath_);

    if (!adapter.Powered()) {
        adapter.Powered(true);
        savedAdapterState_.powered = false;
    }
    if (adapter.Discoverable()) {
        adapter.Discoverable(false);
        savedAdapterState_.discoverable = true;
    }
    if (adapter.Pairable()) {
        adapter.Pairable(false);
        savedAdapterState_.pairable = true;
    }
}

void BluezBLEImpl::releaseAdapter()
{
    BluetoothAdapter adapter(adapterPath_);

    if (savedAdapterState_.powered.has_value()) {
        if (adapter.Powered() != savedAdapterState_.powered.value()) {
            adapter.Powered(savedAdapterState_.powered.value());
        }
        savedAdapterState_.powered.reset();
    }

    if (savedAdapterState_.discoverable.has_value()) {
        if (adapter.Discoverable() != savedAdapterState_.discoverable.value()) {
            adapter.Discoverable(savedAdapterState_.discoverable.value());
        }
        savedAdapterState_.discoverable.reset();
    }

    if (savedAdapterState_.pairable.has_value()) {
        if (adapter.Pairable() != savedAdapterState_.pairable.value()) {
            adapter.Pairable(savedAdapterState_.pairable.value());
        }
        savedAdapterState_.pairable.reset();
    }
}

void BluezBLEImpl::setAdvertisingInterval(uint32_t min, uint32_t max) {
    const std::string adapterName = quasar::split(adapterPath_, "/").back();
    const std::string debugFsPath = "/sys/kernel/debug/bluetooth/" + adapterName;

    try {
        std::ofstream advMinInterval(debugFsPath + "/adv_min_interval");
        advMinInterval.exceptions(std::ofstream::badbit | std::ofstream::failbit);
        advMinInterval << min;
        advMinInterval.flush();
        YIO_LOG_INFO("Set min advertising interval to " << min);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("BluezBLEImpl.SetAdvertisingIntervalMin", "Failed to set min advertising interval: " << e.what());
        return;
    }

    try {
        std::ofstream advMaxInterval(debugFsPath + "/adv_max_interval");
        advMaxInterval.exceptions(std::ofstream::badbit | std::ofstream::failbit);
        advMaxInterval << max;
        advMaxInterval.flush();
        YIO_LOG_INFO("Set max advertising interval to " << max);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("BluezBLEImpl.SetAdvertisingIntervalMax", "Failed to set max advertising interval: " << e.what());
        return;
    }
}
