#include "bluez_bluetooth_impl.h"

#include "bluetooth_adapter.h"
#include "object_manager.h"

#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("bluez");

namespace {

    constexpr uint16_t MAX_TRANSPORT_VOLUME = 127;

} // namespace

namespace quasar {

    BluezBluetoothImpl::BluezBluetoothImpl(const std::string& name)
        : Bluetooth{name}
        , PropertiesProxy{
              bluez_impl::BLUEZ_BUS,
              bluez_impl::ObjectManager::getBluezObjectPath(bluez_impl::BluetoothAdapter::INTERFACE_NAME)}
        , ObjectManagerProxy{bluez_impl::BLUEZ_BUS, bluez_impl::BLUEZ_OM_PATH}
        , BlueAlsaManagerProxy{bluez_impl::BLUEALSA_BUS, bluez_impl::BLUEALSA_MANAGER_PATH}
        , adapterPath_{bluez_impl::ObjectManager::getBluezObjectPath(bluez_impl::BluetoothAdapter::INTERFACE_NAME)}
    {
        PropertiesProxy::registerProxy();
        ObjectManagerProxy::registerProxy();
        BlueAlsaManagerProxy::registerProxy();
        setName(name);
    }

    BluezBluetoothImpl::~BluezBluetoothImpl() {
        callbackQueue_.destroy();
        BlueAlsaManagerProxy::unregisterProxy();
        ObjectManagerProxy::unregisterProxy();
        PropertiesProxy::unregisterProxy();
    }

    int BluezBluetoothImpl::setName(const std::string& name) {
        bluez_impl::BluetoothAdapter adapter{adapterPath_};
        try {
            if (adapter.Alias() != name) {
                YIO_LOG_INFO("Set new name: " << name);
                adapter.Alias(name);
            }
        } catch (const sdbus::Error& error) {
            YIO_LOG_ERROR_EVENT("BluezBluetoothImpl.SetName", "failed to set bluetooth device alias: " << error.what());
            return -1;
        }

        return 0;
    }

    int BluezBluetoothImpl::scanNetworks() {
        YIO_LOG_WARN(__func__ << " is not implemented");
        return 0;
    }

    int BluezBluetoothImpl::stopScanNetworks() {
        YIO_LOG_WARN(__func__ << " is not implemented");
        return 0;
    }

    int BluezBluetoothImpl::disconnectAll(Bluetooth::BtRole /*role*/) {
        callbackQueue_.add([this] {
            if (mediaPlayer_ != nullptr) {
                mediaPlayer_->getDeviceProxy()->Disconnect();
                onSourceDisconnected();
            }
        });
        return 0;
    }

    int BluezBluetoothImpl::pairWithSink(const Bluetooth::BtNetwork& /*network*/) {
        YIO_LOG_WARN(__func__ << " is not implemented");
        return 0;
    }

    int BluezBluetoothImpl::setVisibility(bool isDiscoverable, bool isConnectable) {
        YIO_LOG_INFO("Set visibility. discoverable: " << isDiscoverable << ", connectable: " << isConnectable);

        bluez_impl::BluetoothAdapter adapter{adapterPath_};
        try {
            adapter.Pairable(isConnectable);
            adapter.Discoverable(isDiscoverable);
        } catch (const sdbus::Error& error) {
            YIO_LOG_ERROR_EVENT("BluezBluetoothImpl.SetVisibility", "failed to change visibility " << error.what());
            const auto event = eventFromState(adapter.Discoverable(), adapter.Pairable());
            EventResult result;
            result.result = EventResult::Result::ERROR;
            sinkEventCallbackLocked(event, result);
            return -1;
        }
        const auto event = eventFromState(adapter.Discoverable(), adapter.Pairable());
        EventResult result;
        sinkEventCallbackLocked(event, result);
        return 0;
    }

    int BluezBluetoothImpl::asSinkPlayNext(const Bluetooth::BtNetwork& /*network*/) {
        YIO_LOG_INFO("Play next as bluetooth sink");
        callbackQueue_.add([this] {
            if (mediaPlayer_ != nullptr) {
                mediaPlayer_->Next();
            }
        });

        return 0;
    }

    int BluezBluetoothImpl::asSinkPlayPrev(const Bluetooth::BtNetwork& /*network*/) {
        YIO_LOG_INFO("Play prev as bluetooth sink");
        callbackQueue_.add([this] {
            if (mediaPlayer_ != nullptr) {
                mediaPlayer_->Previous();
            }
        });

        return 0;
    }

    int BluezBluetoothImpl::asSinkPlayPause(const Bluetooth::BtNetwork& /*network*/) {
        YIO_LOG_INFO("Pause playing as bluetooth sink");
        callbackQueue_.add([this] {
            if (mediaPlayer_ != nullptr) {
                mediaPlayer_->Pause();
            }
        });

        return 0;
    }

    int BluezBluetoothImpl::asSinkPlayStart(const Bluetooth::BtNetwork& /*network*/) {
        YIO_LOG_INFO("Start playing as bluetooth sink");

        callbackQueue_.add([this] {
            if (mediaPlayer_ != nullptr) {
                mediaPlayer_->Play();
            }
        });

        return 0;
    }

    int BluezBluetoothImpl::asSinkSetVolumeAbs(int volume) {
        YIO_LOG_INFO("Set volume as bluetooth sink. Value: " << volume);

        callbackQueue_.add([this, volume] {
            if (mediaTransport_ != nullptr) {
                mediaTransport_->setVolume(volume);
            }
        });

        return 0;
    }

    int BluezBluetoothImpl::powerOn() {
        return powerOnOff(true);
    }

    int BluezBluetoothImpl::powerOff() {
        return powerOnOff(false);
    }

    void BluezBluetoothImpl::freeAudioFocus() {
        YIO_LOG_INFO("Free audio focus");

        callbackQueue_.add([this] {
            if (mediaTransport_ != nullptr && blueAlsaPcm_ != nullptr) {
                blueAlsaPcm_->SoftVolume(true);
                isFadedOut_.store(true);
                stashedVolume_ = mediaTransport_->getVolume();
                mediaTransport_->setVolume(MAX_TRANSPORT_VOLUME / 2);
            }
        });
    }

    void BluezBluetoothImpl::takeAudioFocus() {
        YIO_LOG_INFO("Take audio focus");

        callbackQueue_.add([this] {
            if (mediaTransport_ && blueAlsaPcm_ != nullptr && isFadedOut_.load()) {
                blueAlsaPcm_->SoftVolume(false);
                isFadedOut_.store(false);
                mediaTransport_->setVolume(stashedVolume_);
            }
        });
    }

    Bluetooth::PowerState BluezBluetoothImpl::getPowerState() const {
        bluez_impl::BluetoothAdapter adapter{adapterPath_};
        return adapter.Powered() ? Bluetooth::PowerState::ON : Bluetooth::PowerState::OFF;
    }

    bool BluezBluetoothImpl::isAsSinkPlaying() const {
        return isPlaying_.load();
    }

    void BluezBluetoothImpl::factoryReset() {
        YIO_LOG_WARN(__func__ << " is not implemented");
    }

    void BluezBluetoothImpl::onPropertiesChanged(const std::string& interfaceName,
                                                 const std::map<std::string, sdbus::Variant>& changedProperties,
                                                 const std::vector<std::string>& /*invalidatedProperties*/)
    {
        if (interfaceName != bluez_impl::BluetoothAdapter::INTERFACE_NAME) {
            return;
        }

        for (const auto& [propName, propValue] : changedProperties) {
            if (propName == bluez_impl::BluetoothAdapter::POWERED_PROPERTY_NAME) {
                const auto event = propValue.get<bool>() ? BaseEvent::POWER_ON : BaseEvent::POWER_OFF;
                baseEventCallbackLocked(event, {});
            } else if (propName == bluez_impl::BluetoothAdapter::DISCOVERABLE_PROPERTY_NAME) {
                const auto event = eventFromState(propValue.get<bool>(), bluez_impl::BluetoothAdapter{adapterPath_}.Pairable());
                sinkEventCallbackLocked(event, {});
            } else if (propName == bluez_impl::BluetoothAdapter::PAIRABLE_PROPERTY_NAME) {
                const auto event = eventFromState(bluez_impl::BluetoothAdapter{adapterPath_}.Discoverable(), propValue.get<bool>());
                sinkEventCallbackLocked(event, {});
            }
        }
    }

    void BluezBluetoothImpl::onInterfacesAdded(const sdbus::ObjectPath& objectPath,
                                               const std::map<std::string, std::map<std::string, sdbus::Variant>>& interfaces)
    {
        callbackQueue_.add([this, objectPath, interfaces] {
            for (const auto& [interfaceName, propertiesMap] : interfaces) {
                if (interfaceName != org::bluez::MediaTransport1_proxy::INTERFACE_NAME &&
                    interfaceName != org::bluez::MediaPlayer1_proxy::INTERFACE_NAME)
                {
                    continue;
                }

                const auto deviceProp = propertiesMap.find("Device");
                if (deviceProp == propertiesMap.end()) {
                    YIO_LOG_WARN("Received interface without Device property, skipping it");
                    return;
                }
                const auto newDevicePath = deviceProp->second.get<sdbus::ObjectPath>();

                if (mediaTransport_ != nullptr || mediaPlayer_ != nullptr) {
                    // if we have transport or player from another device, we should disconnect from it
                    const auto currDevicePath = mediaTransport_ != nullptr ? mediaTransport_->Device() : mediaPlayer_->Device();
                    if (currDevicePath != newDevicePath) {
                        if (mediaTransport_ != nullptr) {
                            mediaTransport_->getDeviceProxy()->Disconnect();
                            if (mediaPlayer_ != nullptr) {
                                onSourceDisconnected();
                            }
                        } else {
                            mediaPlayer_->getDeviceProxy()->Disconnect();
                        }

                        mediaTransport_ = nullptr;
                        mediaPlayer_ = nullptr;
                    }
                }

                if (interfaceName == org::bluez::MediaTransport1_proxy::INTERFACE_NAME) {
                    mediaTransport_ = std::make_shared<bluez_impl::MediaTransport>(objectPath, shared_from_this());
                } else {
                    mediaPlayer_ = std::make_shared<bluez_impl::MediaPlayer>(objectPath, shared_from_this());
                }

                if (mediaTransport_ != nullptr && mediaPlayer_ != nullptr) {
                    EventResult res;
                    res.result = EventResult::Result::OK;
                    res.network = mediaPlayer_->getDeviceProxy()->getBtNetwork();

                    sinkEventCallbackLocked(SinkEvent::CONNECTED, res);
                    YIO_LOG_INFO("Bt source connected. Name: " << res.network.name << " Addr: " << res.network.addr);
                }
            }
        });
    }

    void BluezBluetoothImpl::onInterfacesRemoved(const sdbus::ObjectPath& objectPath,
                                                 const std::vector<std::string>& interfaces)
    {
        callbackQueue_.add([this, objectPath, interfaces] {
            const auto playerIt = std::find(interfaces.begin(), interfaces.end(), org::bluez::MediaPlayer1_proxy::INTERFACE_NAME);
            if (playerIt != interfaces.end() && mediaPlayer_ != nullptr && mediaPlayer_->getObjectPath() == objectPath) {
                if (mediaTransport_ != nullptr) {
                    onSourceDisconnected();
                } else {
                    mediaPlayer_ = nullptr;
                }
            }

            const auto transportIt = std::find(interfaces.begin(), interfaces.end(), org::bluez::MediaTransport1_proxy::INTERFACE_NAME);
            if (transportIt != interfaces.end() && mediaTransport_ != nullptr && mediaTransport_->getObjectPath() == objectPath) {
                if (mediaPlayer_ != nullptr) {
                    onSourceDisconnected();
                } else {
                    mediaTransport_ = nullptr;
                }
            }
        });
    }

    void BluezBluetoothImpl::onPlaying() {
        isPlaying_.store(true);
        EventResult res;
        res.avrcpEvent = AVRCP::PLAY_START;
        sinkEventCallbackLocked(SinkEvent::AVRCP_IN, res);
    }

    void BluezBluetoothImpl::onStopped() {
        isPlaying_.store(false);
        EventResult res;
        res.avrcpEvent = AVRCP::PLAY_STOP;
        sinkEventCallbackLocked(SinkEvent::AVRCP_IN, res);
    }

    void BluezBluetoothImpl::onPaused() {
        isPlaying_.store(false);
        EventResult res;
        res.avrcpEvent = AVRCP::PLAY_PAUSE;
        sinkEventCallbackLocked(SinkEvent::AVRCP_IN, res);
    }

    void BluezBluetoothImpl::onError() {
        onStopped();
    }

    void BluezBluetoothImpl::onTrackInfo(const Bluetooth::TrackInfo& trackInfo) {
        EventResult res;
        res.avrcpEvent = AVRCP::TRACK_META_INFO;
        res.trackInfo = trackInfo;
        sinkEventCallbackLocked(SinkEvent::AVRCP_IN, res);
    }

    void BluezBluetoothImpl::onVolumeChanged(uint16_t volume) {
        // do not handle volume change in fade out state
        if (!isFadedOut_.load()) {
            EventResult res;
            res.avrcpEvent = AVRCP::CHANGE_VOLUME_ABS;
            res.volumeAbs = volume;
            sinkEventCallbackLocked(SinkEvent::AVRCP_IN, res);
        }
    }

    void BluezBluetoothImpl::onPCMAdded(const sdbus::ObjectPath& path, const std::map<std::string, sdbus::Variant>& /*props*/) {
        callbackQueue_.add([this, path] {
            blueAlsaPcm_ = std::make_unique<bluez_impl::BlueAlsaPcm>(path);
            blueAlsaPcm_->SoftVolume(false);
        });
    }

    void BluezBluetoothImpl::onPCMRemoved(const sdbus::ObjectPath& path) {
        callbackQueue_.add([this, path] {
            if (blueAlsaPcm_ && blueAlsaPcm_->getObjectPath() == path) {
                blueAlsaPcm_.reset(nullptr);
            }
        });
    }

    Bluetooth::SinkEvent BluezBluetoothImpl::eventFromState(bool isDiscoverable, bool isPairable) {
        SinkEvent event = SinkEvent::NON_VISIBLE;
        if (isDiscoverable && isPairable) {
            event = SinkEvent::DISCOVERABLE_CONNECTABLE;
        } else if (isDiscoverable) {
            event = SinkEvent::DISCOVERABLE;
        } else if (isPairable) {
            event = SinkEvent::CONNECTABLE;
        }
        return event;
    }

    int BluezBluetoothImpl::powerOnOff(bool powerState) {
        YIO_LOG_INFO("Switch power. State: " << powerState);

        bluez_impl::BluetoothAdapter adapter{adapterPath_};
        try {
            adapter.Powered(powerState);
        } catch (const sdbus::Error& error) {
            YIO_LOG_ERROR_EVENT("BluezBluetoothImpl.PowerOnOff", "failed to change power state: " << error.what());
            const auto event = adapter.Powered() ? BaseEvent::POWER_ON : BaseEvent::POWER_OFF;
            EventResult result;
            result.result = EventResult::Result::ERROR;
            baseEventCallbackLocked(event, result);
            return -1;
        }

        return 0;
    }

    void BluezBluetoothImpl::onSourceDisconnected() {
        EventResult res;
        res.result = EventResult::Result::OK;
        res.network = mediaPlayer_->getDeviceProxy()->getBtNetwork();

        isPlaying_.store(false);
        mediaPlayer_ = nullptr;
        mediaTransport_ = nullptr;

        sinkEventCallbackLocked(SinkEvent::DISCONNECTED, res);
        YIO_LOG_INFO("Bt source disconnected. Name: " << res.network.name << " Addr: " << res.network.addr);
    }

} // namespace quasar
