#include "gatt_characteristic.h"

#include <yandex_io/libs/logging/logging.h>

using namespace bluez_impl;

GattCharacteristic::GattCharacteristic(sdbus::IConnection& connection, std::string servicePath, uint16_t index, std::shared_ptr<IGattCharacteristic> charPtr)
    : sdbus::AdaptorInterfaces<org::bluez::GattCharacteristic1_adaptor, sdbus::Properties_adaptor>(connection, servicePath + "/char" + std::to_string(index))
    , service_(std::move(servicePath))
    , characteristicPtr_(std::move(charPtr))
    , flags_(BluezFlags::covertFromQuasarPermission(characteristicPtr_->getPermissions()))
{
    registerAdaptor();
    YIO_LOG_INFO("Characteristic " << characteristicPtr_->getName() << " path: " << getObjectPath() << " created successfully");
}

GattCharacteristic::~GattCharacteristic()
{
    characteristicPtr_->enableNotifications(nullptr);

    unregisterAdaptor();
}

std::vector<uint8_t> GattCharacteristic::ReadValue(const std::map<std::string, sdbus::Variant>& options)
{
    sdbus::ObjectPath objectDevice;
    if (options.count("device")) {
        objectDevice = options.at("device").get<sdbus::ObjectPath>();
    }
    uint16_t mtu = MIN_MTU;
    if (options.count("mtu")) {
        mtu = options.at("mtu").get<uint16_t>();
    }
    uint16_t offset = 0;
    if (options.count("offset")) {
        offset = options.at("offset").get<uint16_t>();
    }

    YIO_LOG_INFO("Read characteristic \'" << characteristicPtr_->getName() << "\': device " << objectDevice << ", mtu " << mtu << ", offset " << offset);

    try {
        return characteristicPtr_->readValue(objectDevice, mtu, offset);
    } catch (const std::runtime_error& e) {
        throw sdbus::Error("org.bluez.Error.Failed", e.what());
    }
}

void GattCharacteristic::WriteValue(const std::vector<uint8_t>& value, const std::map<std::string, sdbus::Variant>& options)
{
    sdbus::ObjectPath objectDevice;
    if (options.count("device")) {
        objectDevice = options.at("device").get<sdbus::ObjectPath>();
    }

    YIO_LOG_INFO("Write characteristic \'" << characteristicPtr_->getName() << "\': device " << objectDevice);

    try {
        characteristicPtr_->writeValue(objectDevice, value);
    } catch (const std::runtime_error& e) {
        throw sdbus::Error("org.bluez.Error.Failed", e.what());
    }
}

void GattCharacteristic::StartNotify()
{
    YIO_LOG_INFO("Start notifying characteristic \'" << characteristicPtr_->getName() << "\'");

    characteristicPtr_->enableNotifications([this]() {
        YIO_LOG_INFO("Sending notification for characteristic \'" << characteristicPtr_->getName() << "\'");
        emitPropertiesChangedSignal(GattCharacteristic1_adaptor::INTERFACE_NAME, {"Value"});
    });
}

void GattCharacteristic::StopNotify()
{
    YIO_LOG_INFO("Stop notifying characteristic \'" << characteristicPtr_->getName() << "\'");

    characteristicPtr_->enableNotifications(nullptr);
}

void GattCharacteristic::Confirm()
{
    YIO_LOG_INFO(__func__);
}

std::string GattCharacteristic::UUID()
{
    return characteristicPtr_->getUuid();
}

std::vector<std::string> GattCharacteristic::Flags()
{
    return flags_;
}

sdbus::ObjectPath GattCharacteristic::Service()
{
    return service_;
}

std::vector<sdbus::ObjectPath> GattCharacteristic::Descriptors()
{
    return {};
}

uint16_t GattCharacteristic::Handle()
{
    YIO_LOG_INFO("Getting characteristic \'" << characteristicPtr_->getName() << "\' handle: " << handle_);
    return handle_;
}

void GattCharacteristic::Handle(const uint16_t& value)
{
    YIO_LOG_INFO("Setting new characteristic \'" << characteristicPtr_->getName() << "\' handle: " << std::to_string(value));
    handle_ = value;

    emitPropertiesChangedSignal(GattCharacteristic1_adaptor::INTERFACE_NAME, {"Handle"});
}

std::vector<uint8_t> GattCharacteristic::Value()
{
    YIO_LOG_INFO("Getting characteristic \'" << characteristicPtr_->getName() << "\' value");
    return characteristicPtr_->getValue();
}

std::string GattCharacteristic::BluezFlags::covertFromQuasarPermission(gatt_permission::GattPermission permission)
{
    switch (permission) {
        case gatt_permission::GattPermission::BROADCAST:
            return BROADCAST;
        case gatt_permission::GattPermission::READ:
            return READ;
        case gatt_permission::GattPermission::WRITE_WO_RESPONSE:
            return WRITE_WO_RESPONSE;
        case gatt_permission::GattPermission::WRITE:
            return WRITE;
        case gatt_permission::GattPermission::NOTIFY:
            return NOTIFY;
        case gatt_permission::GattPermission::INDICATE:
            return INDICATE;
        case gatt_permission::GattPermission::AUTH_SIGNED_WRITES:
            return AUTH_SIGNED_WRITES;
        case gatt_permission::GattPermission::EXTENDED_PROPERTIES:
            return EXTENDED_PROPERTIES;
        case gatt_permission::GattPermission::RELIABLE_WRITE:
            return RELIABLE_WRITE;
        case gatt_permission::GattPermission::WRITABLE_AUXILIARIES:
            return WRITABLE_AUXILIARIES;
        case gatt_permission::GattPermission::ENCRYPT_READ:
            return ENCRYPT_READ;
        case gatt_permission::GattPermission::ENCRYPT_WRITE:
            return ENCRYPT_WRITE;
        case gatt_permission::GattPermission::ENCRYPT_AUTH_READ:
            return ENCRYPT_AUTH_READ;
        case gatt_permission::GattPermission::ENCRYPT_AUTH_WRITE:
            return ENCRYPT_AUTH_WRITE;
        case gatt_permission::GattPermission::SECURE_READ:
            return SECURE_READ;
        case gatt_permission::GattPermission::SECURE_WRITE:
            return SECURE_WRITE;
        case gatt_permission::GattPermission::AUTHORIZE:
            return AUTHORIZE;
        default:
            return "";
    }
}

std::vector<std::string> GattCharacteristic::BluezFlags::covertFromQuasarPermission(
    const std::set<gatt_permission::GattPermission>& permissions)
{
    std::vector<std::string> flags;
    for (auto perm : permissions) {
        auto flag = covertFromQuasarPermission(perm);
        if (!flag.empty()) {
            flags.emplace_back(flag);
        }
    }
    return flags;
}
