#include "le_advertising_reenabler.h"

#include <yandex_io/libs/logging/logging.h>

#include <contrib/libs/bluez-clean-headers/bluetooth/bluetooth.h>
#include <contrib/libs/bluez-clean-headers/bluetooth/hci.h>

#include <unistd.h>
#include <sys/eventfd.h>
#include <sys/poll.h>
#include <sys/socket.h>

YIO_DEFINE_LOG_MODULE("bluez");

namespace bluez_impl {

    namespace {

        constexpr uint8_t HCI_ROLE_SLAVE = 0x01;

    } // namespace

    LEAdvertisingReenabler::LEAdvertisingReenabler() {
        hci_ = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI);
        if (hci_ < 0) {
            throw std::runtime_error("Failed to create HCI socket");
        }

        struct hci_filter filter = {};
        filter.type_mask = (1 << HCI_EVENT_PKT);
        filter.event_mask[1] = (1 << (EVT_LE_META_EVENT - 32));

        if (setsockopt(hci_, SOL_HCI, HCI_FILTER, &filter, sizeof(filter)) != 0) {
            close(hci_);
            throw std::runtime_error("Failed to set HCI filter");
        }

        struct sockaddr_hci addr = {};
        addr.hci_family = AF_BLUETOOTH;
        addr.hci_dev = 0;
        addr.hci_channel = HCI_CHANNEL_RAW;

        if (bind(hci_, reinterpret_cast<struct sockaddr*>(&addr), sizeof(addr)) != 0) {
            close(hci_);
            throw std::runtime_error("Failed to bind HCI socket");
        }

        stopFd_ = eventfd(0, 0);
        if (stopFd_ < 0) {
            close(hci_);
            throw std::runtime_error("Failed to create eventfd");
        }

        loopThread_ = std::thread(&LEAdvertisingReenabler::loop, this);

        YIO_LOG_INFO("LE advertising reenabler started");
    }

    LEAdvertisingReenabler::~LEAdvertisingReenabler() {
        stopped_ = true;
        eventfd_write(stopFd_, 1);

        if (loopThread_.joinable()) {
            loopThread_.join();
        }

        close(hci_);
        close(stopFd_);

        YIO_LOG_INFO("LE advertising reenabler stopped");
    }

    void LEAdvertisingReenabler::loop() noexcept {
        std::array<struct pollfd, 2> fds = {{{hci_, POLLIN, 0},
                                             {stopFd_, POLLIN, 0}}};

        while (!stopped_) {
            int ret = poll(fds.data(), fds.size(), -1);
            if (ret <= 0) {
                continue;
            }

            if (fds[0].revents & POLLIN) {
                char data[1024];
                ssize_t length = read(hci_, data, sizeof(data));
                if (length > 0) {
                    handleEvent({
                        data,
                        static_cast<size_t>(length),
                    });
                }
            }
        }
    }

    void LEAdvertisingReenabler::handleEvent(std::string_view frame) {
        uint8_t packetType = frame[0];
        frame.remove_prefix(1);

        if (packetType != HCI_EVENT_PKT) {
            return;
        }

        hci_event_hdr hdr;
        std::memcpy(&hdr, frame.data(), sizeof(hdr));
        frame.remove_prefix(HCI_EVENT_HDR_SIZE);

        if (hdr.evt != EVT_LE_META_EVENT) {
            return;
        }

        evt_le_meta_event mevt;
        std::memcpy(&mevt, frame.data(), sizeof(mevt));
        frame.remove_prefix(EVT_LE_META_EVENT_SIZE);

        if (mevt.subevent != EVT_LE_CONN_COMPLETE) {
            return;
        }

        evt_le_connection_complete evt;
        std::memcpy(&evt, frame.data(), sizeof(evt));
        frame.remove_prefix(EVT_LE_CONN_COMPLETE_SIZE);

        if (evt.role != HCI_ROLE_SLAVE) {
            return;
        }

        YIO_LOG_INFO("New LE connection");
        enableAdvertising();
    }

    void LEAdvertisingReenabler::enableAdvertising() const {
        YIO_LOG_INFO("Enabling advertising");

        struct {
            uint8_t type = HCI_COMMAND_PKT;
            hci_command_hdr hdr = {
                cmd_opcode_pack(OGF_LE_CTL, OCF_LE_SET_ADVERTISE_ENABLE),
                LE_SET_ADVERTISE_ENABLE_CP_SIZE,
            };
            le_set_advertise_enable_cp msg = {
                0x01,
            };
        } cmd;

        if (write(hci_, &cmd, sizeof(cmd)) != sizeof(cmd)) {
            YIO_LOG_ERROR_EVENT("LEAdvertisingReenabler.EnableAdvertising", "Failed to send HCI command");
        }
    }

} // namespace bluez_impl
