#include "media_player.h"

#include "common.h"

using namespace bluez_impl;

MediaPlayer::MediaPlayer(std::string dbusPath, std::weak_ptr<Listener> listener)
    : sdbus::ProxyInterfaces<org::bluez::MediaPlayer1_proxy, sdbus::Properties_proxy>{
          BLUEZ_BUS,
          std::move(dbusPath)}
    , listener_(std::move(listener))
    , bluetoothDevice_(std::make_shared<BluetoothDevice>(Device()))
{
    registerProxy();
}

MediaPlayer::~MediaPlayer() {
    unregisterProxy();
}

bool MediaPlayer::isPlaying() {
    return Status() == "playing";
}

std::shared_ptr<BluetoothDevice> MediaPlayer::getDeviceProxy() {
    return bluetoothDevice_;
}

void MediaPlayer::onPropertiesChanged(const std::string& interfaceName,
                                      const std::map<std::string, sdbus::Variant>& changedProperties,
                                      const std::vector<std::string>& /*invalidatedProperties*/)
{
    if (interfaceName != org::bluez::MediaPlayer1_proxy::INTERFACE_NAME) {
        return;
    }

    for (const auto& [propName, propValue] : changedProperties) {
        if (propName == "Status") {
            onStatusChanged(propValue.get<std::string>());
        } else if (propName == "Track") {
            onTrackChanged(propValue.get<std::map<std::string, sdbus::Variant>>());
        } else if (propName == "Position") {
            onPositionChanged(propValue.get<uint32_t>());
        }
    }
}

void MediaPlayer::onStatusChanged(std::string_view status) {
    if (auto listener = listener_.lock()) {
        if (status == "playing") {
            listener->onPlaying();
        } else if (status == "stopped") {
            listener->onStopped();
        } else if (status == "paused") {
            listener->onPaused();
        } else if (status == "error") {
            listener->onError();
        }
    }
}

void MediaPlayer::onTrackChanged(const std::map<std::string, sdbus::Variant>& trackData) {
    for (const auto& [key, value] : trackData) {
        if (key == "Title") {
            trackInfo_.title = value.get<std::string>();
        } else if (key == "Artist") {
            trackInfo_.artist = value.get<std::string>();
        } else if (key == "Album") {
            trackInfo_.album = value.get<std::string>();
        } else if (key == "Genre") {
            trackInfo_.genre = value.get<std::string>();
        } else if (key == "Duration") {
            trackInfo_.songLenMs = value.get<uint32_t>();
        }
    }

    if (auto listener = listener_.lock()) {
        listener->onTrackInfo(trackInfo_);
    }
}

void MediaPlayer::onPositionChanged(uint32_t positionMs) {
    trackInfo_.currPosMs = positionMs;

    if (auto listener = listener_.lock()) {
        listener->onTrackInfo(trackInfo_);
    }
}
