#include "media_transport.h"

#include "common.h"

using namespace bluez_impl;

MediaTransport::MediaTransport(std::string objectPath, std::weak_ptr<Listener> listener)
    : sdbus::ProxyInterfaces<org::bluez::MediaTransport1_proxy, sdbus::Properties_proxy>{BLUEZ_BUS, std::move(objectPath)}
    , listener_(std::move(listener))
    , bluetoothDevice_(std::make_shared<BluetoothDevice>(Device()))
{
    registerProxy();
    volume_ = Volume();
}

MediaTransport::~MediaTransport() {
    unregisterProxy();
}

std::shared_ptr<BluetoothDevice> MediaTransport::getDeviceProxy() {
    return bluetoothDevice_;
}

uint16_t MediaTransport::getVolume() const {
    return volume_;
}

void MediaTransport::setVolume(uint16_t volume) {
    volume_ = volume;
    Volume(volume);
}

void MediaTransport::onPropertiesChanged(const std::string& interfaceName,
                                         const std::map<std::string, sdbus::Variant>& changedProperties,
                                         const std::vector<std::string>& /*invalidatedProperties*/)
{
    if (interfaceName != org::bluez::MediaTransport1_proxy::INTERFACE_NAME) {
        return;
    }

    for (const auto& [propName, propValue] : changedProperties) {
        if (propName == "Volume") {
            onVolumeChanged(propValue.get<uint16_t>());
        }
    }
}

void MediaTransport::onVolumeChanged(uint16_t volume) {
    if (volume_ == volume) {
        return;
    }
    volume_ = volume;
    if (auto listener = listener_.lock()) {
        listener->onVolumeChanged(volume);
    }
}
