#include "object_manager.h"

#include "common.h"

#include <yandex_io/libs/logging/logging.h>

using namespace bluez_impl;

ObjectManager::ObjectManager(std::string serviceName, std::string path)
    : sdbus::ProxyInterfaces<sdbus::ObjectManager_proxy>(std::move(serviceName), std::move(path))
{
    registerProxy();
}

ObjectManager::~ObjectManager()
{
    unregisterProxy();
}

sdbus::ObjectPath ObjectManager::getObjectPath(const std::string& serviceName,
                                               const std::string& omPath,
                                               const std::string& interface)
{
    ObjectManager objectManager{serviceName, omPath};
    auto objects = objectManager.GetManagedObjects();
    auto found = std::find_if(objects.begin(), objects.end(), [&interface](const auto& o) {
        return o.second.find(interface) != o.second.end();
    });
    if (found == objects.end()) {
        throw std::runtime_error{"cannot find default object with interface " + interface};
    }
    return found->first;
}

sdbus::ObjectPath ObjectManager::getBluezObjectPath(const std::string& interface)
{
    return ObjectManager::getObjectPath(BLUEZ_BUS, BLUEZ_OM_PATH, interface);
}

void ObjectManager::onInterfacesAdded(const sdbus::ObjectPath& objectPath,
                                      const std::map<std::string, std::map<std::string, sdbus::Variant>>& interfacesAndProperties)
{
    if (YIO_LOG_DEBUG_ENABLED()) {
        std::stringstream ss;
        ss << "For object: " << objectPath << " added interfaces:\n";
        for (const auto& [interface, props] : interfacesAndProperties) {
            ss << " -- " << interface << "\n";
            for (const auto& [name, value] : props) {
                ss << "\t -- " << name << "\n";
            }
        }
        YIO_LOG_DEBUG(ss.str());
    }
}

void ObjectManager::onInterfacesRemoved(const sdbus::ObjectPath& objectPath,
                                        const std::vector<std::string>& interfaces)
{
    if (YIO_LOG_DEBUG_ENABLED()) {
        std::stringstream ss;
        ss << "For object: " << objectPath << " removed interfaces:\n";
        for (const auto& interface : interfaces) {
            ss << " -- " << interface << "\n";
        }
        YIO_LOG_DEBUG(ss.str());
    }
}
